"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkLoadBalancer = void 0;
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const aws_iam_1 = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_load_balancer_1 = require("../shared/base-load-balancer");
const network_listener_1 = require("./network-listener");
/**
 * (experimental) Define a new network load balancer.
 *
 * @experimental
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
class NetworkLoadBalancer extends base_load_balancer_1.BaseLoadBalancer {
    /**
     * @experimental
     */
    static fromNetworkLoadBalancerAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.loadBalancerArn = attrs.loadBalancerArn;
                this.vpc = attrs.vpc;
            }
            addListener(lid, props) {
                return new network_listener_1.NetworkListener(this, lid, {
                    loadBalancer: this,
                    ...props,
                });
            }
            get loadBalancerCanonicalHostedZoneId() {
                if (attrs.loadBalancerCanonicalHostedZoneId) {
                    return attrs.loadBalancerCanonicalHostedZoneId;
                }
                // eslint-disable-next-line max-len
                throw new Error(`'loadBalancerCanonicalHostedZoneId' was not provided when constructing Network Load Balancer ${this.node.path} from attributes`);
            }
            get loadBalancerDnsName() {
                if (attrs.loadBalancerDnsName) {
                    return attrs.loadBalancerDnsName;
                }
                // eslint-disable-next-line max-len
                throw new Error(`'loadBalancerDnsName' was not provided when constructing Network Load Balancer ${this.node.path} from attributes`);
            }
        }
        return new Import(scope, id);
    }
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'network',
        });
        if (props.crossZoneEnabled) {
            this.setAttribute('load_balancing.cross_zone.enabled', 'true');
        }
    }
    /**
     * (experimental) Add a listener to this load balancer.
     *
     * @returns The newly created listener
     * @experimental
     */
    addListener(id, props) {
        return new network_listener_1.NetworkListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    /**
     * (experimental) Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     *
     * This is extending the BaseLoadBalancer.logAccessLogs method to match the bucket permissions described
     * at https://docs.aws.amazon.com/elasticloadbalancing/latest/network/load-balancer-access-logs.html#access-logging-bucket-requirements
     *
     * @experimental
     */
    logAccessLogs(bucket, prefix) {
        super.logAccessLogs(bucket, prefix);
        const logsDeliveryServicePrincipal = new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com');
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [logsDeliveryServicePrincipal],
            resources: [
                bucket.arnForObjects(`${prefix ? prefix + '/' : ''}AWSLogs/${this.stack.account}/*`),
            ],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:GetBucketAcl'],
            principals: [logsDeliveryServicePrincipal],
            resources: [bucket.bucketArn],
        }));
    }
    /**
     * (experimental) Return the given named metric for this Network Load Balancer.
     *
     * @default Average over 5 minutes
     * @experimental
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/NetworkELB',
            metricName,
            dimensions: { LoadBalancer: this.loadBalancerFullName },
            ...props,
        }).attachTo(this);
    }
    /**
     * (experimental) The total number of concurrent TCP flows (or connections) from clients to targets.
     *
     * This metric includes connections in the SYN_SENT and ESTABLISHED states.
     * TCP connections are not terminated at the load balancer, so a client
     * opening a TCP connection to a target counts as a single flow.
     *
     * @default Average over 5 minutes
     * @experimental
     */
    metricActiveFlowCount(props) {
        return this.metric('ActiveFlowCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * (experimental) The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricConsumedLCUs(props) {
        return this.metric('ConsumedLCUs', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (deprecated) The number of targets that are considered healthy.
     *
     * @default Average over 5 minutes
     * @deprecated use ``NetworkTargetGroup.metricHealthyHostCount`` instead
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * (deprecated) The number of targets that are considered unhealthy.
     *
     * @default Average over 5 minutes
     * @deprecated use ``NetworkTargetGroup.metricUnHealthyHostCount`` instead
     */
    metricUnHealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * (experimental) The total number of new TCP flows (or connections) established from clients to targets in the time period.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricNewFlowCount(props) {
        return this.metric('NewFlowCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The total number of bytes processed by the load balancer, including TCP/IP headers.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricProcessedBytes(props) {
        return this.metric('ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The total number of reset (RST) packets sent from a client to a target.
     *
     * These resets are generated by the client and forwarded by the load balancer.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricTcpClientResetCount(props) {
        return this.metric('TCP_Client_Reset_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The total number of reset (RST) packets generated by the load balancer.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricTcpElbResetCount(props) {
        return this.metric('TCP_ELB_Reset_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The total number of reset (RST) packets sent from a target to a client.
     *
     * These resets are generated by the target and forwarded by the load balancer.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricTcpTargetResetCount(props) {
        return this.metric('TCP_Target_Reset_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
}
exports.NetworkLoadBalancer = NetworkLoadBalancer;
//# sourceMappingURL=data:application/json;base64,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