"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpCodeTarget = exports.HttpCodeElb = exports.ApplicationLoadBalancer = void 0;
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_load_balancer_1 = require("../shared/base-load-balancer");
const enums_1 = require("../shared/enums");
const application_listener_1 = require("./application-listener");
const application_listener_action_1 = require("./application-listener-action");
/**
 * (experimental) Define an Application Load Balancer.
 *
 * @experimental
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
class ApplicationLoadBalancer extends base_load_balancer_1.BaseLoadBalancer {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props, {
            type: 'application',
            securityGroups: core_1.Lazy.listValue({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
            ipAddressType: props.ipAddressType,
        });
        this.ipAddressType = (_a = props.ipAddressType) !== null && _a !== void 0 ? _a : enums_1.IpAddressType.IPV4;
        const securityGroups = [props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: `Automatically created Security Group for ELB ${core_1.Names.uniqueId(this)}`,
                allowAllOutbound: false,
            })];
        this.connections = new ec2.Connections({ securityGroups });
        if (props.http2Enabled === false) {
            this.setAttribute('routing.http2.enabled', 'false');
        }
        if (props.idleTimeout !== undefined) {
            this.setAttribute('idle_timeout.timeout_seconds', props.idleTimeout.toSeconds().toString());
        }
    }
    /**
     * (experimental) Import an existing Application Load Balancer.
     *
     * @experimental
     */
    static fromApplicationLoadBalancerAttributes(scope, id, attrs) {
        return new ImportedApplicationLoadBalancer(scope, id, attrs);
    }
    /**
     * (experimental) Add a new listener to this load balancer.
     *
     * @experimental
     */
    addListener(id, props) {
        return new application_listener_1.ApplicationListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    /**
     * (experimental) Add a redirection listener to this load balancer.
     *
     * @experimental
     */
    addRedirect(props = {}) {
        var _a, _b, _c, _d;
        const sourcePort = (_a = props.sourcePort) !== null && _a !== void 0 ? _a : 80;
        const targetPort = ((_b = props.targetPort) !== null && _b !== void 0 ? _b : 443).toString();
        return this.addListener(`Redirect${sourcePort}To${targetPort}`, {
            protocol: (_c = props.sourceProtocol) !== null && _c !== void 0 ? _c : enums_1.ApplicationProtocol.HTTP,
            port: sourcePort,
            open: true,
            defaultAction: application_listener_action_1.ListenerAction.redirect({
                port: targetPort,
                protocol: (_d = props.targetProtocol) !== null && _d !== void 0 ? _d : enums_1.ApplicationProtocol.HTTPS,
                permanent: true,
            }),
        });
    }
    /**
     * (experimental) Add a security group to this load balancer.
     *
     * @experimental
     */
    addSecurityGroup(securityGroup) {
        this.connections.addSecurityGroup(securityGroup);
    }
    /**
     * (experimental) Return the given named metric for this Application Load Balancer.
     *
     * @default Average over 5 minutes
     * @experimental
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApplicationELB',
            metricName,
            dimensions: { LoadBalancer: this.loadBalancerFullName },
            ...props,
        });
    }
    /**
     * (experimental) The total number of concurrent TCP connections active from clients to the load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricActiveConnectionCount(props) {
        return this.metric('ActiveConnectionCount', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of TLS connections initiated by the client that did not establish a session with the load balancer.
     *
     * Possible causes include a
     * mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricClientTlsNegotiationErrorCount(props) {
        return this.metric('ClientTLSNegotiationErrorCount', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricConsumedLCUs(props) {
        return this.metric('ConsumedLCUs', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of fixed-response actions that were successful.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricHttpFixedResponseCount(props) {
        return this.metric('HTTP_Fixed_Response_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of redirect actions that were successful.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricHttpRedirectCount(props) {
        return this.metric('HTTP_Redirect_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of redirect actions that couldn't be completed because the URL in the response location header is larger than 8K.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricHttpRedirectUrlLimitExceededCount(props) {
        return this.metric('HTTP_Redirect_Url_Limit_Exceeded_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of HTTP 3xx/4xx/5xx codes that originate from the load balancer.
     *
     * This does not include any response codes generated by the targets.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricHttpCodeElb(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets in the load balancer.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricHttpCodeTarget(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The total number of bytes processed by the load balancer over IPv6.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricIpv6ProcessedBytes(props) {
        return this.metric('IPv6ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of IPv6 requests received by the load balancer.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricIpv6RequestCount(props) {
        return this.metric('IPv6RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The total number of new TCP connections established from clients to the load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricNewConnectionCount(props) {
        return this.metric('NewConnectionCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The total number of bytes processed by the load balancer over IPv4 and IPv6.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricProcessedBytes(props) {
        return this.metric('ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of connections that were rejected because the load balancer had reached its maximum number of connections.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricRejectedConnectionCount(props) {
        return this.metric('RejectedConnectionCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricRequestCount(props) {
        return this.metric('RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of rules processed by the load balancer given a request rate averaged over an hour.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricRuleEvaluations(props) {
        return this.metric('RuleEvaluations', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricTargetConnectionErrorCount(props) {
        return this.metric('TargetConnectionErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     * @experimental
     */
    metricTargetResponseTime(props) {
        return this.metric('TargetResponseTime', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * (experimental) The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricTargetTLSNegotiationErrorCount(props) {
        return this.metric('TargetTLSNegotiationErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of user authentications that could not be completed.
     *
     * Because an authenticate action was misconfigured, the load balancer
     * couldn't establish a connection with the IdP, or the load balancer
     * couldn't complete the authentication flow due to an internal error.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricElbAuthError(props) {
        return this.metric('ELBAuthError', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The number of user authentications that could not be completed because the IdP denied access to the user or an authorization code was used more than once.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricElbAuthFailure(props) {
        return this.metric('ELBAuthFailure', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (experimental) The time elapsed, in milliseconds, to query the IdP for the ID token and user info.
     *
     * If one or more of these operations fail, this is the time to failure.
     *
     * @default Average over 5 minutes
     * @experimental
     */
    metricElbAuthLatency(props) {
        return this.metric('ELBAuthLatency', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * (experimental) The number of authenticate actions that were successful.
     *
     * This metric is incremented at the end of the authentication workflow,
     * after the load balancer has retrieved the user claims from the IdP.
     *
     * @default Sum over 5 minutes
     * @experimental
     */
    metricElbAuthSuccess(props) {
        return this.metric('ELBAuthSuccess', {
            statistic: 'Sum',
            ...props,
        });
    }
}
exports.ApplicationLoadBalancer = ApplicationLoadBalancer;
/**
 * (experimental) Count of HTTP status originating from the load balancer.
 *
 * This count does not include any response codes generated by the targets.
 *
 * @experimental
 */
var HttpCodeElb;
(function (HttpCodeElb) {
    HttpCodeElb["ELB_3XX_COUNT"] = "HTTPCode_ELB_3XX_Count";
    HttpCodeElb["ELB_4XX_COUNT"] = "HTTPCode_ELB_4XX_Count";
    HttpCodeElb["ELB_5XX_COUNT"] = "HTTPCode_ELB_5XX_Count";
})(HttpCodeElb = exports.HttpCodeElb || (exports.HttpCodeElb = {}));
/**
 * (experimental) Count of HTTP status originating from the targets.
 *
 * @experimental
 */
var HttpCodeTarget;
(function (HttpCodeTarget) {
    HttpCodeTarget["TARGET_2XX_COUNT"] = "HTTPCode_Target_2XX_Count";
    HttpCodeTarget["TARGET_3XX_COUNT"] = "HTTPCode_Target_3XX_Count";
    HttpCodeTarget["TARGET_4XX_COUNT"] = "HTTPCode_Target_4XX_Count";
    HttpCodeTarget["TARGET_5XX_COUNT"] = "HTTPCode_Target_5XX_Count";
})(HttpCodeTarget = exports.HttpCodeTarget || (exports.HttpCodeTarget = {}));
/**
 * An ApplicationLoadBalancer that has been defined elsewhere
 */
class ImportedApplicationLoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.vpc = props.vpc;
        this.loadBalancerArn = props.loadBalancerArn;
        this.connections = new ec2.Connections({
            securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(this, 'SecurityGroup', props.securityGroupId, {
                    allowAllOutbound: props.securityGroupAllowsAllOutbound,
                })],
        });
    }
    addListener(id, props) {
        return new application_listener_1.ApplicationListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    get loadBalancerCanonicalHostedZoneId() {
        if (this.props.loadBalancerCanonicalHostedZoneId) {
            return this.props.loadBalancerCanonicalHostedZoneId;
        }
        // eslint-disable-next-line max-len
        throw new Error(`'loadBalancerCanonicalHostedZoneId' was not provided when constructing Application Load Balancer ${this.node.path} from attributes`);
    }
    get loadBalancerDnsName() {
        if (this.props.loadBalancerDnsName) {
            return this.props.loadBalancerDnsName;
        }
        // eslint-disable-next-line max-len
        throw new Error(`'loadBalancerDnsName' was not provided when constructing Application Load Balancer ${this.node.path} from attributes`);
    }
}
//# sourceMappingURL=data:application/json;base64,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