"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationListener = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const listener_certificate_1 = require("../shared/listener-certificate");
const util_1 = require("../shared/util");
const application_listener_action_1 = require("./application-listener-action");
const application_listener_certificate_1 = require("./application-listener-certificate");
const application_listener_rule_1 = require("./application-listener-rule");
const application_target_group_1 = require("./application-target-group");
/**
 * (experimental) Define an ApplicationListener.
 *
 * @experimental
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class ApplicationListener extends base_listener_1.BaseListener {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        if (protocol === undefined || port === undefined) {
            throw new Error('At least one of \'port\' or \'protocol\' is required');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            certificates: core_1.Lazy.anyValue({ produce: () => this.certificateArns.map(certificateArn => ({ certificateArn })) }, { omitEmptyArray: true }),
            protocol,
            port,
            sslPolicy: props.sslPolicy,
        });
        this.loadBalancer = props.loadBalancer;
        this.protocol = protocol;
        this.certificateArns = [];
        // Attach certificates
        if (props.certificateArns && props.certificateArns.length > 0) {
            this.addCertificateArns('ListenerCertificate', props.certificateArns);
        }
        if (props.certificates && props.certificates.length > 0) {
            this.addCertificates('DefaultCertificates', props.certificates);
        }
        // This listener edits the securitygroup of the load balancer,
        // but adds its own default port.
        this.connections = new ec2.Connections({
            securityGroups: props.loadBalancer.connections.securityGroups,
            defaultPort: ec2.Port.tcp(port),
        });
        if (props.defaultAction && props.defaultTargetGroups) {
            throw new Error('Specify at most one of \'defaultAction\' and \'defaultTargetGroups\'');
        }
        if (props.defaultAction) {
            this.setDefaultAction(props.defaultAction);
        }
        if (props.defaultTargetGroups) {
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.defaultTargetGroups));
        }
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv4(), `Allow from anyone on port ${port}`);
            if (this.loadBalancer.ipAddressType === enums_1.IpAddressType.DUAL_STACK) {
                this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv6(), `Allow from anyone on port ${port}`);
            }
        }
    }
    /**
     * (experimental) Import an existing listener.
     *
     * @experimental
     */
    static fromApplicationListenerAttributes(scope, id, attrs) {
        return new ImportedApplicationListener(scope, id, attrs);
    }
    /**
     * (deprecated) Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     *
     * @deprecated Use `addCertificates` instead.
     */
    addCertificateArns(id, arns) {
        this.addCertificates(id, arns.map(listener_certificate_1.ListenerCertificate.fromArn));
    }
    /**
     * (experimental) Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     *
     * @experimental
     */
    addCertificates(id, certificates) {
        const additionalCerts = [...certificates];
        if (this.certificateArns.length === 0 && additionalCerts.length > 0) {
            const first = additionalCerts.splice(0, 1)[0];
            this.certificateArns.push(first.certificateArn);
        }
        if (additionalCerts.length > 0) {
            new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
                listener: this,
                certificates: additionalCerts,
            });
        }
    }
    /**
     * (experimental) Perform the given default action on incoming requests.
     *
     * This allows full control of the default action of the load balancer,
     * including Action chaining, fixed responses and redirect responses. See
     * the `ListenerAction` class for all options.
     *
     * It's possible to add routing conditions to the Action added in this way.
     * At least one Action must be added without conditions (which becomes the
     * default Action).
     *
     * @experimental
     */
    addAction(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                action: props.action,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(props.action);
        }
    }
    /**
     * (experimental) Load balance incoming requests to the given target groups.
     *
     * All target groups will be load balanced to with equal weight and without
     * stickiness. For a more complex configuration than that, use `addAction()`.
     *
     * It's possible to add routing conditions to the TargetGroups added in this
     * way. At least one TargetGroup must be added without conditions (which will
     * become the default Action for this listener).
     *
     * @experimental
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                targetGroups: props.targetGroups,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.targetGroups));
        }
    }
    /**
     * (experimental) Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved, and a 'forward' action to route traffic to the given TargetGroup.
     *
     * If you want more control over the precise setup, create the TargetGroup
     * and use `addAction` yourself.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     * @experimental
     */
    addTargets(id, props) {
        if (!this.loadBalancer.vpc) {
            // eslint-disable-next-line max-len
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or an imported Load Balancer with specified vpc; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new application_target_group_1.ApplicationTargetGroup(this, id + 'Group', {
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            port: props.port,
            protocol: props.protocol,
            slowStart: props.slowStart,
            stickinessCookieDuration: props.stickinessCookieDuration,
            targetGroupName: props.targetGroupName,
            targets: props.targets,
            vpc: this.loadBalancer.vpc,
        });
        this.addTargetGroups(id, {
            conditions: props.conditions,
            hostHeader: props.hostHeader,
            pathPattern: props.pathPattern,
            pathPatterns: props.pathPatterns,
            priority: props.priority,
            targetGroups: [group],
        });
        return group;
    }
    /**
     * (deprecated) Add a fixed response.
     *
     * @deprecated Use `addAction()` instead
     */
    addFixedResponse(id, props) {
        checkAddRuleProps(props);
        const fixedResponse = {
            statusCode: props.statusCode,
            contentType: props.contentType,
            messageBody: props.messageBody,
        };
        application_listener_rule_1.validateFixedResponse(fixedResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                fixedResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.fixedResponse(core_1.Token.asNumber(props.statusCode), {
                contentType: props.contentType,
                messageBody: props.messageBody,
            }));
        }
    }
    /**
     * (deprecated) Add a redirect response.
     *
     * @deprecated Use `addAction()` instead
     */
    addRedirectResponse(id, props) {
        checkAddRuleProps(props);
        const redirectResponse = {
            host: props.host,
            path: props.path,
            port: props.port,
            protocol: props.protocol,
            query: props.query,
            statusCode: props.statusCode,
        };
        application_listener_rule_1.validateRedirectResponse(redirectResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                redirectResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.redirect({
                host: props.host,
                path: props.path,
                port: props.port,
                protocol: props.protocol,
                query: props.query,
                permanent: props.statusCode === 'HTTP_301',
            }));
        }
    }
    /**
     * (experimental) Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     *
     * @experimental
     */
    registerConnectable(connectable, portRange) {
        connectable.connections.allowFrom(this.loadBalancer, portRange, 'Load balancer to target');
    }
    /**
     * (experimental) Validate this listener.
     *
     * @experimental
     */
    validate() {
        const errors = super.validate();
        if (this.protocol === enums_1.ApplicationProtocol.HTTPS && this.certificateArns.length === 0) {
            errors.push('HTTPS Listener needs at least one certificate (call addCertificates)');
        }
        return errors;
    }
    /**
     * Wrapper for _setDefaultAction which does a type-safe bind
     */
    setDefaultAction(action) {
        action.bind(this, this);
        this._setDefaultAction(action);
    }
}
exports.ApplicationListener = ApplicationListener;
class ImportedApplicationListener extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.listenerArn = props.listenerArn;
        const defaultPort = props.defaultPort !== undefined ? ec2.Port.tcp(props.defaultPort) : undefined;
        let securityGroup;
        if (props.securityGroup) {
            securityGroup = props.securityGroup;
        }
        else if (props.securityGroupId) {
            securityGroup = ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', props.securityGroupId, {
                allowAllOutbound: props.securityGroupAllowsAllOutbound,
            });
        }
        else {
            throw new Error('Either `securityGroup` or `securityGroupId` must be specified to import an application listener.');
        }
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort,
        });
    }
    /**
     * Add one or more certificates to this listener.
     */
    addCertificateArns(id, arns) {
        new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
            listener: this,
            certificateArns: arns,
        });
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * It's possible to add conditions to the TargetGroups added in this way.
     * At least one TargetGroup must be added without conditions.
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            new application_listener_rule_1.ApplicationListenerRule(this, id, {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                targetGroups: props.targetGroups,
            });
        }
        else {
            throw new Error('Cannot add default Target Groups to imported ApplicationListener');
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(_id, _props) {
        // eslint-disable-next-line max-len
        throw new Error('Can only call addTargets() when using a constructed ApplicationListener; construct a new TargetGroup and use addTargetGroup.');
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        this.connections.allowTo(connectable, portRange, 'Load balancer to target');
    }
}
function checkAddRuleProps(props) {
    var _a;
    const conditionsCount = ((_a = props.conditions) === null || _a === void 0 ? void 0 : _a.length) || 0;
    const hasAnyConditions = conditionsCount !== 0 ||
        props.hostHeader !== undefined || props.pathPattern !== undefined || props.pathPatterns !== undefined;
    const hasPriority = props.priority !== undefined;
    if (hasAnyConditions !== hasPriority) {
        throw new Error('Setting \'conditions\', \'pathPattern\' or \'hostHeader\' also requires \'priority\', and vice versa');
    }
}
//# sourceMappingURL=data:application/json;base64,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