"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerPort = exports.LoadBalancer = exports.LoadBalancingProtocol = void 0;
const aws_ec2_1 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
/**
 * @experimental
 */
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * (experimental) A load balancer with a single listener.
 *
 * Routes to a fleet of of instances in a VPC.
 *
 * @experimental
 */
class LoadBalancer extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * (experimental) An object controlling specifically the connections for each listener added to this load balancer.
         *
         * @experimental
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.anyValue({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: (props.crossZone === undefined || props.crossZone) ? true : false,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * (experimental) Add a backend to the load balancer.
     *
     * @returns A ListenerPort object that controls connections to the listener port
     * @experimental
     */
    addListener(listener) {
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: listener.sslCertificateId,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    /**
     * @experimental
     */
    addTarget(target) {
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @experimental
     * @attribute true
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @experimental
     * @attribute true
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @experimental
     * @attribute true
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @experimental
     * @attribute true
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @experimental
     * @attribute true
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @experimental
     * @attribute true
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
/**
 * (experimental) Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *     const listener = elb.addListener(...);
 *
 *     listener.connections.allowDefaultPortFromAnyIPv4();
 *     // or
 *     instance.connections.allowToDefaultPort(listener);
 *
 * @experimental
 */
class ListenerPort {
    /**
     * @experimental
     */
    constructor(securityGroup, defaultPort) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
exports.ListenerPort = ListenerPort;
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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