"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateProfile = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * (experimental) Fargate profiles allows an administrator to declare which pods run on Fargate.
 *
 * This declaration is done through the profile’s selectors. Each
 * profile can have up to five selectors that contain a namespace and optional
 * labels. You must define a namespace for every selector. The label field
 * consists of multiple optional key-value pairs. Pods that match a selector (by
 * matching a namespace for the selector and all of the labels specified in the
 * selector) are scheduled on Fargate. If a namespace selector is defined
 * without any labels, Amazon EKS will attempt to schedule all pods that run in
 * that namespace onto Fargate using the profile. If a to-be-scheduled pod
 * matches any of the selectors in the Fargate profile, then that pod is
 * scheduled on Fargate.
 *
 * If a pod matches multiple Fargate profiles, Amazon EKS picks one of the
 * matches at random. In this case, you can specify which profile a pod should
 * use by adding the following Kubernetes label to the pod specification:
 * eks.amazonaws.com/fargate-profile: profile_name. However, the pod must still
 * match a selector in that profile in order to be scheduled onto Fargate.
 *
 * @experimental
 */
class FargateProfile extends core_2.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this, {
            adminRole: props.cluster.adminRole,
        });
        this.podExecutionRole = (_a = props.podExecutionRole) !== null && _a !== void 0 ? _a : new iam.Role(this, 'PodExecutionRole', {
            assumedBy: new iam.ServicePrincipal('eks-fargate-pods.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSFargatePodExecutionRolePolicy')],
        });
        this.podExecutionRole.grantPassRole(props.cluster.adminRole);
        let subnets;
        if (props.vpc) {
            const selection = (_b = props.subnetSelection) !== null && _b !== void 0 ? _b : { subnetType: ec2.SubnetType.PRIVATE };
            subnets = props.vpc.selectSubnets(selection).subnetIds;
        }
        if (props.selectors.length < 1) {
            throw new Error('Fargate profile requires at least one selector');
        }
        if (props.selectors.length > 5) {
            throw new Error('Fargate profile supports up to five selectors');
        }
        this.tags = new core_1.TagManager(core_1.TagType.MAP, 'AWS::EKS::FargateProfile');
        const resource = new core_1.CustomResource(this, 'Resource', {
            serviceToken: provider.serviceToken,
            resourceType: consts_1.FARGATE_PROFILE_RESOURCE_TYPE,
            properties: {
                AssumeRoleArn: props.cluster.adminRole.roleArn,
                Config: {
                    clusterName: props.cluster.clusterName,
                    fargateProfileName: props.fargateProfileName,
                    podExecutionRoleArn: this.podExecutionRole.roleArn,
                    selectors: props.selectors,
                    subnets,
                    tags: core_1.Lazy.anyValue({ produce: () => this.tags.renderTags() }),
                },
            },
        });
        this.fargateProfileArn = resource.getAttString('fargateProfileArn');
        this.fargateProfileName = resource.ref;
        // Fargate profiles must be created sequentially. If other profile(s) already
        // exist on the same cluster, create a dependency to force sequential creation.
        const clusterFargateProfiles = props.cluster._attachFargateProfile(this);
        if (clusterFargateProfiles.length > 1) {
            const previousProfile = clusterFargateProfiles[clusterFargateProfiles.length - 2];
            resource.node.addDependency(previousProfile);
        }
        // map the fargate pod execution role to the relevant groups in rbac
        // see https://github.com/aws/aws-cdk/issues/7981
        props.cluster.awsAuth.addRoleMapping(this.podExecutionRole, {
            username: 'system:node:{{SessionName}}',
            groups: [
                'system:bootstrappers',
                'system:nodes',
                'system:node-proxier',
            ],
        });
    }
}
exports.FargateProfile = FargateProfile;
//# sourceMappingURL=data:application/json;base64,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