"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HelmChart = void 0;
const path = require("path");
const aws_cloudformation_1 = require("../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const kubectl_layer_1 = require("./kubectl-layer");
/**
 * (experimental) Represents a helm chart within the Kubernetes system.
 *
 * Applies/deletes the resources using `kubectl` in sync with the resource.
 *
 * @experimental
 */
class HelmChart extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        const stack = core_1.Stack.of(this);
        // we maintain a single manifest custom resource handler for each cluster
        const handler = this.getOrCreateHelmChartHandler(props.cluster);
        if (!handler) {
            throw new Error('Cannot define a Helm chart on a cluster with kubectl disabled');
        }
        new aws_cloudformation_1.CustomResource(this, 'Resource', {
            provider: aws_cloudformation_1.CustomResourceProvider.lambda(handler),
            resourceType: HelmChart.RESOURCE_TYPE,
            properties: {
                Release: props.release || core_1.Names.uniqueId(this).slice(-63).toLowerCase(),
                Chart: props.chart,
                Version: props.version,
                Values: (props.values ? stack.toJsonString(props.values) : undefined),
                Namespace: props.namespace || 'default',
                Repository: props.repository,
            },
        });
    }
    getOrCreateHelmChartHandler(cluster) {
        if (!cluster.kubectlEnabled) {
            return undefined;
        }
        let handler = cluster.node.tryFindChild('HelmChartHandler');
        if (!handler) {
            handler = new lambda.Function(cluster, 'HelmChartHandler', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'helm-chart')),
                runtime: lambda.Runtime.PYTHON_3_7,
                handler: 'index.handler',
                timeout: core_1.Duration.minutes(15),
                layers: [kubectl_layer_1.KubectlLayer.getOrCreate(this, { version: '2.0.0-beta1' })],
                memorySize: 256,
                environment: {
                    CLUSTER_NAME: cluster.clusterName,
                },
                // NOTE: we must use the default IAM role that's mapped to "system:masters"
                // as the execution role of this custom resource handler. This is the only
                // way to be able to interact with the cluster after it's been created.
                role: cluster._defaultMastersRole,
            });
        }
        return handler;
    }
}
exports.HelmChart = HelmChart;
/**
 * (experimental) The CloudFormation reosurce type.
 *
 * @experimental
 */
HelmChart.RESOURCE_TYPE = 'Custom::AWSCDK-EKS-HelmChart';
//# sourceMappingURL=data:application/json;base64,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