"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsAuth = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const k8s_resource_1 = require("./k8s-resource");
/**
 * (experimental) Manages mapping between IAM users and roles to Kubernetes RBAC configuration.
 *
 * @see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
 * @experimental
 */
class AwsAuth extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.roleMappings = new Array();
        this.userMappings = new Array();
        this.accounts = new Array();
        this.stack = core_1.Stack.of(this);
        new k8s_resource_1.KubernetesResource(this, 'manifest', {
            cluster: props.cluster,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'aws-auth',
                        namespace: 'kube-system',
                    },
                    data: {
                        mapRoles: this.synthesizeMapRoles(),
                        mapUsers: this.synthesizeMapUsers(),
                        mapAccounts: this.synthesizeMapAccounts(),
                    },
                },
            ],
        });
    }
    /**
     * (experimental) Adds the specified IAM role to the `system:masters` RBAC group, which means that anyone that can assume it will be able to administer this Kubernetes system.
     *
     * @param role The IAM role to add.
     * @param username Optional user (defaults to the role ARN).
     * @experimental
     */
    addMastersRole(role, username) {
        this.addRoleMapping(role, {
            username,
            groups: ['system:masters'],
        });
    }
    /**
     * (experimental) Adds a mapping between an IAM role to a Kubernetes user and groups.
     *
     * @param role The IAM role to map.
     * @param mapping Mapping to k8s user name and groups.
     * @experimental
     */
    addRoleMapping(role, mapping) {
        this.roleMappings.push({ role, mapping });
    }
    /**
     * (experimental) Adds a mapping between an IAM user to a Kubernetes user and groups.
     *
     * @param user The IAM user to map.
     * @param mapping Mapping to k8s user name and groups.
     * @experimental
     */
    addUserMapping(user, mapping) {
        this.userMappings.push({ user, mapping });
    }
    /**
     * (experimental) Additional AWS account to add to the aws-auth configmap.
     *
     * @param accountId account number.
     * @experimental
     */
    addAccount(accountId) {
        this.accounts.push(accountId);
    }
    synthesizeMapRoles() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.roleMappings.map(m => ({
                rolearn: m.role.roleArn,
                username: m.mapping.username,
                groups: m.mapping.groups,
            }))),
        });
    }
    synthesizeMapUsers() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.userMappings.map(m => ({
                userarn: m.user.userArn,
                username: m.mapping.username,
                groups: m.mapping.groups,
            }))),
        });
    }
    synthesizeMapAccounts() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.accounts),
        });
    }
}
exports.AwsAuth = AwsAuth;
//# sourceMappingURL=data:application/json;base64,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