"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystem = exports.ThroughputMode = exports.PerformanceMode = exports.LifecyclePolicy = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const access_point_1 = require("./access-point");
const efs_generated_1 = require("./efs.generated");
/**
 * (experimental) EFS Lifecycle Policy, if a file is not accessed for given days, it will move to EFS Infrequent Access.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-lifecyclepolicies
 * @experimental
 */
var LifecyclePolicy;
(function (LifecyclePolicy) {
    LifecyclePolicy["AFTER_7_DAYS"] = "AFTER_7_DAYS";
    LifecyclePolicy["AFTER_14_DAYS"] = "AFTER_14_DAYS";
    LifecyclePolicy["AFTER_30_DAYS"] = "AFTER_30_DAYS";
    LifecyclePolicy["AFTER_60_DAYS"] = "AFTER_60_DAYS";
    LifecyclePolicy["AFTER_90_DAYS"] = "AFTER_90_DAYS";
})(LifecyclePolicy = exports.LifecyclePolicy || (exports.LifecyclePolicy = {}));
/**
 * (experimental) EFS Performance mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-efs-filesystem-performancemode
 * @experimental
 */
var PerformanceMode;
(function (PerformanceMode) {
    PerformanceMode["GENERAL_PURPOSE"] = "generalPurpose";
    PerformanceMode["MAX_IO"] = "maxIO";
})(PerformanceMode = exports.PerformanceMode || (exports.PerformanceMode = {}));
/**
 * (experimental) EFS Throughput mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-throughputmode
 * @experimental
 */
var ThroughputMode;
(function (ThroughputMode) {
    ThroughputMode["BURSTING"] = "bursting";
    ThroughputMode["PROVISIONED"] = "provisioned";
})(ThroughputMode = exports.ThroughputMode || (exports.ThroughputMode = {}));
/**
 * (experimental) The Elastic File System implementation of IFileSystem.
 *
 * It creates a new, empty file system in Amazon Elastic File System (Amazon EFS).
 * It also creates mount target (AWS::EFS::MountTarget) implicitly to mount the
 * EFS file system on an Amazon Elastic Compute Cloud (Amazon EC2) instance or another resource.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html
 * @experimental
 * @resource AWS::EFS::FileSystem
 */
class FileSystem extends core_1.Resource {
    /**
     * (experimental) Constructor for creating a new EFS FileSystem.
     *
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this._mountTargetsAvailable = new core_1.ConcreteDependable();
        if (props.throughputMode === ThroughputMode.PROVISIONED && props.provisionedThroughputPerSecond === undefined) {
            throw new Error('Property provisionedThroughputPerSecond is required when throughputMode is PROVISIONED');
        }
        const filesystem = new efs_generated_1.CfnFileSystem(this, 'Resource', {
            encrypted: props.encrypted,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lifecyclePolicies: (props.lifecyclePolicy ? [{ transitionToIa: props.lifecyclePolicy }] : undefined),
            performanceMode: props.performanceMode,
            throughputMode: props.throughputMode,
            provisionedThroughputInMibps: (_a = props.provisionedThroughputPerSecond) === null || _a === void 0 ? void 0 : _a.toMebibytes(),
            backupPolicy: props.enableAutomaticBackups ? { status: 'ENABLED' } : undefined,
        });
        filesystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = filesystem.ref;
        core_1.Tags.of(this).add('Name', props.fileSystemName || this.node.path);
        const securityGroup = (props.securityGroup || new ec2.SecurityGroup(this, 'EfsSecurityGroup', {
            vpc: props.vpc,
        }));
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        const subnets = props.vpc.selectSubnets(props.vpcSubnets);
        // We now have to create the mount target for each of the mentioned subnet
        let mountTargetCount = 0;
        this.mountTargetsAvailable = [];
        subnets.subnetIds.forEach((subnetId) => {
            const mountTarget = new efs_generated_1.CfnMountTarget(this, 'EfsMountTarget' + (++mountTargetCount), {
                fileSystemId: this.fileSystemId,
                securityGroups: Array.of(securityGroup.securityGroupId),
                subnetId,
            });
            this._mountTargetsAvailable.add(mountTarget);
        });
        this.mountTargetsAvailable = this._mountTargetsAvailable;
    }
    /**
     * (experimental) Import an existing File System from the given properties.
     *
     * @experimental
     */
    static fromFileSystemAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.fileSystemId = attrs.fileSystemId;
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
                });
                this.mountTargetsAvailable = new core_1.ConcreteDependable();
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) create access point from this filesystem.
     *
     * @experimental
     */
    addAccessPoint(id, accessPointOptions = {}) {
        return new access_point_1.AccessPoint(this, id, {
            fileSystem: this,
            ...accessPointOptions,
        });
    }
}
exports.FileSystem = FileSystem;
/**
 * The default port File System listens on.
 */
FileSystem.DEFAULT_PORT = 2049;
//# sourceMappingURL=data:application/json;base64,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