"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FirelensLogRouter = exports.obtainDefaultFluentBitECRImage = exports.FirelensConfigFileType = exports.FirelensLogRouterType = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const container_definition_1 = require("./container-definition");
const container_image_1 = require("./container-image");
/**
 * (experimental) Firelens log router type, fluentbit or fluentd.
 *
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html
 *
 * @experimental
 */
var FirelensLogRouterType;
(function (FirelensLogRouterType) {
    FirelensLogRouterType["FLUENTBIT"] = "fluentbit";
    FirelensLogRouterType["FLUENTD"] = "fluentd";
})(FirelensLogRouterType = exports.FirelensLogRouterType || (exports.FirelensLogRouterType = {}));
/**
 * (experimental) Firelens configuration file type, s3 or file path.
 *
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-taskdef-customconfig
 *
 * @experimental
 */
var FirelensConfigFileType;
(function (FirelensConfigFileType) {
    FirelensConfigFileType["S3"] = "s3";
    FirelensConfigFileType["FILE"] = "file";
})(FirelensConfigFileType = exports.FirelensConfigFileType || (exports.FirelensConfigFileType = {}));
/**
 * Render to CfnTaskDefinition.FirelensConfigurationProperty from FirelensConfig
 */
function renderFirelensConfig(firelensConfig) {
    if (!firelensConfig.options) {
        return { type: firelensConfig.type };
    }
    else {
        // firelensConfig.options.configFileType has been filled with s3 or file type in constructor.
        return {
            type: firelensConfig.type,
            options: {
                'enable-ecs-log-metadata': firelensConfig.options.enableECSLogMetadata ? 'true' : 'false',
                'config-file-type': firelensConfig.options.configFileType,
                'config-file-value': firelensConfig.options.configFileValue,
            },
        };
    }
}
/**
 * SSM parameters for latest fluent bit docker image in ECR
 * https://github.com/aws/aws-for-fluent-bit#using-ssm-to-find-available-versions
 */
const fluentBitImageSSMPath = '/aws/service/aws-for-fluent-bit';
/**
 * Obtain Fluent Bit image in Amazon ECR and setup corresponding IAM permissions.
 * ECR image pull permissions will be granted in task execution role.
 * Cloudwatch logs, Kinesis data stream or firehose permissions will be grant by check options in logDriverConfig.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-using-fluentbit
 */
function obtainDefaultFluentBitECRImage(task, logDriverConfig, imageTag) {
    // grant ECR image pull permissions to executor role
    task.addToExecutionRolePolicy(new iam.PolicyStatement({
        actions: [
            'ecr:GetAuthorizationToken',
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:BatchGetImage',
        ],
        resources: ['*'],
    }));
    // grant cloudwatch or firehose permissions to task role
    const logName = logDriverConfig && logDriverConfig.logDriver === 'awsfirelens'
        && logDriverConfig.options && logDriverConfig.options.Name;
    if (logName === 'cloudwatch') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:DescribeLogStreams',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'firehose') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'firehose:PutRecordBatch',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'kinesis') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'kinesis:PutRecords',
            ],
            resources: ['*'],
        }));
    }
    const fluentBitImageTag = imageTag || 'latest';
    const fluentBitImage = `${fluentBitImageSSMPath}/${fluentBitImageTag}`;
    // Not use ContainerImage.fromEcrRepository since it's not support parsing ECR repo URI,
    // use repo ARN might result in complex Fn:: functions in cloudformation template.
    return container_image_1.ContainerImage.fromRegistry(ssm.StringParameter.valueForStringParameter(task, fluentBitImage));
}
exports.obtainDefaultFluentBitECRImage = obtainDefaultFluentBitECRImage;
/**
 * (experimental) Firelens log router.
 *
 * @experimental
 */
class FirelensLogRouter extends container_definition_1.ContainerDefinition {
    /**
     * (experimental) Constructs a new instance of the FirelensLogRouter class.
     *
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        const options = props.firelensConfig.options;
        if (options) {
            const enableECSLogMetadata = options.enableECSLogMetadata || options.enableECSLogMetadata === undefined;
            const configFileType = (options.configFileType === undefined || options.configFileType === FirelensConfigFileType.S3) &&
                (cdk.Token.isUnresolved(options.configFileValue) || /arn:aws[a-zA-Z-]*:s3:::.+/.test(options.configFileValue))
                ? FirelensConfigFileType.S3 : FirelensConfigFileType.FILE;
            this.firelensConfig = {
                type: props.firelensConfig.type,
                options: {
                    enableECSLogMetadata,
                    configFileType,
                    configFileValue: options.configFileValue,
                },
            };
            // grant s3 access permissions
            if (configFileType === FirelensConfigFileType.S3) {
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetObject',
                    ],
                    resources: [options.configFileValue],
                }));
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetBucketLocation',
                    ],
                    resources: [options.configFileValue.split('/')[0]],
                }));
            }
        }
        else {
            this.firelensConfig = props.firelensConfig;
        }
    }
    /**
     * (experimental) Render this container definition to a CloudFormation object.
     *
     * @experimental
     */
    renderContainerDefinition(_taskDefinition) {
        return {
            ...(super.renderContainerDefinition()),
            firelensConfiguration: this.firelensConfig && renderFirelensConfig(this.firelensConfig),
        };
    }
}
exports.FirelensLogRouter = FirelensLogRouter;
//# sourceMappingURL=data:application/json;base64,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