"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Compatibility = exports.Scope = exports.PidMode = exports.IpcMode = exports.NetworkMode = exports.TaskDefinition = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const container_definition_1 = require("../container-definition");
const ecs_generated_1 = require("../ecs.generated");
const firelens_log_router_1 = require("../firelens-log-router");
const aws_log_driver_1 = require("../log-drivers/aws-log-driver");
class TaskDefinitionBase extends core_1.Resource {
    /**
     * (experimental) Return true if the task definition can be run on an EC2 cluster.
     *
     * @experimental
     */
    get isEc2Compatible() {
        return isEc2Compatible(this.compatibility);
    }
    /**
     * (experimental) Return true if the task definition can be run on a Fargate cluster.
     *
     * @experimental
     */
    get isFargateCompatible() {
        return isFargateCompatible(this.compatibility);
    }
}
/**
 * (experimental) The base class for all task definitions.
 *
 * @experimental
 */
class TaskDefinition extends TaskDefinitionBase {
    /**
     * (experimental) Constructs a new instance of the TaskDefinition class.
     *
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * (experimental) The container definitions.
         *
         * @experimental
         */
        this.containers = new Array();
        /**
         * All volumes
         */
        this.volumes = [];
        /**
         * Placement constraints for task instances
         */
        this.placementConstraints = new Array();
        this.family = props.family || core_1.Names.uniqueId(this);
        this.compatibility = props.compatibility;
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        this.networkMode = props.networkMode !== undefined ? props.networkMode :
            this.isFargateCompatible ? NetworkMode.AWS_VPC : NetworkMode.BRIDGE;
        if (this.isFargateCompatible && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`Fargate tasks can only have AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.proxyConfiguration && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`ProxyConfiguration can only be used with AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.placementConstraints && props.placementConstraints.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        if (this.isFargateCompatible && (!props.cpu || !props.memoryMiB)) {
            throw new Error(`Fargate-compatible tasks require both CPU (${props.cpu}) and memory (${props.memoryMiB}) specifications`);
        }
        this._executionRole = props.executionRole;
        this.taskRole = props.taskRole || new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        const taskDef = new ecs_generated_1.CfnTaskDefinition(this, 'Resource', {
            containerDefinitions: core_1.Lazy.anyValue({ produce: () => this.renderContainers() }, { omitEmptyArray: true }),
            volumes: core_1.Lazy.anyValue({ produce: () => this.renderVolumes() }, { omitEmptyArray: true }),
            executionRoleArn: core_1.Lazy.stringValue({ produce: () => this.executionRole && this.executionRole.roleArn }),
            family: this.family,
            taskRoleArn: this.taskRole.roleArn,
            requiresCompatibilities: [
                ...(isEc2Compatible(props.compatibility) ? ['EC2'] : []),
                ...(isFargateCompatible(props.compatibility) ? ['FARGATE'] : []),
            ],
            networkMode: this.renderNetworkMode(this.networkMode),
            placementConstraints: core_1.Lazy.anyValue({
                produce: () => !isFargateCompatible(this.compatibility) ? this.placementConstraints : undefined,
            }, { omitEmptyArray: true }),
            proxyConfiguration: props.proxyConfiguration ? props.proxyConfiguration.bind(this.stack, this) : undefined,
            cpu: props.cpu,
            memory: props.memoryMiB,
            ipcMode: props.ipcMode,
            pidMode: props.pidMode,
        });
        if (props.placementConstraints) {
            props.placementConstraints.forEach(pc => this.addPlacementConstraint(pc));
        }
        this.taskDefinitionArn = taskDef.ref;
    }
    /**
     * (experimental) Imports a task definition from the specified task definition ARN.
     *
     * The task will have a compatibility of EC2+Fargate.
     *
     * @experimental
     */
    static fromTaskDefinitionArn(scope, id, taskDefinitionArn) {
        class Import extends TaskDefinitionBase {
            constructor() {
                super(...arguments);
                this.taskDefinitionArn = taskDefinitionArn;
                this.compatibility = Compatibility.EC2_AND_FARGATE;
                this.executionRole = undefined;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Execution role for this task definition.
     *
     * @experimental
     */
    get executionRole() {
        return this._executionRole;
    }
    renderVolumes() {
        return this.volumes.map(renderVolume);
        function renderVolume(spec) {
            return {
                host: spec.host,
                name: spec.name,
                dockerVolumeConfiguration: spec.dockerVolumeConfiguration && {
                    autoprovision: spec.dockerVolumeConfiguration.autoprovision,
                    driver: spec.dockerVolumeConfiguration.driver,
                    driverOpts: spec.dockerVolumeConfiguration.driverOpts,
                    labels: spec.dockerVolumeConfiguration.labels,
                    scope: spec.dockerVolumeConfiguration.scope,
                },
                efsVolumeConfiguration: spec.efsVolumeConfiguration && {
                    fileSystemId: spec.efsVolumeConfiguration.fileSystemId,
                    authorizationConfig: spec.efsVolumeConfiguration.authorizationConfig,
                    rootDirectory: spec.efsVolumeConfiguration.rootDirectory,
                    transitEncryption: spec.efsVolumeConfiguration.transitEncryption,
                    transitEncryptionPort: spec.efsVolumeConfiguration.transitEncryptionPort,
                },
            };
        }
    }
    /**
     * Validate the existence of the input target and set default values.
     *
     * @internal
     */
    _validateTarget(options) {
        const targetContainer = this.findContainer(options.containerName);
        if (targetContainer === undefined) {
            throw new Error(`No container named '${options.containerName}'. Did you call "addContainer()"?`);
        }
        const targetProtocol = options.protocol || container_definition_1.Protocol.TCP;
        const targetContainerPort = options.containerPort || targetContainer.containerPort;
        const portMapping = targetContainer.findPortMapping(targetContainerPort, targetProtocol);
        if (portMapping === undefined) {
            // eslint-disable-next-line max-len
            throw new Error(`Container '${targetContainer}' has no mapping for port ${options.containerPort} and protocol ${targetProtocol}. Did you call "container.addPortMappings()"?`);
        }
        return {
            containerName: options.containerName,
            portMapping,
        };
    }
    /**
     * Returns the port range to be opened that match the provided container name and container port.
     *
     * @internal
     */
    _portRangeFromPortMapping(portMapping) {
        if (portMapping.hostPort !== undefined && portMapping.hostPort !== 0) {
            return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.hostPort) : ec2.Port.tcp(portMapping.hostPort);
        }
        if (this.networkMode === NetworkMode.BRIDGE || this.networkMode === NetworkMode.NAT) {
            return EPHEMERAL_PORT_RANGE;
        }
        return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.containerPort) : ec2.Port.tcp(portMapping.containerPort);
    }
    /**
     * (experimental) Adds a policy statement to the task IAM role.
     *
     * @experimental
     */
    addToTaskRolePolicy(statement) {
        this.taskRole.addToPolicy(statement);
    }
    /**
     * (experimental) Adds a policy statement to the task execution IAM role.
     *
     * @experimental
     */
    addToExecutionRolePolicy(statement) {
        this.obtainExecutionRole().addToPolicy(statement);
    }
    /**
     * (experimental) Adds a new container to the task definition.
     *
     * @experimental
     */
    addContainer(id, props) {
        return new container_definition_1.ContainerDefinition(this, id, { taskDefinition: this, ...props });
    }
    /**
     * (experimental) Adds a firelens log router to the task definition.
     *
     * @experimental
     */
    addFirelensLogRouter(id, props) {
        // only one firelens log router is allowed in each task.
        if (this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
            throw new Error('Firelens log router is already added in this task.');
        }
        return new firelens_log_router_1.FirelensLogRouter(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Links a container to this task definition.
     * @internal
     */
    _linkContainer(container) {
        this.containers.push(container);
        if (this.defaultContainer === undefined && container.essential) {
            this.defaultContainer = container;
        }
    }
    /**
     * (experimental) Adds a volume to the task definition.
     *
     * @experimental
     */
    addVolume(volume) {
        this.volumes.push(volume);
    }
    /**
     * (experimental) Adds the specified placement constraint to the task definition.
     *
     * @experimental
     */
    addPlacementConstraint(constraint) {
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        this.placementConstraints.push(...constraint.toJson());
    }
    /**
     * (experimental) Adds the specified extention to the task definition.
     *
     * Extension can be used to apply a packaged modification to
     * a task definition.
     *
     * @experimental
     */
    addExtension(extension) {
        extension.extend(this);
    }
    /**
     * (experimental) Creates the task execution IAM role if it doesn't already exist.
     *
     * @experimental
     */
    obtainExecutionRole() {
        if (!this._executionRole) {
            this._executionRole = new iam.Role(this, 'ExecutionRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
        }
        return this._executionRole;
    }
    /**
     * (experimental) Validates the task definition.
     *
     * @experimental
     */
    validate() {
        const ret = super.validate();
        if (isEc2Compatible(this.compatibility)) {
            // EC2 mode validations
            // Container sizes
            for (const container of this.containers) {
                if (!container.memoryLimitSpecified) {
                    ret.push(`ECS Container ${container.containerName} must have at least one of 'memoryLimitMiB' or 'memoryReservationMiB' specified`);
                }
            }
        }
        return ret;
    }
    /**
     * Returns the container that match the provided containerName.
     */
    findContainer(containerName) {
        return this.containers.find(c => c.containerName === containerName);
    }
    renderNetworkMode(networkMode) {
        return (networkMode === NetworkMode.NAT) ? undefined : networkMode;
    }
    renderContainers() {
        // add firelens log router container if any application container is using firelens log driver,
        // also check if already created log router container
        for (const container of this.containers) {
            if (container.logDriverConfig && container.logDriverConfig.logDriver === 'awsfirelens'
                && !this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
                this.addFirelensLogRouter('log-router', {
                    image: firelens_log_router_1.obtainDefaultFluentBitECRImage(this, container.logDriverConfig),
                    firelensConfig: {
                        type: firelens_log_router_1.FirelensLogRouterType.FLUENTBIT,
                    },
                    logging: new aws_log_driver_1.AwsLogDriver({ streamPrefix: 'firelens' }),
                    memoryReservationMiB: 50,
                });
                break;
            }
        }
        return this.containers.map(x => x.renderContainerDefinition());
    }
}
exports.TaskDefinition = TaskDefinition;
/**
 * The port range to open up for dynamic port mapping
 */
const EPHEMERAL_PORT_RANGE = ec2.Port.tcpRange(32768, 65535);
/**
 * (experimental) The networking mode to use for the containers in the task.
 *
 * @experimental
 */
var NetworkMode;
(function (NetworkMode) {
    NetworkMode["NONE"] = "none";
    NetworkMode["BRIDGE"] = "bridge";
    NetworkMode["AWS_VPC"] = "awsvpc";
    NetworkMode["HOST"] = "host";
    NetworkMode["NAT"] = "nat";
})(NetworkMode = exports.NetworkMode || (exports.NetworkMode = {}));
/**
 * (experimental) The IPC resource namespace to use for the containers in the task.
 *
 * @experimental
 */
var IpcMode;
(function (IpcMode) {
    IpcMode["NONE"] = "none";
    IpcMode["HOST"] = "host";
    IpcMode["TASK"] = "task";
})(IpcMode = exports.IpcMode || (exports.IpcMode = {}));
/**
 * (experimental) The process namespace to use for the containers in the task.
 *
 * @experimental
 */
var PidMode;
(function (PidMode) {
    PidMode["HOST"] = "host";
    PidMode["TASK"] = "task";
})(PidMode = exports.PidMode || (exports.PidMode = {}));
/**
 * (experimental) The scope for the Docker volume that determines its lifecycle.
 *
 * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
 * Docker volumes that are scoped as shared persist after the task stops.
 *
 * @experimental
 */
var Scope;
(function (Scope) {
    Scope["TASK"] = "task";
    Scope["SHARED"] = "shared";
})(Scope = exports.Scope || (exports.Scope = {}));
/**
 * (experimental) The task launch type compatibility requirement.
 *
 * @experimental
 */
var Compatibility;
(function (Compatibility) {
    Compatibility[Compatibility["EC2"] = 0] = "EC2";
    Compatibility[Compatibility["FARGATE"] = 1] = "FARGATE";
    Compatibility[Compatibility["EC2_AND_FARGATE"] = 2] = "EC2_AND_FARGATE";
})(Compatibility = exports.Compatibility || (exports.Compatibility = {}));
/**
 * Return true if the given task definition can be run on an EC2 cluster
 */
function isEc2Compatible(compatibility) {
    return [Compatibility.EC2, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
/**
 * Return true if the given task definition can be run on a Fargate cluster
 */
function isFargateCompatible(compatibility) {
    return [Compatibility.FARGATE, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
//# sourceMappingURL=data:application/json;base64,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