"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkLoadBalancedServiceBase = exports.NetworkLoadBalancedServiceRecordType = void 0;
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const aws_elasticloadbalancingv2_1 = require("../../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const aws_route53_1 = require("../../../aws-route53"); // Automatically re-written from '@aws-cdk/aws-route53'
const aws_route53_targets_1 = require("../../../aws-route53-targets"); // Automatically re-written from '@aws-cdk/aws-route53-targets'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * (experimental) Describes the type of DNS record the service should create.
 *
 * @experimental
 */
var NetworkLoadBalancedServiceRecordType;
(function (NetworkLoadBalancedServiceRecordType) {
    NetworkLoadBalancedServiceRecordType[NetworkLoadBalancedServiceRecordType["ALIAS"] = 0] = "ALIAS";
    NetworkLoadBalancedServiceRecordType[NetworkLoadBalancedServiceRecordType["CNAME"] = 1] = "CNAME";
    NetworkLoadBalancedServiceRecordType[NetworkLoadBalancedServiceRecordType["NONE"] = 2] = "NONE";
})(NetworkLoadBalancedServiceRecordType = exports.NetworkLoadBalancedServiceRecordType || (exports.NetworkLoadBalancedServiceRecordType = {}));
/**
 * (experimental) The base class for NetworkLoadBalancedEc2Service and NetworkLoadBalancedFargateService services.
 *
 * @experimental
 */
class NetworkLoadBalancedServiceBase extends cdk.Construct {
    /**
     * (experimental) Constructs a new instance of the NetworkLoadBalancedServiceBase class.
     *
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        this.desiredCount = props.desiredCount || 1;
        const internetFacing = props.publicLoadBalancer !== undefined ? props.publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        const loadBalancer = props.loadBalancer !== undefined ? props.loadBalancer :
            new aws_elasticloadbalancingv2_1.NetworkLoadBalancer(this, 'LB', lbProps);
        const listenerPort = props.listenerPort !== undefined ? props.listenerPort : 80;
        const targetProps = {
            port: 80,
        };
        this.listener = loadBalancer.addListener('PublicListener', { port: listenerPort });
        this.targetGroup = this.listener.addTargets('ECS', targetProps);
        if (typeof props.domainName !== 'undefined') {
            if (typeof props.domainZone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            switch ((_a = props.recordType) !== null && _a !== void 0 ? _a : NetworkLoadBalancedServiceRecordType.ALIAS) {
                case NetworkLoadBalancedServiceRecordType.ALIAS:
                    new aws_route53_1.ARecord(this, 'DNS', {
                        zone: props.domainZone,
                        recordName: props.domainName,
                        target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
                    });
                    break;
                case NetworkLoadBalancedServiceRecordType.CNAME:
                    new aws_route53_1.CnameRecord(this, 'DNS', {
                        zone: props.domainZone,
                        recordName: props.domainName,
                        domainName: loadBalancer.loadBalancerDnsName,
                    });
                    break;
                case NetworkLoadBalancedServiceRecordType.NONE:
                    // Do not create a DNS record
                    break;
            }
        }
        if (loadBalancer instanceof aws_elasticloadbalancingv2_1.NetworkLoadBalancer) {
            this._networkLoadBalancer = loadBalancer;
        }
        if (props.loadBalancer === undefined) {
            new cdk.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
        }
    }
    /**
     * (experimental) The Network Load Balancer for the service.
     *
     * @experimental
     */
    get loadBalancer() {
        if (!this._networkLoadBalancer) {
            throw new Error('.loadBalancer can only be accessed if the class was constructed with an owned, not imported, load balancer');
        }
        return this._networkLoadBalancer;
    }
    /**
     * (experimental) Returns the default cluster.
     *
     * @experimental
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = cdk.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * (experimental) Adds service as a target of the target group.
     *
     * @experimental
     */
    addServiceAsTarget(service) {
        this.targetGroup.addTarget(service);
    }
    /**
     * @experimental
     */
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
exports.NetworkLoadBalancedServiceBase = NetworkLoadBalancedServiceBase;
//# sourceMappingURL=data:application/json;base64,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