"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationMultipleTargetGroupsServiceBase = void 0;
const aws_certificatemanager_1 = require("../../../aws-certificatemanager"); // Automatically re-written from '@aws-cdk/aws-certificatemanager'
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const aws_elasticloadbalancingv2_1 = require("../../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const aws_route53_1 = require("../../../aws-route53"); // Automatically re-written from '@aws-cdk/aws-route53'
const aws_route53_targets_1 = require("../../../aws-route53-targets"); // Automatically re-written from '@aws-cdk/aws-route53-targets'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * (experimental) The base class for ApplicationMultipleTargetGroupsEc2Service and ApplicationMultipleTargetGroupsFargateService classes.
 *
 * @experimental
 */
class ApplicationMultipleTargetGroupsServiceBase extends core_2.Construct {
    /**
     * (experimental) Constructs a new instance of the ApplicationMultipleTargetGroupsServiceBase class.
     *
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * @experimental
         */
        this.listeners = new Array();
        /**
         * @experimental
         */
        this.targetGroups = new Array();
        this.loadBalancers = new Array();
        this.validateInput(props);
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        this.desiredCount = props.desiredCount || 1;
        if (props.taskImageOptions) {
            this.logDriver = this.createLogDriver(props.taskImageOptions.enableLogging, props.taskImageOptions.logDriver);
        }
        if (props.loadBalancers) {
            for (const lbProps of props.loadBalancers) {
                const lb = this.createLoadBalancer(lbProps.name, lbProps.publicLoadBalancer);
                this.loadBalancers.push(lb);
                const protocolType = new Set();
                for (const listenerProps of lbProps.listeners) {
                    const protocol = this.createListenerProtocol(listenerProps.protocol, listenerProps.certificate);
                    if (listenerProps.certificate !== undefined && protocol !== undefined && protocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                        throw new Error('The HTTPS protocol must be used when a certificate is given');
                    }
                    protocolType.add(protocol);
                    const listener = this.configListener(protocol, {
                        certificate: listenerProps.certificate,
                        domainName: lbProps.domainName,
                        domainZone: lbProps.domainZone,
                        listenerName: listenerProps.name,
                        loadBalancer: lb,
                        port: listenerProps.port,
                    });
                    this.listeners.push(listener);
                }
                const domainName = this.createDomainName(lb, lbProps.domainName, lbProps.domainZone);
                new core_1.CfnOutput(this, `LoadBalancerDNS${lb.node.id}`, { value: lb.loadBalancerDnsName });
                for (const protocol of protocolType) {
                    new core_1.CfnOutput(this, `ServiceURL${lb.node.id}${protocol.toLowerCase()}`, { value: protocol.toLowerCase() + '://' + domainName });
                }
            }
            // set up default load balancer and listener.
            this.loadBalancer = this.loadBalancers[0];
            this.listener = this.listeners[0];
        }
        else {
            this.loadBalancer = this.createLoadBalancer('LB');
            const protocol = this.createListenerProtocol();
            this.listener = this.configListener(protocol, {
                listenerName: 'PublicListener',
                loadBalancer: this.loadBalancer,
            });
            const domainName = this.createDomainName(this.loadBalancer);
            new core_1.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
            new core_1.CfnOutput(this, 'ServiceURL', { value: protocol.toLowerCase() + '://' + domainName });
        }
    }
    /**
     * (experimental) Returns the default cluster.
     *
     * @experimental
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs.
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * @experimental
     */
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
    /**
     * @experimental
     */
    findListener(name) {
        if (!name) {
            return this.listener;
        }
        for (const listener of this.listeners) {
            if (listener.node.id === name) {
                return listener;
            }
        }
        throw new Error(`Listener ${name} is not defined. Did you define listener with name ${name}?`);
    }
    /**
     * @experimental
     */
    registerECSTargets(service, container, targets) {
        for (const targetProps of targets) {
            const targetGroup = this.findListener(targetProps.listener).addTargets(`ECSTargetGroup${container.containerName}${targetProps.containerPort}`, {
                port: 80,
                targets: [
                    service.loadBalancerTarget({
                        containerName: container.containerName,
                        containerPort: targetProps.containerPort,
                        protocol: targetProps.protocol,
                    }),
                ],
                hostHeader: targetProps.hostHeader,
                pathPattern: targetProps.pathPattern,
                priority: targetProps.priority,
            });
            this.targetGroups.push(targetGroup);
        }
        if (this.targetGroups.length === 0) {
            throw new Error('At least one target group should be specified.');
        }
        return this.targetGroups[0];
    }
    /**
     * @experimental
     */
    addPortMappingForTargets(container, targets) {
        for (const target of targets) {
            if (!container.findPortMapping(target.containerPort, target.protocol || aws_ecs_1.Protocol.TCP)) {
                container.addPortMappings({
                    containerPort: target.containerPort,
                    protocol: target.protocol,
                });
            }
        }
    }
    /**
     * Create log driver if logging is enabled.
     */
    createLogDriver(enableLoggingProp, logDriverProp) {
        const enableLogging = enableLoggingProp !== undefined ? enableLoggingProp : true;
        const logDriver = logDriverProp !== undefined
            ? logDriverProp : enableLogging
            ? this.createAWSLogDriver(this.node.id) : undefined;
        return logDriver;
    }
    configListener(protocol, props) {
        const listener = this.createListener(props.listenerName, props.loadBalancer, protocol, props.port);
        let certificate;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            certificate = this.createListenerCertificate(props.listenerName, props.certificate, props.domainName, props.domainZone);
        }
        else {
            certificate = undefined;
        }
        if (certificate !== undefined) {
            listener.addCertificateArns(`Arns${props.listenerName}`, [certificate.certificateArn]);
        }
        return listener;
    }
    validateInput(props) {
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        if (props.loadBalancers) {
            if (props.loadBalancers.length === 0) {
                throw new Error('At least one load balancer must be specified');
            }
            for (const lbProps of props.loadBalancers) {
                if (lbProps.listeners.length === 0) {
                    throw new Error('At least one listener must be specified');
                }
            }
        }
    }
    createLoadBalancer(name, publicLoadBalancer) {
        const internetFacing = publicLoadBalancer !== undefined ? publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        return new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, name, lbProps);
    }
    createListenerProtocol(listenerProtocol, certificate) {
        return listenerProtocol !== undefined ? listenerProtocol : (certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
    }
    createListenerCertificate(listenerName, certificate, domainName, domainZone) {
        if (typeof domainName === 'undefined' || typeof domainZone === 'undefined') {
            throw new Error('A domain name and zone is required when using the HTTPS protocol');
        }
        if (certificate !== undefined) {
            return certificate;
        }
        else {
            return new aws_certificatemanager_1.Certificate(this, `Certificate${listenerName}`, {
                domainName,
                validation: aws_certificatemanager_1.CertificateValidation.fromDns(domainZone),
            });
        }
    }
    createListener(name, lb, protocol, port) {
        return lb.addListener(name, {
            protocol,
            open: true,
            port,
        });
    }
    createDomainName(loadBalancer, name, zone) {
        let domainName = loadBalancer.loadBalancerDnsName;
        if (typeof name !== 'undefined') {
            if (typeof zone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            const record = new aws_route53_1.ARecord(this, `DNS${loadBalancer.node.id}`, {
                zone,
                recordName: name,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
            domainName = record.domainName;
        }
        return domainName;
    }
}
exports.ApplicationMultipleTargetGroupsServiceBase = ApplicationMultipleTargetGroupsServiceBase;
//# sourceMappingURL=data:application/json;base64,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