"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerImageAsset = void 0;
const fs = require("fs");
const path = require("path");
const assets = require("../../assets"); // Automatically re-written from '@aws-cdk/assets'
const ecr = require("../../aws-ecr"); // Automatically re-written from '@aws-cdk/aws-ecr'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const minimatch = require("minimatch");
/**
 * (experimental) An asset that represents a Docker image.
 *
 * The image will be created in build time and uploaded to an ECR repository.
 *
 * @experimental
 */
class DockerImageAsset extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        // none of the properties use tokens
        validateProps(props);
        // resolve full path
        const dir = path.resolve(props.directory);
        if (!fs.existsSync(dir)) {
            throw new Error(`Cannot find image directory at ${dir}`);
        }
        // validate the docker file exists
        const file = path.join(dir, props.file || 'Dockerfile');
        if (!fs.existsSync(file)) {
            throw new Error(`Cannot find file at ${file}`);
        }
        let exclude = props.exclude || [];
        const ignore = path.join(dir, '.dockerignore');
        if (fs.existsSync(ignore)) {
            exclude = [...exclude, ...fs.readFileSync(ignore).toString().split('\n').filter(e => !!e)];
        }
        // make sure the docker file and the dockerignore file end up in the staging area
        // see https://github.com/aws/aws-cdk/issues/6004
        exclude = exclude.filter(ignoreExpression => {
            return !(minimatch(file, ignoreExpression, { matchBase: true }) ||
                minimatch(ignore, ignoreExpression, { matchBase: true }));
        });
        if (props.repositoryName) {
            core_1.Annotations.of(this).addWarning('DockerImageAsset.repositoryName is deprecated. Override "core.Stack.addDockerImageAsset" to control asset locations');
        }
        // include build context in "extra" so it will impact the hash
        const extraHash = {};
        if (props.extraHash) {
            extraHash.user = props.extraHash;
        }
        if (props.buildArgs) {
            extraHash.buildArgs = props.buildArgs;
        }
        if (props.target) {
            extraHash.target = props.target;
        }
        if (props.file) {
            extraHash.file = props.file;
        }
        if (props.repositoryName) {
            extraHash.repositoryName = props.repositoryName;
        }
        // add "salt" to the hash in order to invalidate the image in the upgrade to
        // 1.21.0 which removes the AdoptedRepository resource (and will cause the
        // deletion of the ECR repository the app used).
        extraHash.version = '1.21.0';
        const staging = new assets.Staging(this, 'Staging', {
            ...props,
            exclude,
            sourcePath: dir,
            extraHash: Object.keys(extraHash).length === 0
                ? undefined
                : JSON.stringify(extraHash),
        });
        this.sourceHash = staging.sourceHash;
        const stack = core_1.Stack.of(this);
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: staging.relativeStagedPath(stack),
            dockerBuildArgs: props.buildArgs,
            dockerBuildTarget: props.target,
            dockerFile: props.file,
            repositoryName: props.repositoryName,
            sourceHash: staging.sourceHash,
        });
        this.repository = ecr.Repository.fromRepositoryName(this, 'Repository', location.repositoryName);
        this.imageUri = location.imageUri;
    }
}
exports.DockerImageAsset = DockerImageAsset;
function validateProps(props) {
    for (const [key, value] of Object.entries(props)) {
        if (core_1.Token.isUnresolved(value)) {
            throw new Error(`Cannot use Token as value of '${key}': this value is used before deployment starts`);
        }
    }
    validateBuildArgs(props.buildArgs);
}
function validateBuildArgs(buildArgs) {
    for (const [key, value] of Object.entries(buildArgs || {})) {
        if (core_1.Token.isUnresolved(key) || core_1.Token.isUnresolved(value)) {
            throw new Error('Cannot use tokens in keys or values of "buildArgs" since they are needed before deployment');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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