"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowLog = exports.FlowLogDestination = exports.FlowLogResourceType = exports.FlowLogDestinationType = exports.FlowLogTrafficType = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ec2_generated_1 = require("./ec2.generated");
/**
 * (experimental) The type of VPC traffic to log.
 *
 * @experimental
 */
var FlowLogTrafficType;
(function (FlowLogTrafficType) {
    FlowLogTrafficType["ACCEPT"] = "ACCEPT";
    FlowLogTrafficType["ALL"] = "ALL";
    FlowLogTrafficType["REJECT"] = "REJECT";
})(FlowLogTrafficType = exports.FlowLogTrafficType || (exports.FlowLogTrafficType = {}));
/**
 * (experimental) The available destination types for Flow Logs.
 *
 * @experimental
 */
var FlowLogDestinationType;
(function (FlowLogDestinationType) {
    FlowLogDestinationType["CLOUD_WATCH_LOGS"] = "cloud-watch-logs";
    FlowLogDestinationType["S3"] = "s3";
})(FlowLogDestinationType = exports.FlowLogDestinationType || (exports.FlowLogDestinationType = {}));
/**
 * (experimental) The type of resource to create the flow log for.
 *
 * @experimental
 */
class FlowLogResourceType {
    /**
     * (experimental) The subnet to attach the Flow Log to.
     *
     * @experimental
     */
    static fromSubnet(subnet) {
        return {
            resourceType: 'Subnet',
            resourceId: subnet.subnetId,
        };
    }
    /**
     * (experimental) The VPC to attach the Flow Log to.
     *
     * @experimental
     */
    static fromVpc(vpc) {
        return {
            resourceType: 'VPC',
            resourceId: vpc.vpcId,
        };
    }
    /**
     * (experimental) The Network Interface to attach the Flow Log to.
     *
     * @experimental
     */
    static fromNetworkInterfaceId(id) {
        return {
            resourceType: 'NetworkInterface',
            resourceId: id,
        };
    }
}
exports.FlowLogResourceType = FlowLogResourceType;
/**
 * (experimental) The destination type for the flow log.
 *
 * @experimental
 */
class FlowLogDestination {
    /**
     * (experimental) Use CloudWatch logs as the destination.
     *
     * @experimental
     */
    static toCloudWatchLogs(logGroup, iamRole) {
        return new CloudWatchLogsDestination({
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        });
    }
    /**
     * (experimental) Use S3 as the destination.
     *
     * @experimental
     */
    static toS3(bucket, keyPrefix) {
        return new S3Destination({
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket: bucket,
            keyPrefix,
        });
    }
}
exports.FlowLogDestination = FlowLogDestination;
/**
 * @experimental
 */
class S3Destination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let s3Bucket;
        if (this.props.s3Bucket === undefined) {
            s3Bucket = new s3.Bucket(scope, 'Bucket', {
                encryption: s3.BucketEncryption.UNENCRYPTED,
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        else {
            s3Bucket = this.props.s3Bucket;
        }
        return {
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket,
            keyPrefix: this.props.keyPrefix,
        };
    }
}
/**
 * @experimental
 */
class CloudWatchLogsDestination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let iamRole;
        let logGroup;
        if (this.props.iamRole === undefined) {
            iamRole = new iam.Role(scope, 'IAMRole', {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
        }
        else {
            iamRole = this.props.iamRole;
        }
        if (this.props.logGroup === undefined) {
            logGroup = new logs.LogGroup(scope, 'LogGroup');
        }
        else {
            logGroup = this.props.logGroup;
        }
        iamRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'logs:DescribeLogStreams',
            ],
            effect: iam.Effect.ALLOW,
            resources: [logGroup.logGroupArn],
        }));
        iamRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            effect: iam.Effect.ALLOW,
            resources: [iamRole.roleArn],
        }));
        return {
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        };
    }
}
/**
 * The base class for a Flow Log
 *
 * @experimental
 */
class FlowLogBase extends core_1.Resource {
}
/**
 * (experimental) A VPC flow log.
 *
 * @experimental
 * @resource AWS::EC2::FlowLog
 */
class FlowLog extends FlowLogBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.flowLogName,
        });
        const destination = props.destination || FlowLogDestination.toCloudWatchLogs();
        const destinationConfig = destination.bind(this, this);
        this.logGroup = destinationConfig.logGroup;
        this.bucket = destinationConfig.s3Bucket;
        this.iamRole = destinationConfig.iamRole;
        this.keyPrefix = destinationConfig.keyPrefix;
        let logDestination = undefined;
        if (this.bucket) {
            logDestination = this.keyPrefix ? this.bucket.arnForObjects(this.keyPrefix) : this.bucket.bucketArn;
        }
        const flowLog = new ec2_generated_1.CfnFlowLog(this, 'FlowLog', {
            deliverLogsPermissionArn: this.iamRole ? this.iamRole.roleArn : undefined,
            logDestinationType: destinationConfig.logDestinationType,
            logGroupName: this.logGroup ? this.logGroup.logGroupName : undefined,
            resourceId: props.resourceType.resourceId,
            resourceType: props.resourceType.resourceType,
            trafficType: props.trafficType
                ? props.trafficType
                : FlowLogTrafficType.ALL,
            logDestination,
        });
        this.flowLogId = flowLog.ref;
    }
    /**
     * (experimental) Import a Flow Log by it's Id.
     *
     * @experimental
     */
    static fromFlowLogId(scope, id, flowLogId) {
        class Import extends FlowLogBase {
            constructor() {
                super(...arguments);
                this.flowLogId = flowLogId;
            }
        }
        return new Import(scope, id);
    }
}
exports.FlowLog = FlowLog;
//# sourceMappingURL=data:application/json;base64,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