"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.allRouteTableIds = exports.range = exports.ImportSubnetGroup = exports.subnetId = exports.subnetGroupNameFromConstructId = exports.defaultSubnetName = exports.slugify = void 0;
const vpc_1 = require("./vpc");
/**
 * Turn an arbitrary string into one that can be used as a CloudFormation identifier by stripping special characters
 *
 * (At the moment, no efforts are taken to prevent collissions, but we can add that later when it becomes necessary).
 */
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
exports.slugify = slugify;
/**
 * The default names for every subnet type
 */
function defaultSubnetName(type) {
    switch (type) {
        case vpc_1.SubnetType.PUBLIC: return 'Public';
        case vpc_1.SubnetType.PRIVATE: return 'Private';
        case vpc_1.SubnetType.ISOLATED: return 'Isolated';
    }
}
exports.defaultSubnetName = defaultSubnetName;
/**
 * Return a subnet name from its construct ID
 *
 * All subnet names look like NAME <> "Subnet" <> INDEX
 */
function subnetGroupNameFromConstructId(subnet) {
    return subnet.node.id.replace(/Subnet\d+$/, '');
}
exports.subnetGroupNameFromConstructId = subnetGroupNameFromConstructId;
/**
 * Make the subnet construct ID from a name and number
 */
function subnetId(name, i) {
    return `${name}Subnet${i + 1}`;
}
exports.subnetId = subnetId;
class ImportSubnetGroup {
    constructor(subnetIds, names, routeTableIds, type, availabilityZones, idField, nameField, routeTableIdField) {
        this.availabilityZones = availabilityZones;
        this.subnetIds = subnetIds || [];
        this.routeTableIds = routeTableIds || [];
        this.groups = this.subnetIds.length / this.availabilityZones.length;
        if (Math.floor(this.groups) !== this.groups) {
            // eslint-disable-next-line max-len
            throw new Error(`Number of ${idField} (${this.subnetIds.length}) must be a multiple of availability zones (${this.availabilityZones.length}).`);
        }
        if (this.routeTableIds.length !== this.subnetIds.length && routeTableIds != null) {
            // We don't err if no routeTableIds were provided to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            /* eslint-disable max-len */
            throw new Error(`Number of ${routeTableIdField} (${this.routeTableIds.length}) must be equal to the amount of ${idField} (${this.subnetIds.length}).`);
        }
        this.names = this.normalizeNames(names, defaultSubnetName(type), nameField);
    }
    import(scope) {
        return range(this.subnetIds.length).map(i => {
            const k = Math.floor(i / this.availabilityZones.length);
            return vpc_1.Subnet.fromSubnetAttributes(scope, subnetId(this.names[k], i), {
                availabilityZone: this.pickAZ(i),
                subnetId: this.subnetIds[i],
                routeTableId: this.routeTableIds[i],
            });
        });
    }
    /**
     * Return a list with a name for every subnet
     */
    normalizeNames(names, defaultName, fieldName) {
        // If not given, return default
        if (names === undefined || names.length === 0) {
            return [defaultName];
        }
        // If given, must match given subnets
        if (names.length !== this.groups) {
            throw new Error(`${fieldName} must have an entry for every corresponding subnet group, got: ${JSON.stringify(names)}`);
        }
        return names;
    }
    /**
     * Return the i'th AZ
     */
    pickAZ(i) {
        return this.availabilityZones[i % this.availabilityZones.length];
    }
}
exports.ImportSubnetGroup = ImportSubnetGroup;
/**
 * Generate the list of numbers of [0..n)
 */
function range(n) {
    const ret = [];
    for (let i = 0; i < n; i++) {
        ret.push(i);
    }
    return ret;
}
exports.range = range;
/**
 * Return the union of table IDs from all selected subnets
 */
function allRouteTableIds(subnets) {
    const ret = new Set();
    for (const subnet of subnets) {
        if (subnet.routeTable && subnet.routeTable.routeTableId) {
            ret.add(subnet.routeTable.routeTableId);
        }
    }
    return Array.from(ret);
}
exports.allRouteTableIds = allRouteTableIds;
function flatten(xs) {
    return Array.prototype.concat.apply([], xs);
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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