"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserData = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const machine_image_1 = require("./machine-image");
/**
 * (experimental) Instance User Data.
 *
 * @experimental
 */
class UserData {
    /**
     * (experimental) Create a userdata object for Linux hosts.
     *
     * @experimental
     */
    static forLinux(options = {}) {
        return new LinuxUserData(options);
    }
    /**
     * (experimental) Create a userdata object for Windows hosts.
     *
     * @experimental
     */
    static forWindows() {
        return new WindowsUserData();
    }
    /**
     * (experimental) Create a userdata object with custom content.
     *
     * @experimental
     */
    static custom(content) {
        const userData = new CustomUserData();
        userData.addCommands(content);
        return userData;
    }
    /**
     * @experimental
     */
    static forOperatingSystem(os) {
        switch (os) {
            case machine_image_1.OperatingSystemType.LINUX: return UserData.forLinux();
            case machine_image_1.OperatingSystemType.WINDOWS: return UserData.forWindows();
            case machine_image_1.OperatingSystemType.UNKNOWN: throw new Error('Cannot determine UserData for unknown operating system type');
        }
    }
}
exports.UserData = UserData;
/**
 * Linux Instance User Data
 */
class LinuxUserData extends UserData {
    constructor(props = {}) {
        super();
        this.props = props;
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        const shebang = this.props.shebang !== undefined ? this.props.shebang : '#!/bin/bash';
        return [shebang, ...(this.renderOnExitLines()), ...this.lines].join('\n');
    }
    addS3DownloadCommand(params) {
        const s3Path = `s3://${params.bucket.bucketName}/${params.bucketKey}`;
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `/tmp/${params.bucketKey}`;
        this.addCommands(`mkdir -p $(dirname '${localPath}')`, `aws s3 cp '${s3Path}' '${localPath}'`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        var _a;
        this.addCommands('set -e', `chmod +x '${params.filePath}'`, `'${params.filePath}' ${(_a = params.arguments) !== null && _a !== void 0 ? _a : ''}`.trim());
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`/opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} -e $exitCode || echo 'Failed to send Cloudformation Signal'`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['function exitTrap(){', 'exitCode=$?', ...this.onExitLines, '}', 'trap exitTrap EXIT'];
        }
        return [];
    }
}
/**
 * Windows Instance User Data
 */
class WindowsUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        return `<powershell>${[...(this.renderOnExitLines()),
            ...this.lines,
            ...(this.onExitLines.length > 0 ? ['throw "Success"'] : [])].join('\n')}</powershell>`;
    }
    addS3DownloadCommand(params) {
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `C:/temp/${params.bucketKey}`;
        this.addCommands(`mkdir (Split-Path -Path '${localPath}' ) -ea 0`, `Read-S3Object -BucketName '${params.bucket.bucketName}' -key '${params.bucketKey}' -file '${localPath}' -ErrorAction Stop`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        var _a;
        this.addCommands(`&'${params.filePath}' ${(_a = params.arguments) !== null && _a !== void 0 ? _a : ''}`.trim(), `if (!$?) { Write-Error 'Failed to execute the file "${params.filePath}"' -ErrorAction Stop }`);
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} --success ($success.ToString().ToLower())`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['trap {', '$success=($PSItem.Exception.Message -eq "Success")', ...this.onExitLines, 'break', '}'];
        }
        return [];
    }
}
/**
 * Custom Instance User Data
 */
class CustomUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands() {
        throw new Error('CustomUserData does not support addOnExitCommands, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    render() {
        return this.lines.join('\n');
    }
    addS3DownloadCommand() {
        throw new Error('CustomUserData does not support addS3DownloadCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addExecuteFileCommand() {
        throw new Error('CustomUserData does not support addExecuteFileCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addSignalOnExitCommand() {
        throw new Error('CustomUserData does not support addSignalOnExitCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
}
//# sourceMappingURL=data:application/json;base64,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