"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubnetFilter = void 0;
const network_util_1 = require("./network-util");
/**
 * (experimental) Contains logic which chooses a set of subnets from a larger list, in conjunction with SubnetSelection, to determine where to place AWS resources such as VPC endpoints, EC2 instances, etc.
 *
 * @experimental
 */
class SubnetFilter {
    /**
     * (experimental) Chooses subnets which are in one of the given availability zones.
     *
     * @experimental
     */
    static availabilityZones(availabilityZones) {
        return new AvailabilityZoneSubnetFilter(availabilityZones);
    }
    /**
     * (experimental) Chooses subnets such that there is at most one per availability zone.
     *
     * @experimental
     */
    static onePerAz() {
        return new OnePerAZSubnetFilter();
    }
    /**
     * (experimental) Chooses subnets which contain any of the specified IP addresses.
     *
     * @experimental
     */
    static containsIpAddresses(ipv4addrs) {
        return new ContainsIpAddressesSubnetFilter(ipv4addrs);
    }
    /**
     * (experimental) Executes the subnet filtering logic, returning a filtered set of subnets.
     *
     * @experimental
     */
    selectSubnets(_subnets) {
        throw new Error('Cannot select subnets with an abstract SubnetFilter. `selectSubnets` needs to be implmemented.');
    }
}
exports.SubnetFilter = SubnetFilter;
/**
 * Chooses subnets which are in one of the given availability zones.
 */
class AvailabilityZoneSubnetFilter extends SubnetFilter {
    constructor(availabilityZones) {
        super();
        this.availabilityZones = availabilityZones;
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return subnets.filter(s => this.availabilityZones.includes(s.availabilityZone));
    }
}
/**
 * Chooses subnets such that there is at most one per availability zone.
 */
class OnePerAZSubnetFilter extends SubnetFilter {
    constructor() {
        super();
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return this.retainOnePerAz(subnets);
    }
    retainOnePerAz(subnets) {
        const azsSeen = new Set();
        return subnets.filter(subnet => {
            if (azsSeen.has(subnet.availabilityZone)) {
                return false;
            }
            azsSeen.add(subnet.availabilityZone);
            return true;
        });
    }
}
/**
 * Chooses subnets which contain any of the specified IP addresses.
 */
class ContainsIpAddressesSubnetFilter extends SubnetFilter {
    constructor(ipAddresses) {
        super();
        this.ipAddresses = ipAddresses;
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return this.retainByIp(subnets, this.ipAddresses);
    }
    retainByIp(subnets, ips) {
        const cidrBlockObjs = ips.map(ip => {
            const ipNum = network_util_1.NetworkUtils.ipToNum(ip);
            return new network_util_1.CidrBlock(ipNum, 32);
        });
        return subnets.filter(s => {
            const subnetCidrBlock = new network_util_1.CidrBlock(s.ipv4CidrBlock);
            return cidrBlockObjs.some(cidr => subnetCidrBlock.containsCidr(cidr));
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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