"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitSource = exports.InitService = exports.InitPackage = exports.InitUser = exports.InitGroup = exports.InitFile = exports.InitCommand = exports.InitCommandWaitDuration = exports.InitElement = exports.InitServiceRestartHandle = void 0;
const fs = require("fs");
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * (experimental) An object that represents reasons to restart an InitService.
 *
 * Pass an instance of this object to the `InitFile`, `InitCommand`,
 * `InitSource` and `InitPackage` objects, and finally to an `InitService`
 * itself to cause the actions (files, commands, sources, and packages)
 * to trigger a restart of the service.
 *
 * For example, the following will run a custom command to install Nginx,
 * and trigger the nginx service to be restarted after the command has run.
 *
 * ```ts
 * const handle = new ec2.InitServiceRestartHandle();
 * ec2.CloudFormationInit.fromElements(
 *    ec2.InitCommand.shellCommand('/usr/bin/custom-nginx-install.sh', { serviceRestartHandles: [handle] }),
 *    ec2.InitService.enable('nginx', { serviceRestartHandle: handle }),
 * );
 * ```
 *
 * @experimental
 */
class InitServiceRestartHandle {
    constructor() {
        this.commands = new Array();
        this.files = new Array();
        this.sources = new Array();
        this.packages = {};
    }
    /**
     * Add a command key to the restart set
     * @internal
     */
    _addCommand(key) {
        return this.commands.push(key);
    }
    /**
     * Add a file key to the restart set
     * @internal
     */
    _addFile(key) {
        return this.files.push(key);
    }
    /**
     * Add a source key to the restart set
     * @internal
     */
    _addSource(key) {
        return this.sources.push(key);
    }
    /**
     * Add a package key to the restart set
     * @internal
     */
    _addPackage(packageType, key) {
        if (!this.packages[packageType]) {
            this.packages[packageType] = [];
        }
        this.packages[packageType].push(key);
    }
    /**
     * Render the restart handles for use in an InitService declaration
     * @internal
     */
    _renderRestartHandles() {
        const nonEmpty = (x) => x.length > 0 ? x : undefined;
        return {
            commands: nonEmpty(this.commands),
            files: nonEmpty(this.files),
            packages: Object.keys(this.packages).length > 0 ? this.packages : undefined,
            sources: nonEmpty(this.sources),
        };
    }
}
exports.InitServiceRestartHandle = InitServiceRestartHandle;
/**
 * (experimental) Base class for all CloudFormation Init elements.
 *
 * @experimental
 */
class InitElement {
}
exports.InitElement = InitElement;
/**
 * (experimental) Represents a duration to wait after a command has finished, in case of a reboot (Windows only).
 *
 * @experimental
 */
class InitCommandWaitDuration {
    /**
     * (experimental) Wait for a specified duration after a command.
     *
     * @experimental
     */
    static of(duration) {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return duration.toSeconds(); }
        }();
    }
    /**
     * (experimental) Do not wait for this command.
     *
     * @experimental
     */
    static none() {
        return InitCommandWaitDuration.of(core_1.Duration.seconds(0));
    }
    /**
     * (experimental) cfn-init will exit and resume only after a reboot.
     *
     * @experimental
     */
    static forever() {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return 'forever'; }
        }();
    }
}
exports.InitCommandWaitDuration = InitCommandWaitDuration;
/**
 * (experimental) Command to execute on the instance.
 *
 * @experimental
 */
class InitCommand extends InitElement {
    constructor(command, options) {
        super();
        this.command = command;
        this.options = options;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.COMMAND.toString();
    }
    /**
     * (experimental) Run a shell command.
     *
     * Remember that some characters like `&`, `|`, `;`, `>` etc. have special meaning in a shell and
     * need to be preceded by a `\` if you want to treat them as part of a filename.
     *
     * @experimental
     */
    static shellCommand(shellCommand, options = {}) {
        return new InitCommand(shellCommand, options);
    }
    /**
     * (experimental) Run a command from an argv array.
     *
     * You do not need to escape space characters or enclose command parameters in quotes.
     *
     * @experimental
     */
    static argvCommand(argv, options = {}) {
        if (argv.length === 0) {
            throw new Error('Cannot define argvCommand with an empty arguments');
        }
        return new InitCommand(argv, options);
    }
    /** @internal */
    _bind(options) {
        var _a, _b;
        const commandKey = this.options.key || `${options.index}`.padStart(3, '0'); // 001, 005, etc.
        if (options.platform !== cfn_init_internal_1.InitPlatform.WINDOWS && this.options.waitAfterCompletion !== undefined) {
            throw new Error(`Command '${this.command}': 'waitAfterCompletion' is only valid for Windows systems.`);
        }
        for (const handle of (_a = this.options.serviceRestartHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addCommand(commandKey);
        }
        return {
            config: {
                [commandKey]: {
                    command: this.command,
                    env: this.options.env,
                    cwd: this.options.cwd,
                    test: this.options.testCmd,
                    ignoreErrors: this.options.ignoreErrors,
                    waitAfterCompletion: (_b = this.options.waitAfterCompletion) === null || _b === void 0 ? void 0 : _b._render(),
                },
            },
        };
    }
}
exports.InitCommand = InitCommand;
/**
 * (experimental) Create files on the EC2 instance.
 *
 * @experimental
 */
class InitFile extends InitElement {
    /**
     * @experimental
     */
    constructor(fileName, options) {
        super();
        this.fileName = fileName;
        this.options = options;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.FILE.toString();
    }
    /**
     * (experimental) Use a literal string as the file content.
     *
     * @experimental
     */
    static fromString(fileName, content, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content,
                        encoding: this.options.base64Encoded ? 'base64' : 'plain',
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * (experimental) Write a symlink with the given symlink target.
     *
     * @experimental
     */
    static symlink(fileName, target, options = {}) {
        const { mode, ...otherOptions } = options;
        if (mode && mode.slice(0, 3) !== '120') {
            throw new Error('File mode for symlinks must begin with 120XXX');
        }
        return InitFile.fromString(fileName, target, { mode: (mode || '120644'), ...otherOptions });
    }
    /**
     * (experimental) Use a JSON-compatible object as the file content, write it to a JSON file.
     *
     * May contain tokens.
     *
     * @experimental
     */
    static fromObject(fileName, obj, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content: obj,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * (experimental) Read a file from disk and use its contents.
     *
     * The file will be embedded in the template, so care should be taken to not
     * exceed the template size.
     *
     * If options.base64encoded is set to true, this will base64-encode the file's contents.
     *
     * @experimental
     */
    static fromFileInline(targetFileName, sourceFileName, options = {}) {
        const encoding = options.base64Encoded ? 'base64' : 'utf8';
        const fileContents = fs.readFileSync(sourceFileName).toString(encoding);
        return InitFile.fromString(targetFileName, fileContents, options);
    }
    /**
     * (experimental) Download from a URL at instance startup time.
     *
     * @experimental
     */
    static fromUrl(fileName, url, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: url,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * (experimental) Download a file from an S3 bucket at instance startup time.
     *
     * @experimental
     */
    static fromS3Object(fileName, bucket, key, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: bucket.urlForObject(key),
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(fileName, options);
    }
    /**
     * (experimental) Create an asset from the given file.
     *
     * This is appropriate for files that are too large to embed into the template.
     *
     * @experimental
     */
    static fromAsset(targetFileName, path, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetFileName}Asset`, {
                    path,
                    ...options,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetFileName, options);
    }
    /**
     * (experimental) Use a file from an asset at instance startup time.
     *
     * @experimental
     */
    static fromExistingAsset(targetFileName, asset, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetFileName, options);
    }
    /** @internal */
    _bind(bindOptions) {
        var _a;
        for (const handle of (_a = this.options.serviceRestartHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addFile(this.fileName);
        }
        return this._doBind(bindOptions);
    }
    /**
     * Render the standard config block, given content vars
     * @internal
     */
    _standardConfig(fileOptions, platform, contentVars) {
        if (platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            if (fileOptions.group || fileOptions.owner || fileOptions.mode) {
                throw new Error('Owner, group, and mode options not supported for Windows.');
            }
            return {
                [this.fileName]: { ...contentVars },
            };
        }
        return {
            [this.fileName]: {
                ...contentVars,
                mode: fileOptions.mode || '000644',
                owner: fileOptions.owner || 'root',
                group: fileOptions.group || 'root',
            },
        };
    }
}
exports.InitFile = InitFile;
/**
 * (experimental) Create Linux/UNIX groups and assign group IDs.
 *
 * Not supported for Windows systems.
 *
 * @experimental
 */
class InitGroup extends InitElement {
    /**
     * @experimental
     */
    constructor(groupName, groupId) {
        super();
        this.groupName = groupName;
        this.groupId = groupId;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.GROUP.toString();
    }
    /**
     * (experimental) Create a group from its name, and optionally, group id.
     *
     * @experimental
     */
    static fromName(groupName, groupId) {
        return new InitGroup(groupName, groupId);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init groups are not supported on Windows');
        }
        return {
            config: {
                [this.groupName]: this.groupId !== undefined ? { gid: this.groupId } : {},
            },
        };
    }
}
exports.InitGroup = InitGroup;
/**
 * (experimental) Create Linux/UNIX users and to assign user IDs.
 *
 * Users are created as non-interactive system users with a shell of
 * /sbin/nologin. This is by design and cannot be modified.
 *
 * Not supported for Windows systems.
 *
 * @experimental
 */
class InitUser extends InitElement {
    /**
     * @experimental
     */
    constructor(userName, userOptions) {
        super();
        this.userName = userName;
        this.userOptions = userOptions;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.USER.toString();
    }
    /**
     * (experimental) Create a user from user name.
     *
     * @experimental
     */
    static fromName(userName, options = {}) {
        return new InitUser(userName, options);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init users are not supported on Windows');
        }
        return {
            config: {
                [this.userName]: {
                    uid: this.userOptions.userId,
                    groups: this.userOptions.groups,
                    homeDir: this.userOptions.homeDir,
                },
            },
        };
    }
}
exports.InitUser = InitUser;
/**
 * (experimental) A package to be installed during cfn-init time.
 *
 * @experimental
 */
class InitPackage extends InitElement {
    /**
     * @experimental
     */
    constructor(type, versions, packageName, serviceHandles) {
        super();
        this.type = type;
        this.versions = versions;
        this.packageName = packageName;
        this.serviceHandles = serviceHandles;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.PACKAGE.toString();
    }
    /**
     * (experimental) Install an RPM from an HTTP URL or a location on disk.
     *
     * @experimental
     */
    static rpm(location, options = {}) {
        return new InitPackage('rpm', [location], options.key, options.serviceRestartHandles);
    }
    /**
     * (experimental) Install a package using Yum.
     *
     * @experimental
     */
    static yum(packageName, options = {}) {
        var _a;
        return new InitPackage('yum', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * (experimental) Install a package from RubyGems.
     *
     * @experimental
     */
    static rubyGem(gemName, options = {}) {
        var _a;
        return new InitPackage('rubygems', (_a = options.version) !== null && _a !== void 0 ? _a : [], gemName, options.serviceRestartHandles);
    }
    /**
     * (experimental) Install a package from PyPI.
     *
     * @experimental
     */
    static python(packageName, options = {}) {
        var _a;
        return new InitPackage('python', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * (experimental) Install a package using APT.
     *
     * @experimental
     */
    static apt(packageName, options = {}) {
        var _a;
        return new InitPackage('apt', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * (experimental) Install an MSI package from an HTTP URL or a location on disk.
     *
     * @experimental
     */
    static msi(location, options = {}) {
        // The MSI package version must be a string, not an array.
        return new class extends InitPackage {
            renderPackageVersions() { return location; }
        }('msi', [location], options.key, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _a;
        if ((this.type === 'msi') !== (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS)) {
            if (this.type === 'msi') {
                throw new Error('MSI installers are only supported on Windows systems.');
            }
            else {
                throw new Error('Windows only supports the MSI package type');
            }
        }
        if (!this.packageName && !['rpm', 'msi'].includes(this.type)) {
            throw new Error('Package name must be specified for all package types besides RPM and MSI.');
        }
        const packageName = this.packageName || `${options.index}`.padStart(3, '0');
        for (const handle of (_a = this.serviceHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addPackage(this.type, packageName);
        }
        return {
            config: {
                [this.type]: {
                    [packageName]: this.renderPackageVersions(),
                },
            },
        };
    }
    /**
     * @experimental
     */
    renderPackageVersions() {
        return this.versions;
    }
}
exports.InitPackage = InitPackage;
/**
 * (experimental) A services that be enabled, disabled or restarted when the instance is launched.
 *
 * @experimental
 */
class InitService extends InitElement {
    constructor(serviceName, serviceOptions) {
        super();
        this.serviceName = serviceName;
        this.serviceOptions = serviceOptions;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.SERVICE.toString();
    }
    /**
     * (experimental) Enable and start the given service, optionally restarting it.
     *
     * @experimental
     */
    static enable(serviceName, options = {}) {
        var _a;
        const { enabled, ensureRunning, ...otherOptions } = options;
        return new InitService(serviceName, {
            enabled: enabled !== null && enabled !== void 0 ? enabled : true,
            ensureRunning: (_a = ensureRunning !== null && ensureRunning !== void 0 ? ensureRunning : enabled) !== null && _a !== void 0 ? _a : true,
            ...otherOptions,
        });
    }
    /**
     * (experimental) Disable and stop the given service.
     *
     * @experimental
     */
    static disable(serviceName) {
        return new InitService(serviceName, { enabled: false, ensureRunning: false });
    }
    /** @internal */
    _bind(options) {
        var _a;
        const serviceManager = options.platform === cfn_init_internal_1.InitPlatform.LINUX ? 'sysvinit' : 'windows';
        return {
            config: {
                [serviceManager]: {
                    [this.serviceName]: {
                        enabled: this.serviceOptions.enabled,
                        ensureRunning: this.serviceOptions.ensureRunning,
                        ...(_a = this.serviceOptions.serviceRestartHandle) === null || _a === void 0 ? void 0 : _a._renderRestartHandles(),
                    },
                },
            },
        };
    }
}
exports.InitService = InitService;
/**
 * (experimental) Extract an archive into a directory.
 *
 * @experimental
 */
class InitSource extends InitElement {
    /**
     * @experimental
     */
    constructor(targetDirectory, serviceHandles) {
        super();
        this.targetDirectory = targetDirectory;
        this.serviceHandles = serviceHandles;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.SOURCE.toString();
    }
    /**
     * (experimental) Retrieve a URL and extract it into the given directory.
     *
     * @experimental
     */
    static fromUrl(targetDirectory, url, options = {}) {
        return new class extends InitSource {
            _doBind() {
                return {
                    config: { [this.targetDirectory]: url },
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * (experimental) Extract a GitHub branch into a given directory.
     *
     * @experimental
     */
    static fromGitHub(targetDirectory, owner, repo, refSpec, options = {}) {
        return InitSource.fromUrl(targetDirectory, `https://github.com/${owner}/${repo}/tarball/${refSpec !== null && refSpec !== void 0 ? refSpec : 'master'}`, options);
    }
    /**
     * (experimental) Extract an archive stored in an S3 bucket into the given directory.
     *
     * @experimental
     */
    static fromS3Object(targetDirectory, bucket, key, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: { [this.targetDirectory]: bucket.urlForObject(key) },
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * (experimental) Create an InitSource from an asset created from the given path.
     *
     * @experimental
     */
    static fromAsset(targetDirectory, path, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetDirectory}Asset`, {
                    path,
                    ...bindOptions,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * (experimental) Extract a directory from an existing directory asset.
     *
     * @experimental
     */
    static fromExistingAsset(targetDirectory, asset, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _a;
        for (const handle of (_a = this.serviceHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addSource(this.targetDirectory);
        }
        // Delegate actual bind to subclasses
        return this._doBind(options);
    }
}
exports.InitSource = InitSource;
/**
 * Render a standard S3 auth block for use in AWS::CloudFormation::Authentication
 *
 * This block is the same every time (modulo bucket name), so it has the same
 * key every time so the blocks are merged into one in the final render.
 */
function standardS3Auth(role, bucketName) {
    return {
        S3AccessCreds: {
            type: 'S3',
            roleName: role.roleName,
            buckets: [bucketName],
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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