import * as ec2 from "../../aws-ec2";
import * as kms from "../../aws-kms";
import * as secretsmanager from "../../aws-secretsmanager";
import { Duration, RemovalPolicy, Resource } from "../../core";
import { Construct } from 'constructs';
import { DatabaseClusterAttributes, IDatabaseCluster } from './cluster-ref';
import { Endpoint } from './endpoint';
import { IClusterParameterGroup } from './parameter-group';
import { BackupProps, InstanceProps, Login, RotationMultiUserOptions } from './props';
/**
 * (experimental) Properties for a new database cluster.
 *
 * @experimental
 */
export interface DatabaseClusterProps {
    /**
     * (experimental) What version of the database to start.
     *
     * @default - The default engine version.
     * @experimental
     */
    readonly engineVersion?: string;
    /**
     * (experimental) The port the DocumentDB cluster will listen on.
     *
     * @default DatabaseCluster.DEFAULT_PORT
     * @experimental
     */
    readonly port?: number;
    /**
     * (experimental) Username and password for the administrative user.
     *
     * @experimental
     */
    readonly masterUser: Login;
    /**
     * (experimental) Backup settings.
     *
     * @default - Backup retention period for automated backups is 1 day.
     * Backup preferred window is set to a 30-minute window selected at random from an
     * 8-hour block of time for each AWS Region, occurring on a random day of the week.
     * @see https://docs.aws.amazon.com/documentdb/latest/developerguide/backup-restore.db-cluster-snapshots.html#backup-restore.backup-window
     * @experimental
     */
    readonly backup?: BackupProps;
    /**
     * (experimental) The KMS key for storage encryption.
     *
     * @default - default master key.
     * @experimental
     */
    readonly kmsKey?: kms.IKey;
    /**
     * (experimental) Whether to enable storage encryption.
     *
     * @default true
     * @experimental
     */
    readonly storageEncrypted?: boolean;
    /**
     * (experimental) Number of DocDB compute instances.
     *
     * @default 1
     * @experimental
     */
    readonly instances?: number;
    /**
     * (experimental) An optional identifier for the cluster.
     *
     * @default - A name is automatically generated.
     * @experimental
     */
    readonly dbClusterName?: string;
    /**
     * (experimental) Base identifier for instances.
     *
     * Every replica is named by appending the replica number to this string, 1-based.
     *
     * @default - `dbClusterName` is used with the word "Instance" appended. If `dbClusterName` is not provided, the
     * identifier is automatically generated.
     * @experimental
     */
    readonly instanceIdentifierBase?: string;
    /**
     * (experimental) Settings for the individual instances that are launched.
     *
     * @experimental
     */
    readonly instanceProps: InstanceProps;
    /**
     * (experimental) A weekly time range in which maintenance should preferably execute.
     *
     * Must be at least 30 minutes long.
     *
     * Example: 'tue:04:17-tue:04:47'
     *
     * @default - 30-minute window selected at random from an 8-hour block of time for
     * each AWS Region, occurring on a random day of the week.
     * @see https://docs.aws.amazon.com/documentdb/latest/developerguide/db-instance-maintain.html#maintenance-window
     * @experimental
     */
    readonly preferredMaintenanceWindow?: string;
    /**
     * (experimental) Additional parameters to pass to the database engine.
     *
     * @default - No parameter group.
     * @experimental
     */
    readonly parameterGroup?: IClusterParameterGroup;
    /**
     * (experimental) The removal policy to apply when the cluster and its instances are removed or replaced during a stack update, or when the stack is deleted.
     *
     * This
     * removal policy also applies to the implicit security group created for the
     * cluster if one is not supplied as a parameter.
     *
     * @default - Retain cluster.
     * @experimental
     */
    readonly removalPolicy?: RemovalPolicy;
}
/**
 * A new or imported clustered database.
 */
declare abstract class DatabaseClusterBase extends Resource implements IDatabaseCluster {
    /**
     * Identifier of the cluster
     */
    abstract readonly clusterIdentifier: string;
    /**
     * Identifiers of the replicas
     */
    abstract readonly instanceIdentifiers: string[];
    /**
     * The endpoint to use for read/write operations
     */
    abstract readonly clusterEndpoint: Endpoint;
    /**
     * Endpoint to use for load-balanced read-only operations.
     */
    abstract readonly clusterReadEndpoint: Endpoint;
    /**
     * Endpoints which address each individual replica.
     */
    abstract readonly instanceEndpoints: Endpoint[];
    /**
     * Access to the network connections
     */
    abstract readonly connections: ec2.Connections;
    /**
     * Security group identifier of this database
     */
    abstract readonly securityGroupId: string;
    /**
     * (experimental) Renders the secret attachment target specifications.
     *
     * @experimental
     */
    asSecretAttachmentTarget(): secretsmanager.SecretAttachmentTargetProps;
}
/**
 * (experimental) Create a clustered database with a given number of instances.
 *
 * @experimental
 * @resource AWS::DocDB::DBCluster
 */
export declare class DatabaseCluster extends DatabaseClusterBase {
    /**
     * (experimental) The default number of instances in the DocDB cluster if none are specified.
     *
     * @experimental
     */
    static readonly DEFAULT_NUM_INSTANCES = 1;
    /**
     * (experimental) The default port Document DB listens on.
     *
     * @experimental
     */
    static readonly DEFAULT_PORT = 27017;
    /**
     * (experimental) Import an existing DatabaseCluster from properties.
     *
     * @experimental
     */
    static fromDatabaseClusterAttributes(scope: Construct, id: string, attrs: DatabaseClusterAttributes): IDatabaseCluster;
    /**
     * The single user secret rotation application.
     */
    private static readonly SINGLE_USER_ROTATION_APPLICATION;
    /**
     * The multi user secret rotation application.
     */
    private static readonly MULTI_USER_ROTATION_APPLICATION;
    /**
     * (experimental) Identifier of the cluster.
     *
     * @experimental
     */
    readonly clusterIdentifier: string;
    /**
     * (experimental) The endpoint to use for read/write operations.
     *
     * @experimental
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * (experimental) Endpoint to use for load-balanced read-only operations.
     *
     * @experimental
     */
    readonly clusterReadEndpoint: Endpoint;
    /**
     * (experimental) The resource id for the cluster;
     *
     * for example: cluster-ABCD1234EFGH5678IJKL90MNOP. The cluster ID uniquely
     * identifies the cluster and is used in things like IAM authentication policies.
     *
     * @experimental
     * @attribute ClusterResourceId
     */
    readonly clusterResourceIdentifier: string;
    /**
     * (experimental) The connections object to implement IConectable.
     *
     * @experimental
     */
    readonly connections: ec2.Connections;
    /**
     * (experimental) Identifiers of the replicas.
     *
     * @experimental
     */
    readonly instanceIdentifiers: string[];
    /**
     * (experimental) Endpoints which address each individual replica.
     *
     * @experimental
     */
    readonly instanceEndpoints: Endpoint[];
    /**
     * (experimental) Security group identifier of this database.
     *
     * @experimental
     */
    readonly securityGroupId: string;
    /**
     * (experimental) The secret attached to this cluster.
     *
     * @experimental
     */
    readonly secret?: secretsmanager.ISecret;
    /**
     * The VPC where the DB subnet group is created.
     */
    private readonly vpc;
    /**
     * The subnets used by the DB subnet group.
     */
    private readonly vpcSubnets?;
    /**
     * @experimental
     */
    constructor(scope: Construct, id: string, props: DatabaseClusterProps);
    /**
     * (experimental) Adds the single user rotation of the master password to this cluster.
     *
     * @param automaticallyAfter Specifies the number of days after the previous rotation before Secrets Manager triggers the next automatic rotation.
     * @experimental
     */
    addRotationSingleUser(automaticallyAfter?: Duration): secretsmanager.SecretRotation;
    /**
     * (experimental) Adds the multi user rotation to this cluster.
     *
     * @experimental
     */
    addRotationMultiUser(id: string, options: RotationMultiUserOptions): secretsmanager.SecretRotation;
}
export {};
