"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceType = exports.ManagedRuleIdentifiers = exports.CustomRule = exports.ManagedRule = exports.MaximumExecutionFrequency = exports.RuleScope = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const config_generated_1 = require("./config.generated");
/**
 * A new or imported rule.
 */
class RuleBase extends core_1.Resource {
    /**
     * (experimental) Defines an EventBridge event rule which triggers for rule events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @experimental
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.config'],
            detail: {
                configRuleName: [this.configRuleName],
            },
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * (experimental) Defines an EventBridge event rule which triggers for rule compliance events.
     *
     * @experimental
     */
    onComplianceChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Compliance Change'],
        });
        return rule;
    }
    /**
     * (experimental) Defines an EventBridge event rule which triggers for rule re-evaluation status events.
     *
     * @experimental
     */
    onReEvaluationStatus(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Re-evaluation Status'],
        });
        return rule;
    }
}
/**
 * A new managed or custom rule.
 */
class RuleNew extends RuleBase {
    /**
     * (experimental) Imports an existing rule.
     *
     * @param configRuleName the name of the rule.
     * @experimental
     */
    static fromConfigRuleName(scope, id, configRuleName) {
        class Import extends RuleBase {
            constructor() {
                super(...arguments);
                this.configRuleName = configRuleName;
            }
        }
        return new Import(scope, id);
    }
}
/**
 * (experimental) Determines which resources trigger an evaluation of an AWS Config rule.
 *
 * @experimental
 */
class RuleScope {
    constructor(resourceId, resourceTypes, tagKey, tagValue) {
        this.resourceTypes = resourceTypes;
        this.resourceId = resourceId;
        this.key = tagKey;
        this.value = tagValue;
    }
    /**
     * (experimental) restricts scope of changes to a specific resource type or resource identifier.
     *
     * @experimental
     */
    static fromResource(resourceType, resourceId) {
        return new RuleScope(resourceId, [resourceType]);
    }
    /**
     * (experimental) restricts scope of changes to specific resource types.
     *
     * @experimental
     */
    static fromResources(resourceTypes) {
        return new RuleScope(undefined, resourceTypes);
    }
    /**
     * (experimental) restricts scope of changes to a specific tag.
     *
     * @experimental
     */
    static fromTag(key, value) {
        return new RuleScope(undefined, undefined, key, value);
    }
}
exports.RuleScope = RuleScope;
/**
 * (experimental) The maximum frequency at which the AWS Config rule runs evaluations.
 *
 * @experimental
 */
var MaximumExecutionFrequency;
(function (MaximumExecutionFrequency) {
    MaximumExecutionFrequency["ONE_HOUR"] = "One_Hour";
    MaximumExecutionFrequency["THREE_HOURS"] = "Three_Hours";
    MaximumExecutionFrequency["SIX_HOURS"] = "Six_Hours";
    MaximumExecutionFrequency["TWELVE_HOURS"] = "Twelve_Hours";
    MaximumExecutionFrequency["TWENTY_FOUR_HOURS"] = "TwentyFour_Hours";
})(MaximumExecutionFrequency = exports.MaximumExecutionFrequency || (exports.MaximumExecutionFrequency = {}));
/**
 * (experimental) A new managed rule.
 *
 * @experimental
 * @resource AWS::Config::ConfigRule
 */
class ManagedRule extends RuleNew {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        this.ruleScope = props.ruleScope;
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => renderScope(this.ruleScope) }),
            source: {
                owner: 'AWS',
                sourceIdentifier: props.identifier,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isManaged = true;
    }
}
exports.ManagedRule = ManagedRule;
/**
 * (experimental) A new custom rule.
 *
 * @experimental
 * @resource AWS::Config::ConfigRule
 */
class CustomRule extends RuleNew {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        if (!props.configurationChanges && !props.periodic) {
            throw new Error('At least one of `configurationChanges` or `periodic` must be set to true.');
        }
        const sourceDetails = [];
        this.ruleScope = props.ruleScope;
        if (props.configurationChanges) {
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'ConfigurationItemChangeNotification',
            });
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'OversizedConfigurationItemChangeNotification',
            });
        }
        if (props.periodic) {
            sourceDetails.push({
                eventSource: 'aws.config',
                maximumExecutionFrequency: props.maximumExecutionFrequency,
                messageType: 'ScheduledNotification',
            });
        }
        props.lambdaFunction.addPermission('Permission', {
            principal: new iam.ServicePrincipal('config.amazonaws.com'),
        });
        if (props.lambdaFunction.role) {
            props.lambdaFunction.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSConfigRulesExecutionRole'));
        }
        // The lambda permission must be created before the rule
        this.node.addDependency(props.lambdaFunction);
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => renderScope(this.ruleScope) }),
            source: {
                owner: 'CUSTOM_LAMBDA',
                sourceDetails,
                sourceIdentifier: props.lambdaFunction.functionArn,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        if (props.configurationChanges) {
            this.isCustomWithChanges = true;
        }
    }
}
exports.CustomRule = CustomRule;
/**
 * (experimental) Managed rules that are supported by AWS Config.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/managed-rules-by-aws-config.html
 * @experimental
 */
class ManagedRuleIdentifiers {
    // utility class
    constructor() { }
}
exports.ManagedRuleIdentifiers = ManagedRuleIdentifiers;
/**
 * (experimental) Checks that the inline policies attached to your AWS Identity and Access Management users, roles, and groups do not allow blocked actions on all AWS Key Management Service keys.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-inline-policy-blocked-kms-actions.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_INLINE_POLICY_BLOCKED_KMS_ACTIONS = 'IAM_INLINE_POLICY_BLOCKED_KMS_ACTIONS';
/**
 * (experimental) Checks that the managed AWS Identity and Access Management policies that you create do not allow blocked actions on all AWS AWS KMS keys.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-customer-policy-blocked-kms-actions.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_CUSTOMER_POLICY_BLOCKED_KMS_ACTIONS = 'IAM_CUSTOMER_POLICY_BLOCKED_KMS_ACTIONS';
/**
 * (experimental) Checks whether the active access keys are rotated within the number of days specified in maxAccessKeyAge.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 * @experimental
 */
ManagedRuleIdentifiers.ACCESS_KEYS_ROTATED = 'ACCESS_KEYS_ROTATED';
/**
 * (experimental) Checks whether AWS account is part of AWS Organizations.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/account-part-of-organizations.html
 * @experimental
 */
ManagedRuleIdentifiers.ACCOUNT_PART_OF_ORGANIZATIONS = 'ACCOUNT_PART_OF_ORGANIZATIONS';
/**
 * (experimental) Checks whether ACM Certificates in your account are marked for expiration within the specified number of days.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/acm-certificate-expiration-check.html
 * @experimental
 */
ManagedRuleIdentifiers.ACM_CERTIFICATE_EXPIRATION_CHECK = 'ACM_CERTIFICATE_EXPIRATION_CHECK';
/**
 * (experimental) Checks if rule evaluates Application Load Balancers (ALBs) to ensure they are configured to drop http headers.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/alb-http-drop-invalid-header-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.ALB_HTTP_DROP_INVALID_HEADER_ENABLED = 'ALB_HTTP_DROP_INVALID_HEADER_ENABLED';
/**
 * (experimental) Checks whether HTTP to HTTPS redirection is configured on all HTTP listeners of Application Load Balancer.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/alb-http-to-https-redirection-check.html
 * @experimental
 */
ManagedRuleIdentifiers.ALB_HTTP_TO_HTTPS_REDIRECTION_CHECK = 'ALB_HTTP_TO_HTTPS_REDIRECTION_CHECK';
/**
 * (experimental) Checks if Web Application Firewall (WAF) is enabled on Application Load Balancers (ALBs).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/alb-waf-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.ALB_WAF_ENABLED = 'ALB_WAF_ENABLED';
/**
 * (experimental) Checks that all methods in Amazon API Gateway stages have caching enabled and encrypted.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gw-cache-enabled-and-encrypted.html
 * @experimental
 */
ManagedRuleIdentifiers.API_GW_CACHE_ENABLED_AND_ENCRYPTED = 'API_GW_CACHE_ENABLED_AND_ENCRYPTED';
/**
 * (experimental) Checks that Amazon API Gateway APIs are of the type specified in the rule parameter endpointConfigurationType.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gw-endpoint-type-check.html
 * @experimental
 */
ManagedRuleIdentifiers.API_GW_ENDPOINT_TYPE_CHECK = 'API_GW_ENDPOINT_TYPE_CHECK';
/**
 * (experimental) Checks that all methods in Amazon API Gateway stage has logging enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gw-execution-logging-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.API_GW_EXECUTION_LOGGING_ENABLED = 'API_GW_EXECUTION_LOGGING_ENABLED';
/**
 * (experimental) Checks whether running instances are using specified AMIs.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/approved-amis-by-id.html
 * @experimental
 */
ManagedRuleIdentifiers.APPROVED_AMIS_BY_ID = 'APPROVED_AMIS_BY_ID';
/**
 * (experimental) Checks whether running instances are using specified AMIs.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/approved-amis-by-tag.html
 * @experimental
 */
ManagedRuleIdentifiers.APPROVED_AMIS_BY_TAG = 'APPROVED_AMIS_BY_TAG';
/**
 * (experimental) Checks whether your Auto Scaling groups that are associated with a load balancer are using Elastic Load Balancing health checks.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-group-elb-healthcheck-required.html
 * @experimental
 */
ManagedRuleIdentifiers.AUTOSCALING_GROUP_ELB_HEALTHCHECK_REQUIRED = 'AUTOSCALING_GROUP_ELB_HEALTHCHECK_REQUIRED';
/**
 * (experimental) Checks whether an AWS CloudFormation stack's actual configuration differs, or has drifted, from it's expected configuration.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-drift-detection-check.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK = 'CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK';
/**
 * (experimental) Checks whether your CloudFormation stacks are sending event notifications to an SNS topic.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-notification-check.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDFORMATION_STACK_NOTIFICATION_CHECK = 'CLOUDFORMATION_STACK_NOTIFICATION_CHECK';
/**
 * (experimental) Checks if an Amazon CloudFront distribution is configured to return a specific object that is the default root object.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-default-root-object-configured.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDFRONT_DEFAULT_ROOT_OBJECT_CONFIGURED = 'CLOUDFRONT_DEFAULT_ROOT_OBJECT_CONFIGURED';
/**
 * (experimental) Checks that Amazon CloudFront distribution with Amazon S3 Origin type has Origin Access Identity (OAI) configured.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-origin-access-identity-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDFRONT_ORIGIN_ACCESS_IDENTITY_ENABLED = 'CLOUDFRONT_ORIGIN_ACCESS_IDENTITY_ENABLED';
/**
 * (experimental) Checks whether an origin group is configured for the distribution of at least 2 origins in the origin group for Amazon CloudFront.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-origin-failover-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDFRONT_ORIGIN_FAILOVER_ENABLED = 'CLOUDFRONT_ORIGIN_FAILOVER_ENABLED';
/**
 * (experimental) Checks if Amazon CloudFront distributions are using a custom SSL certificate and are configured to use SNI to serve HTTPS requests.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-sni-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDFRONT_SNI_ENABLED = 'CLOUDFRONT_SNI_ENABLED';
/**
 * (experimental) Checks whether your Amazon CloudFront distributions use HTTPS (directly or via a redirection).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-viewer-policy-https.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDFRONT_VIEWER_POLICY_HTTPS = 'CLOUDFRONT_VIEWER_POLICY_HTTPS';
/**
 * (experimental) Checks whether AWS CloudTrail trails are configured to send logs to Amazon CloudWatch Logs.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-cloud-watch-logs-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUD_TRAIL_CLOUD_WATCH_LOGS_ENABLED = 'CLOUD_TRAIL_CLOUD_WATCH_LOGS_ENABLED';
/**
 * (experimental) Checks whether AWS CloudTrail is enabled in your AWS account.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUD_TRAIL_ENABLED = 'CLOUD_TRAIL_ENABLED';
/**
 * (experimental) Checks whether AWS CloudTrail is configured to use the server side encryption (SSE) AWS Key Management Service (AWS KMS) customer master key (CMK) encryption.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-encryption-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUD_TRAIL_ENCRYPTION_ENABLED = 'CLOUD_TRAIL_ENCRYPTION_ENABLED';
/**
 * (experimental) Checks whether AWS CloudTrail creates a signed digest file with logs.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-log-file-validation-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUD_TRAIL_LOG_FILE_VALIDATION_ENABLED = 'CLOUD_TRAIL_LOG_FILE_VALIDATION_ENABLED';
/**
 * (experimental) Checks whether at least one AWS CloudTrail trail is logging Amazon S3 data events for all S3 buckets.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-s3-dataevents-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDTRAIL_S3_DATAEVENTS_ENABLED = 'CLOUDTRAIL_S3_DATAEVENTS_ENABLED';
/**
 * (experimental) Checks that there is at least one AWS CloudTrail trail defined with security best practices.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-security-trail-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDTRAIL_SECURITY_TRAIL_ENABLED = 'CLOUDTRAIL_SECURITY_TRAIL_ENABLED';
/**
 * (experimental) Checks whether CloudWatch alarms have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-action-check.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDWATCH_ALARM_ACTION_CHECK = 'CLOUDWATCH_ALARM_ACTION_CHECK';
/**
 * (experimental) Checks whether the specified resource type has a CloudWatch alarm for the specified metric.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-resource-check.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDWATCH_ALARM_RESOURCE_CHECK = 'CLOUDWATCH_ALARM_RESOURCE_CHECK';
/**
 * (experimental) Checks whether CloudWatch alarms with the given metric name have the specified settings.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-settings-check.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDWATCH_ALARM_SETTINGS_CHECK = 'CLOUDWATCH_ALARM_SETTINGS_CHECK';
/**
 * (experimental) Checks whether a log group in Amazon CloudWatch Logs is encrypted with a AWS Key Management Service (KMS) managed Customer Master Keys (CMK).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-log-group-encrypted.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDWATCH_LOG_GROUP_ENCRYPTED = 'CLOUDWATCH_LOG_GROUP_ENCRYPTED';
/**
 * (experimental) Checks that key rotation is enabled for each key and matches to the key ID of the customer created customer master key (CMK).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cmk-backing-key-rotation-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.CMK_BACKING_KEY_ROTATION_ENABLED = 'CMK_BACKING_KEY_ROTATION_ENABLED';
/**
 * (experimental) Checks whether the project contains environment variables AWS_ACCESS_KEY_ID and AWS_SECRET_ACCESS_KEY.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-envvar-awscred-check.html
 * @experimental
 */
ManagedRuleIdentifiers.CODEBUILD_PROJECT_ENVVAR_AWSCRED_CHECK = 'CODEBUILD_PROJECT_ENVVAR_AWSCRED_CHECK';
/**
 * (experimental) Checks whether the GitHub or Bitbucket source repository URL contains either personal access tokens or user name and password.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-source-repo-url-check.html
 * @experimental
 */
ManagedRuleIdentifiers.CODEBUILD_PROJECT_SOURCE_REPO_URL_CHECK = 'CODEBUILD_PROJECT_SOURCE_REPO_URL_CHECK';
/**
 * (experimental) Checks whether the first deployment stage of the AWS CodePipeline performs more than one deployment.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codepipeline-deployment-count-check.html
 * @experimental
 */
ManagedRuleIdentifiers.CODEPIPELINE_DEPLOYMENT_COUNT_CHECK = 'CODEPIPELINE_DEPLOYMENT_COUNT_CHECK';
/**
 * (experimental) Checks whether each stage in the AWS CodePipeline deploys to more than N times the number of the regions the AWS CodePipeline has deployed in all the previous combined stages, where N is the region fanout number.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codepipeline-region-fanout-check.html
 * @experimental
 */
ManagedRuleIdentifiers.CODEPIPELINE_REGION_FANOUT_CHECK = 'CODEPIPELINE_REGION_FANOUT_CHECK';
/**
 * (experimental) Checks whether Amazon CloudWatch LogGroup retention period is set to specific number of days.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cw-loggroup-retention-period-check.html
 * @experimental
 */
ManagedRuleIdentifiers.CW_LOGGROUP_RETENTION_PERIOD_CHECK = 'CW_LOGGROUP_RETENTION_PERIOD_CHECK';
/**
 * (experimental) Checks that DynamoDB Accelerator (DAX) clusters are encrypted.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dax-encryption-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.DAX_ENCRYPTION_ENABLED = 'DAX_ENCRYPTION_ENABLED';
/**
 * (experimental) Checks whether RDS DB instances have backups enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/db-instance-backup-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.RDS_DB_INSTANCE_BACKUP_ENABLED = 'DB_INSTANCE_BACKUP_ENABLED';
/**
 * (experimental) Checks instances for specified tenancy.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/desired-instance-tenancy.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_DESIRED_INSTANCE_TENANCY = 'DESIRED_INSTANCE_TENANCY';
/**
 * (experimental) Checks whether your EC2 instances are of the specified instance types.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/desired-instance-type.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_DESIRED_INSTANCE_TYPE = 'DESIRED_INSTANCE_TYPE';
/**
 * (experimental) Checks whether AWS Database Migration Service replication instances are public.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dms-replication-not-public.html
 * @experimental
 */
ManagedRuleIdentifiers.DMS_REPLICATION_NOT_PUBLIC = 'DMS_REPLICATION_NOT_PUBLIC';
/**
 * (experimental) Checks whether Auto Scaling or On-Demand is enabled on your DynamoDB tables and/or global secondary indexes.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-autoscaling-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.DYNAMODB_AUTOSCALING_ENABLED = 'DYNAMODB_AUTOSCALING_ENABLED';
/**
 * (experimental) Checks whether Amazon DynamoDB table is present in AWS Backup plans.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-in-backup-plan.html
 * @experimental
 */
ManagedRuleIdentifiers.DYNAMODB_IN_BACKUP_PLAN = 'DYNAMODB_IN_BACKUP_PLAN';
/**
 * (experimental) Checks that point in time recovery (PITR) is enabled for Amazon DynamoDB tables.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-pitr-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.DYNAMODB_PITR_ENABLED = 'DYNAMODB_PITR_ENABLED';
/**
 * (experimental) Checks whether Amazon DynamoDB table is encrypted with AWS Key Management Service (KMS).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-table-encrypted-kms.html
 * @experimental
 */
ManagedRuleIdentifiers.DYNAMODB_TABLE_ENCRYPTED_KMS = 'DYNAMODB_TABLE_ENCRYPTED_KMS';
/**
 * (experimental) Checks whether the Amazon DynamoDB tables are encrypted and checks their status.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-table-encryption-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.DYNAMODB_TABLE_ENCRYPTION_ENABLED = 'DYNAMODB_TABLE_ENCRYPTION_ENABLED';
/**
 * (experimental) Checks whether provisioned DynamoDB throughput is approaching the maximum limit for your account.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-throughput-limit-check.html
 * @experimental
 */
ManagedRuleIdentifiers.DYNAMODB_THROUGHPUT_LIMIT_CHECK = 'DYNAMODB_THROUGHPUT_LIMIT_CHECK';
/**
 * (experimental) Checks if Amazon Elastic Block Store (Amazon EBS) volumes are added in backup plans of AWS Backup.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ebs-in-backup-plan.html
 * @experimental
 */
ManagedRuleIdentifiers.EBS_IN_BACKUP_PLAN = 'EBS_IN_BACKUP_PLAN';
/**
 * (experimental) Checks whether Amazon Elastic File System (Amazon EFS) file systems are added in the backup plans of AWS Backup.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/efs-in-backup-plan.html
 * @experimental
 */
ManagedRuleIdentifiers.EFS_IN_BACKUP_PLAN = 'EFS_IN_BACKUP_PLAN';
/**
 * (experimental) Check that Amazon Elastic Block Store (EBS) encryption is enabled by default.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-ebs-encryption-by-default.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_EBS_ENCRYPTION_BY_DEFAULT = 'EC2_EBS_ENCRYPTION_BY_DEFAULT';
/**
 * (experimental) Checks whether EBS optimization is enabled for your EC2 instances that can be EBS-optimized.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ebs-optimized-instance.html
 * @experimental
 */
ManagedRuleIdentifiers.EBS_OPTIMIZED_INSTANCE = 'EBS_OPTIMIZED_INSTANCE';
/**
 * (experimental) Checks whether Amazon Elastic Block Store snapshots are not publicly restorable.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ebs-snapshot-public-restorable-check.html
 * @experimental
 */
ManagedRuleIdentifiers.EBS_SNAPSHOT_PUBLIC_RESTORABLE_CHECK = 'EBS_SNAPSHOT_PUBLIC_RESTORABLE_CHECK';
/**
 * (experimental) Checks whether detailed monitoring is enabled for EC2 instances.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-detailed-monitoring-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_INSTANCE_DETAILED_MONITORING_ENABLED = 'EC2_INSTANCE_DETAILED_MONITORING_ENABLED';
/**
 * (experimental) Checks whether the Amazon EC2 instances in your account are managed by AWS Systems Manager.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-managed-by-systems-manager.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_INSTANCE_MANAGED_BY_SSM = 'EC2_INSTANCE_MANAGED_BY_SSM';
/**
 * (experimental) Checks whether Amazon Elastic Compute Cloud (Amazon EC2) instances have a public IP association.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-no-public-ip.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_INSTANCE_NO_PUBLIC_IP = 'EC2_INSTANCE_NO_PUBLIC_IP';
/**
 * (experimental) Checks whether your EC2 instances belong to a virtual private cloud (VPC).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instances-in-vpc.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_INSTANCES_IN_VPC = 'INSTANCES_IN_VPC';
/**
 * (experimental) Checks that none of the specified applications are installed on the instance.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-applications-blacklisted.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_APPLICATIONS_BLOCKED = 'EC2_MANAGEDINSTANCE_APPLICATIONS_BLACKLISTED';
/**
 * (experimental) Checks whether all of the specified applications are installed on the instance.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-applications-required.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_APPLICATIONS_REQUIRED = 'EC2_MANAGEDINSTANCE_APPLICATIONS_REQUIRED';
/**
 * (experimental) Checks whether the compliance status of AWS Systems Manager association compliance is COMPLIANT or NON_COMPLIANT after the association execution on the instance.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-association-compliance-status-check.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_ASSOCIATION_COMPLIANCE_STATUS_CHECK = 'EC2_MANAGEDINSTANCE_ASSOCIATION_COMPLIANCE_STATUS_CHECK';
/**
 * (experimental) Checks whether instances managed by AWS Systems Manager are configured to collect blocked inventory types.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-inventory-blacklisted.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_INVENTORY_BLOCKED = 'EC2_MANAGEDINSTANCE_INVENTORY_BLACKLISTED';
/**
 * (experimental) Checks whether the compliance status of the Amazon EC2 Systems Manager patch compliance is COMPLIANT or NON_COMPLIANT after the patch installation on the instance.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-patch-compliance-status-check.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_PATCH_COMPLIANCE_STATUS_CHECK = 'EC2_MANAGEDINSTANCE_PATCH_COMPLIANCE_STATUS_CHECK';
/**
 * (experimental) Checks whether EC2 managed instances have the desired configurations.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-platform-check.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_PLATFORM_CHECK = 'EC2_MANAGEDINSTANCE_PLATFORM_CHECK';
/**
 * (experimental) Checks that security groups are attached to Amazon Elastic Compute Cloud (Amazon EC2) instances or to an elastic network interface.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-security-group-attached-to-eni.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_SECURITY_GROUP_ATTACHED_TO_ENI = 'EC2_SECURITY_GROUP_ATTACHED_TO_ENI';
/**
 * (experimental) Checks whether there are instances stopped for more than the allowed number of days.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-stopped-instance.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_STOPPED_INSTANCE = 'EC2_STOPPED_INSTANCE';
/**
 * (experimental) Checks whether EBS volumes are attached to EC2 instances.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-volume-inuse-check.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_VOLUME_INUSE_CHECK = 'EC2_VOLUME_INUSE_CHECK';
/**
 * (experimental) hecks whether Amazon Elastic File System (Amazon EFS) is configured to encrypt the file data using AWS Key Management Service (AWS KMS).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/efs-encrypted-check.html
 * @experimental
 */
ManagedRuleIdentifiers.EFS_ENCRYPTED_CHECK = 'EFS_ENCRYPTED_CHECK';
/**
 * (experimental) Checks whether all Elastic IP addresses that are allocated to a VPC are attached to EC2 instances or in-use elastic network interfaces (ENIs).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/eip-attached.html
 * @experimental
 */
ManagedRuleIdentifiers.EIP_ATTACHED = 'EIP_ATTACHED';
/**
 * (experimental) Checks whether Amazon Elasticsearch Service (Amazon ES) domains have encryption at rest configuration enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-encrypted-at-rest.html
 * @experimental
 */
ManagedRuleIdentifiers.ELASTICSEARCH_ENCRYPTED_AT_REST = 'ELASTICSEARCH_ENCRYPTED_AT_REST';
/**
 * (experimental) Checks whether Amazon Elasticsearch Service (Amazon ES) domains are in Amazon Virtual Private Cloud (Amazon VPC).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-in-vpc-only.html
 * @experimental
 */
ManagedRuleIdentifiers.ELASTICSEARCH_IN_VPC_ONLY = 'ELASTICSEARCH_IN_VPC_ONLY';
/**
 * (experimental) Check if the Amazon ElastiCache Redis clusters have automatic backup turned on.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elasticache-redis-cluster-automatic-backup-check.html
 * @experimental
 */
ManagedRuleIdentifiers.ELASTICACHE_REDIS_CLUSTER_AUTOMATIC_BACKUP_CHECK = 'ELASTICACHE_REDIS_CLUSTER_AUTOMATIC_BACKUP_CHECK';
/**
 * (experimental) Checks whether your Amazon Elastic Compute Cloud (Amazon EC2) instance metadata version is configured with Instance Metadata Service Version 2 (IMDSv2).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-imdsv2-check.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_IMDSV2_CHECK = 'EC2_IMDSV2_CHECK';
/**
 * (experimental) Checks whether Amazon Elastic Kubernetes Service (Amazon EKS) endpoint is not publicly accessible.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/eks-endpoint-no-public-access.html
 * @experimental
 */
ManagedRuleIdentifiers.EKS_ENDPOINT_NO_PUBLIC_ACCESS = 'EKS_ENDPOINT_NO_PUBLIC_ACCESS';
/**
 * (experimental) Checks whether Amazon Elastic Kubernetes Service clusters are configured to have Kubernetes secrets encrypted using AWS Key Management Service (KMS) keys.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/eks-secrets-encrypted.html
 * @experimental
 */
ManagedRuleIdentifiers.EKS_SECRETS_ENCRYPTED = 'EKS_SECRETS_ENCRYPTED';
/**
 * (experimental) Check that Amazon ElasticSearch Service nodes are encrypted end to end.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-node-to-node-encryption-check.html
 * @experimental
 */
ManagedRuleIdentifiers.ELASTICSEARCH_NODE_TO_NODE_ENCRYPTION_CHECK = 'ELASTICSEARCH_NODE_TO_NODE_ENCRYPTION_CHECK';
/**
 * (experimental) Checks if cross-zone load balancing is enabled for the Classic Load Balancers (CLBs).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-cross-zone-load-balancing-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.ELB_CROSS_ZONE_LOAD_BALANCING_ENABLED = 'ELB_CROSS_ZONE_LOAD_BALANCING_ENABLED';
/**
 * (experimental) Checks whether your Classic Load Balancer is configured with SSL or HTTPS listeners.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-tls-https-listeners-only.html
 * @experimental
 */
ManagedRuleIdentifiers.ELB_TLS_HTTPS_LISTENERS_ONLY = 'ELB_TLS_HTTPS_LISTENERS_ONLY';
/**
 * (experimental) Checks whether the Classic Load Balancers use SSL certificates provided by AWS Certificate Manager.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-acm-certificate-required.html
 * @experimental
 */
ManagedRuleIdentifiers.ELB_ACM_CERTIFICATE_REQUIRED = 'ELB_ACM_CERTIFICATE_REQUIRED';
/**
 * (experimental) Checks whether your Classic Load Balancer SSL listeners are using a custom policy.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-custom-security-policy-ssl-check.html
 * @experimental
 */
ManagedRuleIdentifiers.ELB_CUSTOM_SECURITY_POLICY_SSL_CHECK = 'ELB_CUSTOM_SECURITY_POLICY_SSL_CHECK';
/**
 * (experimental) Checks whether Elastic Load Balancing has deletion protection enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-deletion-protection-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.ELB_DELETION_PROTECTION_ENABLED = 'ELB_DELETION_PROTECTION_ENABLED';
/**
 * (experimental) Checks whether the Application Load Balancer and the Classic Load Balancer have logging enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-logging-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.ELB_LOGGING_ENABLED = 'ELB_LOGGING_ENABLED';
/**
 * (experimental) Checks whether your Classic Load Balancer SSL listeners are using a predefined policy.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-predefined-security-policy-ssl-check.html
 * @experimental
 */
ManagedRuleIdentifiers.ELB_PREDEFINED_SECURITY_POLICY_SSL_CHECK = 'ELB_PREDEFINED_SECURITY_POLICY_SSL_CHECK';
/**
 * (experimental) Checks that Amazon EMR clusters have Kerberos enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/emr-kerberos-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.EMR_KERBEROS_ENABLED = 'EMR_KERBEROS_ENABLED';
/**
 * (experimental) Checks whether Amazon Elastic MapReduce (EMR) clusters' master nodes have public IPs.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/emr-master-no-public-ip.html
 * @experimental
 */
ManagedRuleIdentifiers.EMR_MASTER_NO_PUBLIC_IP = 'EMR_MASTER_NO_PUBLIC_IP';
/**
 * (experimental) Checks whether the EBS volumes that are in an attached state are encrypted.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/encrypted-volumes.html
 * @experimental
 */
ManagedRuleIdentifiers.EBS_ENCRYPTED_VOLUMES = 'ENCRYPTED_VOLUMES';
/**
 * (experimental) Checks whether the security groups associated inScope resources are compliant with the master security groups at each rule level based on allowSecurityGroup and denySecurityGroup flag.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-security-group-audit-policy-check.html
 * @experimental
 */
ManagedRuleIdentifiers.FMS_SECURITY_GROUP_AUDIT_POLICY_CHECK = 'FMS_SECURITY_GROUP_AUDIT_POLICY_CHECK';
/**
 * (experimental) Checks whether AWS Firewall Manager created security groups content is the same as the master security groups.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-security-group-content-check.html
 * @experimental
 */
ManagedRuleIdentifiers.FMS_SECURITY_GROUP_CONTENT_CHECK = 'FMS_SECURITY_GROUP_CONTENT_CHECK';
/**
 * (experimental) Checks whether Amazon EC2 or an elastic network interface is associated with AWS Firewall Manager security groups.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-security-group-resource-association-check.html
 * @experimental
 */
ManagedRuleIdentifiers.FMS_SECURITY_GROUP_RESOURCE_ASSOCIATION_CHECK = 'FMS_SECURITY_GROUP_RESOURCE_ASSOCIATION_CHECK';
/**
 * (experimental) Checks whether an Application Load Balancer, Amazon CloudFront distributions, Elastic Load Balancer or Elastic IP has AWS Shield protection.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-shield-resource-policy-check.html
 * @experimental
 */
ManagedRuleIdentifiers.FMS_SHIELD_RESOURCE_POLICY_CHECK = 'FMS_SHIELD_RESOURCE_POLICY_CHECK';
/**
 * (experimental) Checks whether the web ACL is associated with an Application Load Balancer, API Gateway stage, or Amazon CloudFront distributions.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-webacl-resource-policy-check.html
 * @experimental
 */
ManagedRuleIdentifiers.FMS_WEBACL_RESOURCE_POLICY_CHECK = 'FMS_WEBACL_RESOURCE_POLICY_CHECK';
/**
 * (experimental) Checks that the rule groups associate with the web ACL at the correct priority.
 *
 * The correct priority is decided by the rank of the rule groups in the ruleGroups parameter.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-webacl-rulegroup-association-check.html
 * @experimental
 */
ManagedRuleIdentifiers.FMS_WEBACL_RULEGROUP_ASSOCIATION_CHECK = 'FMS_WEBACL_RULEGROUP_ASSOCIATION_CHECK';
/**
 * (experimental) Checks whether Amazon GuardDuty is enabled in your AWS account and region.
 *
 * If you provide an AWS account for centralization,
 * the rule evaluates the Amazon GuardDuty results in the centralized account.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/guardduty-enabled-centralized.html
 * @experimental
 */
ManagedRuleIdentifiers.GUARDDUTY_ENABLED_CENTRALIZED = 'GUARDDUTY_ENABLED_CENTRALIZED';
/**
 * (experimental) Checks whether the Amazon GuardDuty has findings that are non archived.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/guardduty-non-archived-findings.html
 * @experimental
 */
ManagedRuleIdentifiers.GUARDDUTY_NON_ARCHIVED_FINDINGS = 'GUARDDUTY_NON_ARCHIVED_FINDINGS';
/**
 * (experimental) Checks that inline policy feature is not in use.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-no-inline-policy-check.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_NO_INLINE_POLICY_CHECK = 'IAM_NO_INLINE_POLICY_CHECK';
/**
 * (experimental) Checks whether IAM groups have at least one IAM user.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-group-has-users-check.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_GROUP_HAS_USERS_CHECK = 'IAM_GROUP_HAS_USERS_CHECK';
/**
 * (experimental) Checks whether the account password policy for IAM users meets the specified requirements indicated in the parameters.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-password-policy.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_PASSWORD_POLICY = 'IAM_PASSWORD_POLICY';
/**
 * (experimental) Checks whether for each IAM resource, a policy ARN in the input parameter is attached to the IAM resource.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-blacklisted-check.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_POLICY_BLOCKED_CHECK = 'IAM_POLICY_BLACKLISTED_CHECK';
/**
 * (experimental) Checks whether the IAM policy ARN is attached to an IAM user, or an IAM group with one or more IAM users, or an IAM role with one or more trusted entity.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-in-use.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_POLICY_IN_USE = 'IAM_POLICY_IN_USE';
/**
 * (experimental) Checks the IAM policies that you create for Allow statements that grant permissions to all actions on all resources.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-no-statements-with-admin-access.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_POLICY_NO_STATEMENTS_WITH_ADMIN_ACCESS = 'IAM_POLICY_NO_STATEMENTS_WITH_ADMIN_ACCESS';
/**
 * (experimental) Checks that AWS Identity and Access Management (IAM) policies in a list of policies are attached to all AWS roles.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-role-managed-policy-check.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_ROLE_MANAGED_POLICY_CHECK = 'IAM_ROLE_MANAGED_POLICY_CHECK';
/**
 * (experimental) Checks whether the root user access key is available.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-root-access-key-check.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_ROOT_ACCESS_KEY_CHECK = 'IAM_ROOT_ACCESS_KEY_CHECK';
/**
 * (experimental) Checks whether IAM users are members of at least one IAM group.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-user-group-membership-check.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_USER_GROUP_MEMBERSHIP_CHECK = 'IAM_USER_GROUP_MEMBERSHIP_CHECK';
/**
 * (experimental) Checks whether the AWS Identity and Access Management users have multi-factor authentication (MFA) enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-user-mfa-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_USER_MFA_ENABLED = 'IAM_USER_MFA_ENABLED';
/**
 * (experimental) Checks that none of your IAM users have policies attached.
 *
 * IAM users must inherit permissions from IAM groups or roles.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-user-no-policies-check.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_USER_NO_POLICIES_CHECK = 'IAM_USER_NO_POLICIES_CHECK';
/**
 * (experimental) Checks whether your AWS Identity and Access Management (IAM) users have passwords or active access keys that have not been used within the specified number of days you provided.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-user-unused-credentials-check.html
 * @experimental
 */
ManagedRuleIdentifiers.IAM_USER_UNUSED_CREDENTIALS_CHECK = 'IAM_USER_UNUSED_CREDENTIALS_CHECK';
/**
 * (experimental) Checks that Internet gateways (IGWs) are only attached to an authorized Amazon Virtual Private Cloud (VPCs).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/internet-gateway-authorized-vpc-only.html
 * @experimental
 */
ManagedRuleIdentifiers.INTERNET_GATEWAY_AUTHORIZED_VPC_ONLY = 'INTERNET_GATEWAY_AUTHORIZED_VPC_ONLY';
/**
 * (experimental) Checks whether customer master keys (CMKs) are not scheduled for deletion in AWS Key Management Service (KMS).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/kms-cmk-not-scheduled-for-deletion.html
 * @experimental
 */
ManagedRuleIdentifiers.KMS_CMK_NOT_SCHEDULED_FOR_DELETION = 'KMS_CMK_NOT_SCHEDULED_FOR_DELETION';
/**
 * (experimental) Checks whether the AWS Lambda function is configured with function-level concurrent execution limit.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-concurrency-check.html
 * @experimental
 */
ManagedRuleIdentifiers.LAMBDA_CONCURRENCY_CHECK = 'LAMBDA_CONCURRENCY_CHECK';
/**
 * (experimental) Checks whether an AWS Lambda function is configured with a dead-letter queue.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-dlq-check.html
 * @experimental
 */
ManagedRuleIdentifiers.LAMBDA_DLQ_CHECK = 'LAMBDA_DLQ_CHECK';
/**
 * (experimental) Checks whether the AWS Lambda function policy attached to the Lambda resource prohibits public access.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-function-public-access-prohibited.html
 * @experimental
 */
ManagedRuleIdentifiers.LAMBDA_FUNCTION_PUBLIC_ACCESS_PROHIBITED = 'LAMBDA_FUNCTION_PUBLIC_ACCESS_PROHIBITED';
/**
 * (experimental) Checks that the lambda function settings for runtime, role, timeout, and memory size match the expected values.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-function-settings-check.html
 * @experimental
 */
ManagedRuleIdentifiers.LAMBDA_FUNCTION_SETTINGS_CHECK = 'LAMBDA_FUNCTION_SETTINGS_CHECK';
/**
 * (experimental) Checks whether an AWS Lambda function is in an Amazon Virtual Private Cloud.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-inside-vpc.html
 * @experimental
 */
ManagedRuleIdentifiers.LAMBDA_INSIDE_VPC = 'LAMBDA_INSIDE_VPC';
/**
 * (experimental) Checks whether AWS Multi-Factor Authentication (MFA) is enabled for all IAM users that use a console password.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/mfa-enabled-for-iam-console-access.html
 * @experimental
 */
ManagedRuleIdentifiers.MFA_ENABLED_FOR_IAM_CONSOLE_ACCESS = 'MFA_ENABLED_FOR_IAM_CONSOLE_ACCESS';
/**
 * (experimental) Checks that there is at least one multi-region AWS CloudTrail.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/multi-region-cloudtrail-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.CLOUDTRAIL_MULTI_REGION_ENABLED = 'MULTI_REGION_CLOUD_TRAIL_ENABLED';
/**
 * (experimental) Checks if an Amazon Relational Database Service (Amazon RDS) cluster has deletion protection enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-cluster-deletion-protection-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.RDS_CLUSTER_DELETION_PROTECTION_ENABLED = 'RDS_CLUSTER_DELETION_PROTECTION_ENABLED';
/**
 * (experimental) Checks if an Amazon Relational Database Service (Amazon RDS) instance has deletion protection enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-deletion-protection-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.RDS_INSTANCE_DELETION_PROTECTION_ENABLED = 'RDS_INSTANCE_DELETION_PROTECTION_ENABLED';
/**
 * (experimental) Checks if an Amazon RDS instance has AWS Identity and Access Management (IAM) authentication enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-iam-authentication-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.RDS_INSTANCE_IAM_AUTHENTICATION_ENABLED = 'RDS_INSTANCE_IAM_AUTHENTICATION_ENABLED';
/**
 * (experimental) Checks that respective logs of Amazon Relational Database Service (Amazon RDS) are enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-logging-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.RDS_LOGGING_ENABLED = 'RDS_LOGGING_ENABLED';
/**
 * (experimental) Checks that Amazon Redshift automated snapshots are enabled for clusters.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-backup-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.REDSHIFT_BACKUP_ENABLED = 'REDSHIFT_BACKUP_ENABLED';
/**
 * (experimental) Checks whether enhanced monitoring is enabled for Amazon Relational Database Service (Amazon RDS) instances.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-enhanced-monitoring-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.RDS_ENHANCED_MONITORING_ENABLED = 'RDS_ENHANCED_MONITORING_ENABLED';
/**
 * (experimental) Checks whether Amazon Relational Database Service (Amazon RDS) DB snapshots are encrypted.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshot-encrypted.html
 * @experimental
 */
ManagedRuleIdentifiers.RDS_SNAPSHOT_ENCRYPTED = 'RDS_SNAPSHOT_ENCRYPTED';
/**
 * (experimental) Checks whether Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-require-tls-ssl.html
 * @experimental
 */
ManagedRuleIdentifiers.REDSHIFT_REQUIRE_TLS_SSL = 'REDSHIFT_REQUIRE_TLS_SSL';
/**
 * (experimental) Checks whether Amazon RDS database is present in back plans of AWS Backup.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-in-backup-plan.html
 * @experimental
 */
ManagedRuleIdentifiers.RDS_IN_BACKUP_PLAN = 'RDS_IN_BACKUP_PLAN';
/**
 * (experimental) Check whether the Amazon Relational Database Service instances are not publicly accessible.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-public-access-check.html
 * @experimental
 */
ManagedRuleIdentifiers.RDS_INSTANCE_PUBLIC_ACCESS_CHECK = 'RDS_INSTANCE_PUBLIC_ACCESS_CHECK';
/**
 * (experimental) Checks whether high availability is enabled for your RDS DB instances.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-multi-az-support.html
 * @experimental
 */
ManagedRuleIdentifiers.RDS_MULTI_AZ_SUPPORT = 'RDS_MULTI_AZ_SUPPORT';
/**
 * (experimental) Checks if Amazon Relational Database Service (Amazon RDS) snapshots are public.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshots-public-prohibited.html
 * @experimental
 */
ManagedRuleIdentifiers.RDS_SNAPSHOTS_PUBLIC_PROHIBITED = 'RDS_SNAPSHOTS_PUBLIC_PROHIBITED';
/**
 * (experimental) Checks whether storage encryption is enabled for your RDS DB instances.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-storage-encrypted.html
 * @experimental
 */
ManagedRuleIdentifiers.RDS_STORAGE_ENCRYPTED = 'RDS_STORAGE_ENCRYPTED';
/**
 * (experimental) Checks whether Amazon Redshift clusters have the specified settings.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-configuration-check.html
 * @experimental
 */
ManagedRuleIdentifiers.REDSHIFT_CLUSTER_CONFIGURATION_CHECK = 'REDSHIFT_CLUSTER_CONFIGURATION_CHECK';
/**
 * (experimental) Checks whether Amazon Redshift clusters have the specified maintenance settings.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-maintenancesettings-check.html
 * @experimental
 */
ManagedRuleIdentifiers.REDSHIFT_CLUSTER_MAINTENANCE_SETTINGS_CHECK = 'REDSHIFT_CLUSTER_MAINTENANCESETTINGS_CHECK';
/**
 * (experimental) Checks whether Amazon Redshift clusters are not publicly accessible.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-public-access-check.html
 * @experimental
 */
ManagedRuleIdentifiers.REDSHIFT_CLUSTER_PUBLIC_ACCESS_CHECK = 'REDSHIFT_CLUSTER_PUBLIC_ACCESS_CHECK';
/**
 * (experimental) Checks whether your resources have the tags that you specify.
 *
 * For example, you can check whether your Amazon EC2 instances have the CostCenter tag.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/required-tags.html
 * @experimental
 */
ManagedRuleIdentifiers.REQUIRED_TAGS = 'REQUIRED_TAGS';
/**
 * (experimental) Checks whether the security groups in use do not allow unrestricted incoming TCP traffic to the specified ports.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/restricted-common-ports.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_SECURITY_GROUPS_RESTRICTED_INCOMING_TRAFFIC = 'RESTRICTED_INCOMING_TRAFFIC';
/**
 * (experimental) Checks whether the incoming SSH traffic for the security groups is accessible.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/restricted-ssh.html
 * @experimental
 */
ManagedRuleIdentifiers.EC2_SECURITY_GROUPS_INCOMING_SSH_DISABLED = 'INCOMING_SSH_DISABLED';
/**
 * (experimental) Checks whether your AWS account is enabled to use multi-factor authentication (MFA) hardware device to sign in with root credentials.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/root-account-hardware-mfa-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.ROOT_ACCOUNT_HARDWARE_MFA_ENABLED = 'ROOT_ACCOUNT_HARDWARE_MFA_ENABLED';
/**
 * (experimental) Checks whether users of your AWS account require a multi-factor authentication (MFA) device to sign in with root credentials.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/root-account-mfa-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.ROOT_ACCOUNT_MFA_ENABLED = 'ROOT_ACCOUNT_MFA_ENABLED';
/**
 * (experimental) Checks whether Amazon Simple Storage Service (Amazon S3) bucket has lock enabled, by default.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-default-lock-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_BUCKET_DEFAULT_LOCK_ENABLED = 'S3_BUCKET_DEFAULT_LOCK_ENABLED';
/**
 * (experimental) Checks whether the Amazon Simple Storage Service (Amazon S3) buckets are encrypted with AWS Key Management Service (AWS KMS).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-default-encryption-kms.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_DEFAULT_ENCRYPTION_KMS = 'S3_DEFAULT_ENCRYPTION_KMS';
/**
 * (experimental) Checks that AWS Security Hub is enabled for an AWS account.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/securityhub-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.SECURITYHUB_ENABLED = 'SECURITYHUB_ENABLED';
/**
 * (experimental) Checks whether Amazon SNS topic is encrypted with AWS Key Management Service (AWS KMS).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sns-encrypted-kms.html
 * @experimental
 */
ManagedRuleIdentifiers.SNS_ENCRYPTED_KMS = 'SNS_ENCRYPTED_KMS';
/**
 * (experimental) Checks whether the required public access block settings are configured from account level.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-account-level-public-access-blocks.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_ACCOUNT_LEVEL_PUBLIC_ACCESS_BLOCKS = 'S3_ACCOUNT_LEVEL_PUBLIC_ACCESS_BLOCKS';
/**
 * (experimental) Checks that the Amazon Simple Storage Service bucket policy does not allow blocked bucket-level and object-level actions on resources in the bucket for principals from other AWS accounts.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-blacklisted-actions-prohibited.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_BUCKET_BLOCKED_ACTIONS_PROHIBITED = 'S3_BUCKET_BLACKLISTED_ACTIONS_PROHIBITED';
/**
 * (experimental) Verifies that your Amazon Simple Storage Service bucket policies do not allow other inter-account permissions than the control Amazon S3 bucket policy provided.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-policy-not-more-permissive.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_BUCKET_POLICY_NOT_MORE_PERMISSIVE = 'S3_BUCKET_POLICY_NOT_MORE_PERMISSIVE';
/**
 * (experimental) Checks whether logging is enabled for your S3 buckets.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-logging-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_BUCKET_LOGGING_ENABLED = 'S3_BUCKET_LOGGING_ENABLED';
/**
 * (experimental) Checks that the access granted by the Amazon S3 bucket is restricted by any of the AWS principals, federated users, service principals, IP addresses, or VPCs that you provide.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-policy-grantee-check.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_BUCKET_POLICY_GRANTEE_CHECK = 'S3_BUCKET_POLICY_GRANTEE_CHECK';
/**
 * (experimental) Checks that your Amazon S3 buckets do not allow public read access.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-read-prohibited.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_BUCKET_PUBLIC_READ_PROHIBITED = 'S3_BUCKET_PUBLIC_READ_PROHIBITED';
/**
 * (experimental) Checks that your Amazon S3 buckets do not allow public write access.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-write-prohibited.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_BUCKET_PUBLIC_WRITE_PROHIBITED = 'S3_BUCKET_PUBLIC_WRITE_PROHIBITED';
/**
 * (experimental) Checks whether S3 buckets have cross-region replication enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-replication-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_BUCKET_REPLICATION_ENABLED = 'S3_BUCKET_REPLICATION_ENABLED';
/**
 * (experimental) Checks that your Amazon S3 bucket either has Amazon S3 default encryption enabled or that the S3 bucket policy explicitly denies put-object requests without server side encryption that uses AES-256 or AWS Key Management Service.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-server-side-encryption-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_BUCKET_SERVER_SIDE_ENCRYPTION_ENABLED = 'S3_BUCKET_SERVER_SIDE_ENCRYPTION_ENABLED';
/**
 * (experimental) Checks whether S3 buckets have policies that require requests to use Secure Socket Layer (SSL).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-ssl-requests-only.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_BUCKET_SSL_REQUESTS_ONLY = 'S3_BUCKET_SSL_REQUESTS_ONLY';
/**
 * (experimental) Checks whether versioning is enabled for your S3 buckets.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-versioning-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.S3_BUCKET_VERSIONING_ENABLED = 'S3_BUCKET_VERSIONING_ENABLED';
/**
 * (experimental) Checks whether AWS Key Management Service (KMS) key is configured for an Amazon SageMaker endpoint configuration.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-endpoint-configuration-kms-key-configured.html
 * @experimental
 */
ManagedRuleIdentifiers.SAGEMAKER_ENDPOINT_CONFIGURATION_KMS_KEY_CONFIGURED = 'SAGEMAKER_ENDPOINT_CONFIGURATION_KMS_KEY_CONFIGURED';
/**
 * (experimental) Check whether an AWS Key Management Service (KMS) key is configured for SageMaker notebook instance.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-instance-kms-key-configured.html
 * @experimental
 */
ManagedRuleIdentifiers.SAGEMAKER_NOTEBOOK_INSTANCE_KMS_KEY_CONFIGURED = 'SAGEMAKER_NOTEBOOK_INSTANCE_KMS_KEY_CONFIGURED';
/**
 * (experimental) Checks whether direct internet access is disabled for an Amazon SageMaker notebook instance.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-no-direct-internet-access.html
 * @experimental
 */
ManagedRuleIdentifiers.SAGEMAKER_NOTEBOOK_NO_DIRECT_INTERNET_ACCESS = 'SAGEMAKER_NOTEBOOK_NO_DIRECT_INTERNET_ACCESS';
/**
 * (experimental) Checks whether AWS Secrets Manager secret has rotation enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-rotation-enabled-check.html
 * @experimental
 */
ManagedRuleIdentifiers.SECRETSMANAGER_ROTATION_ENABLED_CHECK = 'SECRETSMANAGER_ROTATION_ENABLED_CHECK';
/**
 * (experimental) Checks whether AWS Secrets Manager secret rotation has rotated successfully as per the rotation schedule.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-scheduled-rotation-success-check.html
 * @experimental
 */
ManagedRuleIdentifiers.SECRETSMANAGER_SCHEDULED_ROTATION_SUCCESS_CHECK = 'SECRETSMANAGER_SCHEDULED_ROTATION_SUCCESS_CHECK';
/**
 * (experimental) Checks whether Service Endpoint for the service provided in rule parameter is created for each Amazon VPC.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/service-vpc-endpoint-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.SERVICE_VPC_ENDPOINT_ENABLED = 'SERVICE_VPC_ENDPOINT_ENABLED';
/**
 * (experimental) Checks whether EBS volumes are attached to EC2 instances.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/shield-advanced-enabled-autorenew.html
 * @experimental
 */
ManagedRuleIdentifiers.SHIELD_ADVANCED_ENABLED_AUTO_RENEW = 'SHIELD_ADVANCED_ENABLED_AUTORENEW';
/**
 * (experimental) Verify that DDoS response team (DRT) can access AWS account.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/shield-drt-access.html
 * @experimental
 */
ManagedRuleIdentifiers.SHIELD_DRT_ACCESS = 'SHIELD_DRT_ACCESS';
/**
 * (experimental) Checks that the default security group of any Amazon Virtual Private Cloud (VPC) does not allow inbound or outbound traffic.
 *
 * The rule returns NOT_APPLICABLE if the security group
 * is not default.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-default-security-group-closed.html
 * @experimental
 */
ManagedRuleIdentifiers.VPC_DEFAULT_SECURITY_GROUP_CLOSED = 'VPC_DEFAULT_SECURITY_GROUP_CLOSED';
/**
 * (experimental) Checks whether Amazon Virtual Private Cloud flow logs are found and enabled for Amazon VPC.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-flow-logs-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.VPC_FLOW_LOGS_ENABLED = 'VPC_FLOW_LOGS_ENABLED';
/**
 * (experimental) Checks whether the security group with 0.0.0.0/0 of any Amazon Virtual Private Cloud (Amazon VPC) allows only specific inbound TCP or UDP traffic.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-sg-open-only-to-authorized-ports.html
 * @experimental
 */
ManagedRuleIdentifiers.VPC_SG_OPEN_ONLY_TO_AUTHORIZED_PORTS = 'VPC_SG_OPEN_ONLY_TO_AUTHORIZED_PORTS';
/**
 * (experimental) Checks that both AWS Virtual Private Network tunnels provided by AWS Site-to-Site VPN are in UP status.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-vpn-2-tunnels-up.html
 * @experimental
 */
ManagedRuleIdentifiers.VPC_VPN_2_TUNNELS_UP = 'VPC_VPN_2_TUNNELS_UP';
/**
 * (experimental) Checks if logging is enabled on AWS Web Application Firewall (WAF) classic global web ACLs.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/waf-classic-logging-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.WAF_CLASSIC_LOGGING_ENABLED = 'WAF_CLASSIC_LOGGING_ENABLED';
/**
 * (experimental) Checks whether logging is enabled on AWS Web Application Firewall (WAFV2) regional and global web access control list (ACLs).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/wafv2-logging-enabled.html
 * @experimental
 */
ManagedRuleIdentifiers.WAFV2_LOGGING_ENABLED = 'WAFV2_LOGGING_ENABLED';
/**
 * (experimental) Resources types that are supported by AWS Config.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html
 * @experimental
 */
class ResourceType {
    constructor(type) {
        this.complianceResourceType = type;
    }
    /**
     * (experimental) A custom resource type to support future cases.
     *
     * @experimental
     */
    static of(type) {
        return new ResourceType(type);
    }
}
exports.ResourceType = ResourceType;
/**
 * (experimental) API Gateway Stage.
 *
 * @experimental
 */
ResourceType.APIGATEWAY_STAGE = new ResourceType('AWS::ApiGateway::Stage');
/**
 * (experimental) API Gatewayv2 Stage.
 *
 * @experimental
 */
ResourceType.APIGATEWAYV2_STAGE = new ResourceType('AWS::ApiGatewayV2::Stage');
/**
 * (experimental) API Gateway REST API.
 *
 * @experimental
 */
ResourceType.APIGATEWAY_REST_API = new ResourceType('AWS::ApiGateway::RestApi');
/**
 * (experimental) API Gatewayv2 API.
 *
 * @experimental
 */
ResourceType.APIGATEWAYV2_API = new ResourceType('AWS::ApiGatewayV2::Api');
/**
 * (experimental) Amazon CloudFront Distribution.
 *
 * @experimental
 */
ResourceType.CLOUDFRONT_DISTRIBUTION = new ResourceType('AWS::CloudFront::Distribution');
/**
 * (experimental) Amazon CloudFront streaming distribution.
 *
 * @experimental
 */
ResourceType.CLOUDFRONT_STREAMING_DISTRIBUTION = new ResourceType('AWS::CloudFront::StreamingDistribution');
/**
 * (experimental) Amazon CloudWatch Alarm.
 *
 * @experimental
 */
ResourceType.CLOUDWATCH_ALARM = new ResourceType('AWS::CloudWatch::Alarm');
/**
 * (experimental) Amazon DynamoDB Table.
 *
 * @experimental
 */
ResourceType.DYNAMODB_TABLE = new ResourceType('AWS::DynamoDB::Table');
/**
 * (experimental) Elastic Block Store (EBS) volume.
 *
 * @experimental
 */
ResourceType.EBS_VOLUME = new ResourceType('AWS::EC2::Volume');
/**
 * (experimental) EC2 host.
 *
 * @experimental
 */
ResourceType.EC2_HOST = new ResourceType('AWS::EC2::Host');
/**
 * (experimental) EC2 Elastic IP.
 *
 * @experimental
 */
ResourceType.EC2_EIP = new ResourceType('AWS::EC2::EIP');
/**
 * (experimental) EC2 instance.
 *
 * @experimental
 */
ResourceType.EC2_INSTANCE = new ResourceType('AWS::EC2::Instance');
/**
 * (experimental) EC2 security group.
 *
 * @experimental
 */
ResourceType.EC2_SECURITY_GROUP = new ResourceType('AWS::EC2::SecurityGroup');
/**
 * (experimental) EC2 NAT gateway.
 *
 * @experimental
 */
ResourceType.EC2_NAT_GATEWAY = new ResourceType('AWS::EC2::NatGateway');
/**
 * (experimental) EC2 Egress only internet gateway.
 *
 * @experimental
 */
ResourceType.EC2_EGRESS_ONLY_INTERNET_GATEWAY = new ResourceType('AWS::EC2::EgressOnlyInternetGateway');
/**
 * (experimental) EC2 flow log.
 *
 * @experimental
 */
ResourceType.EC2_FLOW_LOG = new ResourceType('AWS::EC2::FlowLog');
/**
 * (experimental) EC2 VPC endpoint.
 *
 * @experimental
 */
ResourceType.EC2_VPC_ENDPOINT = new ResourceType('AWS::EC2::VPCEndpoint');
/**
 * (experimental) EC2 VPC endpoint service.
 *
 * @experimental
 */
ResourceType.EC2_VPC_ENDPOINT_SERVICE = new ResourceType('AWS::EC2::VPCEndpointService');
/**
 * (experimental) EC2 VPC peering connection.
 *
 * @experimental
 */
ResourceType.EC2_VPC_PEERING_CONNECTION = new ResourceType('AWS::EC2::VPCPeeringConnection');
/**
 * (experimental) Amazon ElasticSearch domain.
 *
 * @experimental
 */
ResourceType.ELASTICSEARCH_DOMAIN = new ResourceType('AWS::Elasticsearch::Domain');
/**
 * (experimental) Amazon QLDB ledger.
 *
 * @experimental
 */
ResourceType.QLDB_LEDGER = new ResourceType('AWS::QLDB::Ledger');
/**
 * (experimental) Amazon Redshift cluster.
 *
 * @experimental
 */
ResourceType.REDSHIFT_CLUSTER = new ResourceType('AWS::Redshift::Cluster');
/**
 * (experimental) Amazon Redshift cluster parameter group.
 *
 * @experimental
 */
ResourceType.REDSHIFT_CLUSTER_PARAMETER_GROUP = new ResourceType('AWS::Redshift::ClusterParameterGroup');
/**
 * (experimental) Amazon Redshift cluster security group.
 *
 * @experimental
 */
ResourceType.REDSHIFT_CLUSTER_SECURITY_GROUP = new ResourceType('AWS::Redshift::ClusterSecurityGroup');
/**
 * (experimental) Amazon Redshift cluster snapshot.
 *
 * @experimental
 */
ResourceType.REDSHIFT_CLUSTER_SNAPSHOT = new ResourceType('AWS::Redshift::ClusterSnapshot');
/**
 * (experimental) Amazon Redshift cluster subnet group.
 *
 * @experimental
 */
ResourceType.REDSHIFT_CLUSTER_SUBNET_GROUP = new ResourceType('AWS::Redshift::ClusterSubnetGroup');
/**
 * (experimental) Amazon Redshift event subscription.
 *
 * @experimental
 */
ResourceType.REDSHIFT_EVENT_SUBSCRIPTION = new ResourceType('AWS::Redshift::EventSubscription');
/**
 * (experimental) Amazon RDS database instance.
 *
 * @experimental
 */
ResourceType.RDS_DB_INSTANCE = new ResourceType('AWS::RDS::DBInstance');
/**
 * (experimental) Amazon RDS database security group.
 *
 * @experimental
 */
ResourceType.RDS_DB_SECURITY_GROUP = new ResourceType('AWS::RDS::DBSecurityGroup');
/**
 * (experimental) Amazon RDS database snapshot.
 *
 * @experimental
 */
ResourceType.RDS_DB_SNAPSHOT = new ResourceType('AWS::RDS::DBSnapshot');
/**
 * (experimental) Amazon RDS database subnet group.
 *
 * @experimental
 */
ResourceType.RDS_DB_SUBNET_GROUP = new ResourceType('AWS::RDS::DBSubnetGroup');
/**
 * (experimental) Amazon RDS event subscription.
 *
 * @experimental
 */
ResourceType.RDS_EVENT_SUBSCRIPTION = new ResourceType('AWS::RDS::EventSubscription');
/**
 * (experimental) Amazon RDS database cluster.
 *
 * @experimental
 */
ResourceType.RDS_DB_CLUSTER = new ResourceType('AWS::RDS::DBCluster');
/**
 * (experimental) Amazon RDS database cluster snapshot.
 *
 * @experimental
 */
ResourceType.RDS_DB_CLUSTER_SNAPSHOT = new ResourceType('AWS::RDS::DBClusterSnapshot');
/**
 * (experimental) Amazon SQS queue.
 *
 * @experimental
 */
ResourceType.SQS_QUEUE = new ResourceType('AWS::SQS::Queue');
/**
 * (experimental) Amazon SNS topic.
 *
 * @experimental
 */
ResourceType.SNS_TOPIC = new ResourceType('AWS::SNS::Topic');
/**
 * (experimental) Amazon S3 bucket.
 *
 * @experimental
 */
ResourceType.S3_BUCKET = new ResourceType('AWS::S3::Bucket');
/**
 * (experimental) Amazon S3 account public access block.
 *
 * @experimental
 */
ResourceType.S3_ACCOUNT_PUBLIC_ACCESS_BLOCK = new ResourceType('AWS::S3::AccountPublicAccessBlock');
/**
 * (experimental) Amazon EC2 customer gateway.
 *
 * @experimental
 */
ResourceType.EC2_CUSTOMER_GATEWAY = new ResourceType('AWS::EC2::CustomerGateway');
/**
 * (experimental) Amazon EC2 internet gateway.
 *
 * @experimental
 */
ResourceType.EC2_INTERNET_GATEWAY = new ResourceType('AWS::EC2::CustomerGateway');
/**
 * (experimental) Amazon EC2 network ACL.
 *
 * @experimental
 */
ResourceType.EC2_NETWORK_ACL = new ResourceType('AWS::EC2::NetworkAcl');
/**
 * (experimental) Amazon EC2 route table.
 *
 * @experimental
 */
ResourceType.EC2_ROUTE_TABLE = new ResourceType('AWS::EC2::RouteTable');
/**
 * (experimental) Amazon EC2 subnet table.
 *
 * @experimental
 */
ResourceType.EC2_SUBNET = new ResourceType('AWS::EC2::Subnet');
/**
 * (experimental) Amazon EC2 VPC.
 *
 * @experimental
 */
ResourceType.EC2_VPC = new ResourceType('AWS::EC2::VPC');
/**
 * (experimental) Amazon EC2 VPN connection.
 *
 * @experimental
 */
ResourceType.EC2_VPN_CONNECTION = new ResourceType('AWS::EC2::VPNConnection');
/**
 * (experimental) Amazon EC2 VPN gateway.
 *
 * @experimental
 */
ResourceType.EC2_VPN_GATEWAY = new ResourceType('AWS::EC2::VPNGateway');
/**
 * (experimental) AWS Auto Scaling group.
 *
 * @experimental
 */
ResourceType.AUTO_SCALING_GROUP = new ResourceType('AWS::AutoScaling::AutoScalingGroup');
/**
 * (experimental) AWS Auto Scaling launch configuration.
 *
 * @experimental
 */
ResourceType.AUTO_SCALING_LAUNCH_CONFIGURATION = new ResourceType('AWS::AutoScaling::LaunchConfiguration');
/**
 * (experimental) AWS Auto Scaling policy.
 *
 * @experimental
 */
ResourceType.AUTO_SCALING_POLICY = new ResourceType('AWS::AutoScaling::ScalingPolicy');
/**
 * (experimental) AWS Auto Scaling scheduled action.
 *
 * @experimental
 */
ResourceType.AUTO_SCALING_SCHEDULED_ACTION = new ResourceType('AWS::AutoScaling::ScheduledAction');
/**
 * (experimental) AWS Certificate manager certificate.
 *
 * @experimental
 */
ResourceType.ACM_CERTIFICATE = new ResourceType('AWS::ACM::Certificate');
/**
 * (experimental) AWS CloudFormation stack.
 *
 * @experimental
 */
ResourceType.CLOUDFORMATION_STACK = new ResourceType('AWS::CloudFormation::Stack');
/**
 * (experimental) AWS CloudTrail trail.
 *
 * @experimental
 */
ResourceType.CLOUDTRAIL_TRAIL = new ResourceType('AWS::CloudTrail::Trail');
/**
 * (experimental) AWS CodeBuild project.
 *
 * @experimental
 */
ResourceType.CODEBUILD_PROJECT = new ResourceType('AWS::CodeBuild::Project');
/**
 * (experimental) AWS CodePipeline pipeline.
 *
 * @experimental
 */
ResourceType.CODEPIPELINE_PIPELINE = new ResourceType('AWS::CodePipeline::Pipeline');
/**
 * (experimental) AWS Elastic Beanstalk (EB) application.
 *
 * @experimental
 */
ResourceType.ELASTIC_BEANSTALK_APPLICATION = new ResourceType('AWS::ElasticBeanstalk::Application');
/**
 * (experimental) AWS Elastic Beanstalk (EB) application version.
 *
 * @experimental
 */
ResourceType.ELASTIC_BEANSTALK_APPLICATION_VERSION = new ResourceType('AWS::ElasticBeanstalk::ApplicationVersion');
/**
 * (experimental) AWS Elastic Beanstalk (EB) environment.
 *
 * @experimental
 */
ResourceType.ELASTIC_BEANSTALK_ENVIRONMENT = new ResourceType('AWS::ElasticBeanstalk::Environment');
/**
 * (experimental) AWS IAM user.
 *
 * @experimental
 */
ResourceType.IAM_USER = new ResourceType('AWS::IAM::User');
/**
 * (experimental) AWS IAM group.
 *
 * @experimental
 */
ResourceType.IAM_GROUP = new ResourceType('AWS::IAM::Group');
/**
 * (experimental) AWS IAM role.
 *
 * @experimental
 */
ResourceType.IAM_ROLE = new ResourceType('AWS::IAM::Role');
/**
 * (experimental) AWS IAM policy.
 *
 * @experimental
 */
ResourceType.IAM_POLICY = new ResourceType('AWS::IAM::Policy');
/**
 * (experimental) AWS KMS Key.
 *
 * @experimental
 */
ResourceType.KMS_KEY = new ResourceType('AWS::KMS::Key');
/**
 * (experimental) AWS Lambda function.
 *
 * @experimental
 */
ResourceType.LAMBDA_FUNCTION = new ResourceType('AWS::Lambda::Function');
/**
 * (experimental) AWS Secrets Manager secret.
 *
 * @experimental
 */
ResourceType.SECRETS_MANAGER_SECRET = new ResourceType('AWS::SecretsManager::Secret');
/**
 * (experimental) AWS Service Catalog CloudFormation product.
 *
 * @experimental
 */
ResourceType.SERVICE_CATALOG_CLOUDFORMATION_PRODUCT = new ResourceType('AWS::ServiceCatalog::CloudFormationProduct');
/**
 * (experimental) AWS Service Catalog CloudFormation provisioned product.
 *
 * @experimental
 */
ResourceType.SERVICE_CATALOG_CLOUDFORMATION_PROVISIONED_PRODUCT = new ResourceType('AWS::ServiceCatalog::CloudFormationProvisionedProduct');
/**
 * (experimental) AWS Service Catalog portfolio.
 *
 * @experimental
 */
ResourceType.SERVICE_CATALOG_PORTFOLIO = new ResourceType('AWS::ServiceCatalog::Portfolio');
/**
 * (experimental) AWS Shield protection.
 *
 * @experimental
 */
ResourceType.SHIELD_PROTECTION = new ResourceType('AWS::Shield::Protection');
/**
 * (experimental) AWS Shield regional protection.
 *
 * @experimental
 */
ResourceType.SHIELD_REGIONAL_PROTECTION = new ResourceType('AWS::ShieldRegional::Protection');
/**
 * (experimental) AWS Systems Manager managed instance inventory.
 *
 * @experimental
 */
ResourceType.SYSTEMS_MANAGER_MANAGED_INSTANCE_INVENTORY = new ResourceType('AWS::SSM::ManagedInstanceInventory');
/**
 * (experimental) AWS Systems Manager patch compliance.
 *
 * @experimental
 */
ResourceType.SYSTEMS_MANAGER_PATCH_COMPLIANCE = new ResourceType('AWS::SSM::PatchCompliance');
/**
 * (experimental) AWS Systems Manager association compliance.
 *
 * @experimental
 */
ResourceType.SYSTEMS_MANAGER_ASSOCIATION_COMPLIANCE = new ResourceType('AWS::SSM::AssociationCompliance');
/**
 * (experimental) AWS Systems Manager file data.
 *
 * @experimental
 */
ResourceType.SYSTEMS_MANAGER_FILE_DATA = new ResourceType('AWS::SSM::FileData');
/**
 * (experimental) AWS WAF rate based rule.
 *
 * @experimental
 */
ResourceType.WAF_RATE_BASED_RULE = new ResourceType('AWS::WAF::RateBasedRule');
/**
 * (experimental) AWS WAF rule.
 *
 * @experimental
 */
ResourceType.WAF_RULE = new ResourceType('AWS::WAF::Rule');
/**
 * (experimental) AWS WAF web ACL.
 *
 * @experimental
 */
ResourceType.WAF_WEB_ACL = new ResourceType('AWS::WAF::WebACL');
/**
 * (experimental) AWS WAF rule group.
 *
 * @experimental
 */
ResourceType.WAF_RULE_GROUP = new ResourceType('AWS::WAF::RuleGroup');
/**
 * (experimental) AWS WAF regional rate based rule.
 *
 * @experimental
 */
ResourceType.WAF_REGIONAL_RATE_BASED_RULE = new ResourceType('AWS::WAFRegional::RateBasedRule');
/**
 * (experimental) AWS WAF regional rule.
 *
 * @experimental
 */
ResourceType.WAF_REGIONAL_RULE = new ResourceType('AWS::WAFRegional::Rule');
/**
 * (experimental) AWS WAF web ACL.
 *
 * @experimental
 */
ResourceType.WAF_REGIONAL_WEB_ACL = new ResourceType('AWS::WAFRegional::WebACL');
/**
 * (experimental) AWS WAF regional rule group.
 *
 * @experimental
 */
ResourceType.WAF_REGIONAL_RULE_GROUP = new ResourceType('AWS::WAFRegional::RuleGroup');
/**
 * (experimental) AWS WAFv2 web ACL.
 *
 * @experimental
 */
ResourceType.WAFV2_WEB_ACL = new ResourceType('AWS::WAFv2::WebACL');
/**
 * (experimental) AWS WAFv2 rule group.
 *
 * @experimental
 */
ResourceType.WAFV2_RULE_GROUP = new ResourceType('AWS::WAFv2::RuleGroup');
/**
 * (experimental) AWS WAFv2 managed rule set.
 *
 * @experimental
 */
ResourceType.WAFV2_MANAGED_RULE_SET = new ResourceType('AWS::WAFv2::ManagedRuleSet');
/**
 * (experimental) AWS X-Ray encryption configuration.
 *
 * @experimental
 */
ResourceType.XRAY_ENCRYPTION_CONFIGURATION = new ResourceType('AWS::XRay::EncryptionConfig');
/**
 * (experimental) AWS ELB classic load balancer.
 *
 * @experimental
 */
ResourceType.ELB_LOAD_BALANCER = new ResourceType('AWS::ElasticLoadBalancing::LoadBalancer');
/**
 * (experimental) AWS ELBv2 network load balancer or AWS ELBv2 application load balancer.
 *
 * @experimental
 */
ResourceType.ELBV2_LOAD_BALANCER = new ResourceType('AWS::ElasticLoadBalancingV2::LoadBalancer');
function renderScope(ruleScope) {
    var _a;
    return ruleScope ? {
        complianceResourceId: ruleScope.resourceId,
        complianceResourceTypes: (_a = ruleScope.resourceTypes) === null || _a === void 0 ? void 0 : _a.map(resource => resource.complianceResourceType),
        tagKey: ruleScope.key,
        tagValue: ruleScope.value,
    } : undefined;
}
//# sourceMappingURL=data:application/json;base64,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