"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolDomain = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const custom_resources_1 = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
const cognito_generated_1 = require("./cognito.generated");
/**
 * (experimental) Define a user pool domain.
 *
 * @experimental
 */
class UserPoolDomain extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        if (!!props.customDomain === !!props.cognitoDomain) {
            throw new Error('One of, and only one of, cognitoDomain or customDomain must be specified');
        }
        if (((_a = props.cognitoDomain) === null || _a === void 0 ? void 0 : _a.domainPrefix) &&
            !core_1.Token.isUnresolved((_b = props.cognitoDomain) === null || _b === void 0 ? void 0 : _b.domainPrefix) &&
            !/^[a-z0-9-]+$/.test(props.cognitoDomain.domainPrefix)) {
            throw new Error('domainPrefix for cognitoDomain can contain only lowercase alphabets, numbers and hyphens');
        }
        this.isCognitoDomain = !!props.cognitoDomain;
        const domainName = ((_c = props.cognitoDomain) === null || _c === void 0 ? void 0 : _c.domainPrefix) || ((_d = props.customDomain) === null || _d === void 0 ? void 0 : _d.domainName);
        const resource = new cognito_generated_1.CfnUserPoolDomain(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            domain: domainName,
            customDomainConfig: props.customDomain ? { certificateArn: props.customDomain.certificate.certificateArn } : undefined,
        });
        this.domainName = resource.ref;
    }
    /**
     * (experimental) Import a UserPoolDomain given its domain name.
     *
     * @experimental
     */
    static fromDomainName(scope, id, userPoolDomainName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.domainName = userPoolDomainName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) The domain name of the CloudFront distribution associated with the user pool domain.
     *
     * @experimental
     */
    get cloudFrontDomainName() {
        if (!this.cloudFrontCustomResource) {
            const sdkCall = {
                service: 'CognitoIdentityServiceProvider',
                action: 'describeUserPoolDomain',
                parameters: {
                    Domain: this.domainName,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.domainName),
            };
            this.cloudFrontCustomResource = new custom_resources_1.AwsCustomResource(this, 'CloudFrontDomainName', {
                resourceType: 'Custom::UserPoolCloudFrontDomainName',
                onCreate: sdkCall,
                onUpdate: sdkCall,
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                    // DescribeUserPoolDomain only supports access level '*'
                    // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html#amazoncognitouserpools-actions-as-permissions
                    resources: ['*'],
                }),
            });
        }
        return this.cloudFrontCustomResource.getResponseField('DomainDescription.CloudFrontDistribution');
    }
    /**
     * (experimental) The URL to the hosted UI associated with this domain.
     *
     * @experimental
     */
    baseUrl() {
        if (this.isCognitoDomain) {
            return `https://${this.domainName}.auth.${core_1.Stack.of(this).region}.amazoncognito.com`;
        }
        return `https://${this.domainName}`;
    }
    /**
     * (experimental) The URL to the sign in page in this domain using a specific UserPoolClient.
     *
     * @param client [disable-awslint:ref-via-interface] the user pool client that the UI will use to interact with the UserPool.
     * @param options options to customize the behaviour of this method.
     * @experimental
     */
    signInUrl(client, options) {
        var _a;
        let responseType;
        if (client.oAuthFlows.authorizationCodeGrant) {
            responseType = 'code';
        }
        else if (client.oAuthFlows.implicitCodeGrant) {
            responseType = 'token';
        }
        else {
            throw new Error('signInUrl is not supported for clients without authorizationCodeGrant or implicitCodeGrant flow enabled');
        }
        const path = (_a = options.signInPath) !== null && _a !== void 0 ? _a : '/login';
        return `${this.baseUrl()}${path}?client_id=${client.userPoolClientId}&response_type=${responseType}&redirect_uri=${options.redirectUri}`;
    }
}
exports.UserPoolDomain = UserPoolDomain;
//# sourceMappingURL=data:application/json;base64,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