"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cognito_generated_1 = require("./cognito.generated");
/**
 * (experimental) OAuth scopes that are allowed with this client.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 * @experimental
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * (experimental) Custom scope is one that you define for your own resource server in the Resource Servers.
     *
     * The format is 'resource-server-identifier/scope'.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     * @experimental
     */
    static custom(name) {
        return new OAuthScope(name);
    }
}
exports.OAuthScope = OAuthScope;
/**
 * (experimental) Grants access to the 'phone_number' and 'phone_number_verified' claims.
 *
 * Automatically includes access to `OAuthScope.OPENID`.
 *
 * @experimental
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * (experimental) Grants access to the 'email' and 'email_verified' claims.
 *
 * Automatically includes access to `OAuthScope.OPENID`.
 *
 * @experimental
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * (experimental) Returns all user attributes in the ID token that are readable by the client.
 *
 * @experimental
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * (experimental) Grants access to all user attributes that are readable by the client Automatically includes access to `OAuthScope.OPENID`.
 *
 * @experimental
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * (experimental) Grants access to Amazon Cognito User Pool API operations that require access tokens, such as UpdateUserAttributes and VerifyUserAttribute.
 *
 * @experimental
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * (experimental) Identity providers supported by the UserPoolClient.
 *
 * @experimental
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * (experimental) Specify a provider not yet supported by the CDK.
     *
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`.
     * @experimental
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
/**
 * (experimental) Allow users to sign in using 'Facebook Login'.
 *
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 *
 * @experimental
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * (experimental) Allow users to sign in using 'Google Login'.
 *
 * A `UserPoolIdentityProviderGoogle` must be attached to the user pool.
 *
 * @experimental
 */
UserPoolClientIdentityProvider.GOOGLE = new UserPoolClientIdentityProvider('Google');
/**
 * (experimental) Allow users to sign in using 'Login With Amazon'.
 *
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 *
 * @experimental
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * (experimental) Allow users to sign in directly as a user of the User Pool.
 *
 * @experimental
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * (experimental) Define a UserPool App Client.
 *
 * @experimental
 */
class UserPoolClient extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = (_b = (_a = props.oAuth) === null || _a === void 0 ? void 0 : _a.flows) !== null && _b !== void 0 ? _b : {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = (_c = props.oAuth) === null || _c === void 0 ? void 0 : _c.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 && !props.disableOAuth ? callbackUrls : undefined,
            logoutUrLs: (_d = props.oAuth) === null || _d === void 0 ? void 0 : _d.logoutUrls,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
        });
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * (experimental) Import a user pool client given its id.
     *
     * @experimental
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) The client name that was specified via the `userPoolClientName` property during initialization, throws an error otherwise.
     *
     * @experimental
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        if (!props.authFlows)
            return undefined;
        const authFlows = [];
        if (props.authFlows.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if (props.authFlows.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        // refreshToken should always be allowed if authFlows are present
        if (authFlows.length > 0) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        var _a;
        const scopes = (_a = oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes) !== null && _a !== void 0 ? _a : [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
}
exports.UserPoolClient = UserPoolClient;
//# sourceMappingURL=data:application/json;base64,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