"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArtifactPath = exports.Artifact = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const validation = require("./private/validation");
/**
 * (experimental) An output artifact of an action.
 *
 * Artifacts can be used as input by some actions.
 *
 * @experimental
 */
class Artifact {
    /**
     * @experimental
     */
    constructor(artifactName) {
        this.metadata = {};
        validation.validateArtifactName(artifactName);
        this._artifactName = artifactName;
    }
    /**
     * (experimental) A static factory method used to create instances of the Artifact class.
     *
     * Mainly meant to be used from `decdk`.
     *
     * @param name the (required) name of the Artifact.
     * @experimental
     */
    static artifact(name) {
        return new Artifact(name);
    }
    /**
     * @experimental
     */
    get artifactName() {
        return this._artifactName;
    }
    /**
     * (experimental) Returns an ArtifactPath for a file within this artifact.
     *
     * CfnOutput is in the form "<artifact-name>::<file-name>"
     *
     * @param fileName The name of the file.
     * @experimental
     */
    atPath(fileName) {
        return new ArtifactPath(this, fileName);
    }
    /**
     * (experimental) The artifact attribute for the name of the S3 bucket where the artifact is stored.
     *
     * @experimental
     */
    get bucketName() {
        return artifactAttribute(this, 'BucketName');
    }
    /**
     * (experimental) The artifact attribute for The name of the .zip file that contains the artifact that is generated by AWS CodePipeline, such as 1ABCyZZ.zip.
     *
     * @experimental
     */
    get objectKey() {
        return artifactAttribute(this, 'ObjectKey');
    }
    /**
     * (experimental) The artifact attribute of the Amazon Simple Storage Service (Amazon S3) URL of the artifact, such as https://s3-us-west-2.amazonaws.com/artifactstorebucket-yivczw8jma0c/test/TemplateSo/1ABCyZZ.zip.
     *
     * @experimental
     */
    get url() {
        return artifactAttribute(this, 'URL');
    }
    /**
     * (experimental) Returns a token for a value inside a JSON file within this artifact.
     *
     * @param jsonFile The JSON file name.
     * @param keyName The hash key.
     * @experimental
     */
    getParam(jsonFile, keyName) {
        return artifactGetParam(this, jsonFile, keyName);
    }
    /**
     * (experimental) Returns the location of the .zip file in S3 that this Artifact represents. Used by Lambda's `CfnParametersCode` when being deployed in a CodePipeline.
     *
     * @experimental
     */
    get s3Location() {
        return {
            bucketName: this.bucketName,
            objectKey: this.objectKey,
        };
    }
    /**
     * (experimental) Add arbitrary extra payload to the artifact under a given key.
     *
     * This can be used by CodePipeline actions to communicate data between themselves.
     * If metadata was already present under the given key,
     * it will be overwritten with the new value.
     *
     * @experimental
     */
    setMetadata(key, value) {
        this.metadata[key] = value;
    }
    /**
     * (experimental) Retrieve the metadata stored in this artifact under the given key.
     *
     * If there is no metadata stored under the given key,
     * null will be returned.
     *
     * @experimental
     */
    getMetadata(key) {
        return this.metadata[key];
    }
    /**
     * @experimental
     */
    toString() {
        return this.artifactName;
    }
    /** @internal */
    _setName(name) {
        if (this._artifactName) {
            throw new Error(`Artifact already has name '${this._artifactName}', cannot override it`);
        }
        else {
            this._artifactName = name;
        }
    }
}
exports.Artifact = Artifact;
/**
 * (experimental) A specific file within an output artifact.
 *
 * The most common use case for this is specifying the template file
 * for a CloudFormation action.
 *
 * @experimental
 */
class ArtifactPath {
    /**
     * @experimental
     */
    constructor(artifact, fileName) {
        this.artifact = artifact;
        this.fileName = fileName;
    }
    /**
     * @experimental
     */
    static artifactPath(artifactName, fileName) {
        return new ArtifactPath(Artifact.artifact(artifactName), fileName);
    }
    /**
     * @experimental
     */
    get location() {
        const artifactName = this.artifact.artifactName
            ? this.artifact.artifactName
            : core_1.Lazy.stringValue({ produce: () => this.artifact.artifactName });
        return `${artifactName}::${this.fileName}`;
    }
}
exports.ArtifactPath = ArtifactPath;
function artifactAttribute(artifact, attributeName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetArtifactAtt': [lazyArtifactName, attributeName] });
}
function artifactGetParam(artifact, jsonFile, keyName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetParam': [lazyArtifactName, jsonFile, keyName] });
}
//# sourceMappingURL=data:application/json;base64,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