"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDeploymentGroup = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const constructs_1 = require("constructs");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
/**
 * @experimental
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class LambdaDeploymentGroup extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentGroupName,
        });
        this.application = props.application || new application_1.LambdaApplication(this, 'Application');
        this.alarms = props.alarms || [];
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
        });
        this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRoleForLambdaLimited'));
        this.deploymentConfig = props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES;
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            serviceRoleArn: this.role.roleArn,
            deploymentGroupName: this.physicalName,
            deploymentConfigName: this.deploymentConfig.deploymentConfigName,
            deploymentStyle: {
                deploymentType: 'BLUE_GREEN',
                deploymentOption: 'WITH_TRAFFIC_CONTROL',
            },
            alarmConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            sep: ':',
        });
        if (props.preHook) {
            this.addPreHook(props.preHook);
        }
        if (props.postHook) {
            this.addPostHook(props.postHook);
        }
        props.alias.node.defaultChild.cfnOptions.updatePolicy = {
            codeDeployLambdaAliasUpdate: {
                applicationName: this.application.applicationName,
                deploymentGroupName: resource.ref,
                beforeAllowTrafficHook: cdk.Lazy.stringValue({ produce: () => this.preHook && this.preHook.functionName }),
                afterAllowTrafficHook: cdk.Lazy.stringValue({ produce: () => this.postHook && this.postHook.functionName }),
            },
        };
        // If the deployment config is a construct, add a dependency to ensure the deployment config
        // is created before the deployment group is.
        if (this.deploymentConfig instanceof constructs_1.Construct) {
            this.node.addDependency(this.deploymentConfig);
        }
    }
    /**
     * (experimental) Import an Lambda Deployment Group defined either outside the CDK app, or in a different AWS region.
     *
     * @param scope the parent Construct for this new Construct.
     * @param id the logical ID of this new Construct.
     * @param attrs the properties of the referenced Deployment Group.
     * @returns a Construct representing a reference to an existing Deployment Group
     * @experimental
     */
    static fromLambdaDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedLambdaDeploymentGroup(scope, id, attrs);
    }
    /**
     * (experimental) Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group.
     * @experimental
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    /**
     * (experimental) Associate a function to run before deployment begins.
     *
     * @param preHook function to run before deployment beings.
     * @experimental
     * @throws an error if a pre-hook function is already configured
     */
    addPreHook(preHook) {
        if (this.preHook !== undefined) {
            throw new Error('A pre-hook function is already defined for this deployment group');
        }
        this.preHook = preHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.preHook);
        this.preHook.grantInvoke(this.role);
    }
    /**
     * (experimental) Associate a function to run after deployment completes.
     *
     * @param postHook function to run after deployment completes.
     * @experimental
     * @throws an error if a post-hook function is already configured
     */
    addPostHook(postHook) {
        if (this.postHook !== undefined) {
            throw new Error('A post-hook function is already defined for this deployment group');
        }
        this.postHook = postHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.postHook);
        this.postHook.grantInvoke(this.role);
    }
    /**
     * (experimental) Grant a principal permission to codedeploy:PutLifecycleEventHookExecutionStatus on this deployment group resource.
     *
     * @param grantee to grant permission to.
     * @experimental
     */
    grantPutLifecycleEventHookExecutionStatus(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.deploymentGroupArn],
            actions: ['codedeploy:PutLifecycleEventHookExecutionStatus'],
        });
    }
}
exports.LambdaDeploymentGroup = LambdaDeploymentGroup;
class ImportedLambdaDeploymentGroup extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
        this.deploymentConfig = props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES;
    }
}
//# sourceMappingURL=data:application/json;base64,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