"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomLambdaDeploymentConfig = exports.CustomLambdaDeploymentConfigType = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const custom_resources_1 = require("../../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
const utils_1 = require("../utils");
/**
 * (experimental) Lambda Deployment config type.
 *
 * @experimental
 */
var CustomLambdaDeploymentConfigType;
(function (CustomLambdaDeploymentConfigType) {
    CustomLambdaDeploymentConfigType["CANARY"] = "Canary";
    CustomLambdaDeploymentConfigType["LINEAR"] = "Linear";
})(CustomLambdaDeploymentConfigType = exports.CustomLambdaDeploymentConfigType || (exports.CustomLambdaDeploymentConfigType = {}));
/**
 * (experimental) A custom Deployment Configuration for a Lambda Deployment Group.
 *
 * @experimental
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class CustomLambdaDeploymentConfig extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.validateParameters(props);
        // In this section we make the argument for the AWS API call
        const deploymentType = 'TimeBased' + props.type.toString();
        const intervalMinutes = props.interval.toMinutes().toString();
        const percentage = props.percentage.toString();
        let routingConfig; // The argument to the AWS API call
        if (props.type == CustomLambdaDeploymentConfigType.CANARY) {
            routingConfig = {
                type: deploymentType,
                timeBasedCanary: {
                    canaryInterval: intervalMinutes,
                    canaryPercentage: percentage,
                },
            };
        }
        else if (props.type == CustomLambdaDeploymentConfigType.LINEAR) {
            routingConfig = {
                type: deploymentType,
                timeBasedLinear: {
                    linearInterval: intervalMinutes,
                    linearPercentage: percentage,
                },
            };
        }
        // Generates the name of the deployment config. It's also what you'll see in the AWS console
        // The name of the config is <construct unique id>.Lambda<deployment type><percentage>Percent<interval>Minutes
        // Unless the user provides an explicit name
        this.deploymentConfigName = props.deploymentConfigName !== undefined
            ? props.deploymentConfigName
            : `${core_1.Names.uniqueId(this)}.Lambda${props.type}${props.percentage}Percent${props.type === CustomLambdaDeploymentConfigType.LINEAR
                ? 'Every'
                : ''}${props.interval.toMinutes()}Minutes`;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
        // The AWS Custom Resource that calls CodeDeploy to create and delete the resource
        new custom_resources_1.AwsCustomResource(this, 'DeploymentConfig', {
            onCreate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // This `resourceName` is the initial physical ID of the config
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onUpdate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // If `resourceName` is different from the last stack update (or creation),
                // the old config gets deleted and the new one is created
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onDelete: {
                service: 'CodeDeploy',
                action: 'deleteDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                },
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
    }
    // Validate the inputs. The percentage/interval limits come from CodeDeploy
    validateParameters(props) {
        if (!(1 <= props.percentage && props.percentage <= 99)) {
            throw new Error(`Invalid deployment config percentage "${props.percentage.toString()}". \
        Step percentage must be an integer between 1 and 99.`);
        }
        if (props.interval.toMinutes() > 2880) {
            throw new Error(`Invalid deployment config interval "${props.interval.toString()}". \
        Traffic shifting intervals must be positive integers up to 2880 (2 days).`);
        }
    }
}
exports.CustomLambdaDeploymentConfig = CustomLambdaDeploymentConfig;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3VzdG9tLWRlcGxveW1lbnQtY29uZmlnLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY3VzdG9tLWRlcGxveW1lbnQtY29uZmlnLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUEwRCxDQUFDLGdEQUFnRDtBQUMzRyxnRUFBMkcsQ0FBQyw0REFBNEQ7QUFFeEssb0NBQWtEOzs7Ozs7QUFLbEQsSUFBWSxnQ0FTWDtBQVRELFdBQVksZ0NBQWdDO0lBSXhDLHFEQUFpQixDQUFBO0lBSWpCLHFEQUFpQixDQUFBO0FBQ3JCLENBQUMsRUFUVyxnQ0FBZ0MsR0FBaEMsd0NBQWdDLEtBQWhDLHdDQUFnQyxRQVMzQzs7Ozs7OztBQWdDRCxNQUFhLDRCQUE2QixTQUFRLGVBQVE7Ozs7SUFXdEQsWUFBbUIsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBd0M7UUFDckYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDL0IsNERBQTREO1FBQzVELE1BQU0sY0FBYyxHQUFHLFdBQVcsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQzNELE1BQU0sZUFBZSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDOUQsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUMvQyxJQUFJLGFBQWEsQ0FBQyxDQUFDLG1DQUFtQztRQUN0RCxJQUFJLEtBQUssQ0FBQyxJQUFJLElBQUksZ0NBQWdDLENBQUMsTUFBTSxFQUFFO1lBQ3ZELGFBQWEsR0FBRztnQkFDWixJQUFJLEVBQUUsY0FBYztnQkFDcEIsZUFBZSxFQUFFO29CQUNiLGNBQWMsRUFBRSxlQUFlO29CQUMvQixnQkFBZ0IsRUFBRSxVQUFVO2lCQUMvQjthQUNKLENBQUM7U0FDTDthQUNJLElBQUksS0FBSyxDQUFDLElBQUksSUFBSSxnQ0FBZ0MsQ0FBQyxNQUFNLEVBQUU7WUFDNUQsYUFBYSxHQUFHO2dCQUNaLElBQUksRUFBRSxjQUFjO2dCQUNwQixlQUFlLEVBQUU7b0JBQ2IsY0FBYyxFQUFFLGVBQWU7b0JBQy9CLGdCQUFnQixFQUFFLFVBQVU7aUJBQy9CO2FBQ0osQ0FBQztTQUNMO1FBQ0QsNEZBQTRGO1FBQzVGLDhHQUE4RztRQUM5Ryw0Q0FBNEM7UUFDNUMsSUFBSSxDQUFDLG9CQUFvQixHQUFHLEtBQUssQ0FBQyxvQkFBb0IsS0FBSyxTQUFTO1lBQ2hFLENBQUMsQ0FBQyxLQUFLLENBQUMsb0JBQW9CO1lBQzVCLENBQUMsQ0FBQyxHQUFHLFlBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFVBQVUsS0FBSyxDQUFDLElBQUksR0FBRyxLQUFLLENBQUMsVUFBVSxVQUFVLEtBQUssQ0FBQyxJQUFJLEtBQUssZ0NBQWdDLENBQUMsTUFBTTtnQkFDNUgsQ0FBQyxDQUFDLE9BQU87Z0JBQ1QsQ0FBQyxDQUFDLEVBQUUsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUM7UUFDbkQsSUFBSSxDQUFDLG1CQUFtQixHQUFHLDhCQUFzQixDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQzdFLGtGQUFrRjtRQUNsRixJQUFJLG9DQUFpQixDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtZQUM1QyxRQUFRLEVBQUU7Z0JBQ04sT0FBTyxFQUFFLFlBQVk7Z0JBQ3JCLE1BQU0sRUFBRSx3QkFBd0I7Z0JBQ2hDLFVBQVUsRUFBRTtvQkFDUixvQkFBb0IsRUFBRSxJQUFJLENBQUMsb0JBQW9CO29CQUMvQyxlQUFlLEVBQUUsUUFBUTtvQkFDekIsb0JBQW9CLEVBQUUsYUFBYTtpQkFDdEM7Z0JBQ0QsK0RBQStEO2dCQUMvRCxrQkFBa0IsRUFBRSxxQ0FBa0IsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDO2FBQ3ZFO1lBQ0QsUUFBUSxFQUFFO2dCQUNOLE9BQU8sRUFBRSxZQUFZO2dCQUNyQixNQUFNLEVBQUUsd0JBQXdCO2dCQUNoQyxVQUFVLEVBQUU7b0JBQ1Isb0JBQW9CLEVBQUUsSUFBSSxDQUFDLG9CQUFvQjtvQkFDL0MsZUFBZSxFQUFFLFFBQVE7b0JBQ3pCLG9CQUFvQixFQUFFLGFBQWE7aUJBQ3RDO2dCQUNELDJFQUEyRTtnQkFDM0UseURBQXlEO2dCQUN6RCxrQkFBa0IsRUFBRSxxQ0FBa0IsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDO2FBQ3ZFO1lBQ0QsUUFBUSxFQUFFO2dCQUNOLE9BQU8sRUFBRSxZQUFZO2dCQUNyQixNQUFNLEVBQUUsd0JBQXdCO2dCQUNoQyxVQUFVLEVBQUU7b0JBQ1Isb0JBQW9CLEVBQUUsSUFBSSxDQUFDLG9CQUFvQjtpQkFDbEQ7YUFDSjtZQUNELE1BQU0sRUFBRSwwQ0FBdUIsQ0FBQyxZQUFZLENBQUM7Z0JBQ3pDLFNBQVMsRUFBRSwwQ0FBdUIsQ0FBQyxZQUFZO2FBQ2xELENBQUM7U0FDTCxDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0QsMkVBQTJFO0lBQ25FLGtCQUFrQixDQUFDLEtBQXdDO1FBQy9ELElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxLQUFLLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQyxVQUFVLElBQUksRUFBRSxDQUFDLEVBQUU7WUFDcEQsTUFBTSxJQUFJLEtBQUssQ0FBQyx5Q0FBeUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUU7NkRBQ25DLENBQUMsQ0FBQztTQUN0RDtRQUNELElBQUksS0FBSyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsR0FBRyxJQUFJLEVBQUU7WUFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyx1Q0FBdUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUU7a0ZBQ1YsQ0FBQyxDQUFDO1NBQzNFO0lBQ0wsQ0FBQztDQUNKO0FBOUZELG9FQThGQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IER1cmF0aW9uLCBOYW1lcywgUmVzb3VyY2UgfSBmcm9tIFwiLi4vLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IEF3c0N1c3RvbVJlc291cmNlLCBBd3NDdXN0b21SZXNvdXJjZVBvbGljeSwgUGh5c2ljYWxSZXNvdXJjZUlkIH0gZnJvbSBcIi4uLy4uLy4uL2N1c3RvbS1yZXNvdXJjZXNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2N1c3RvbS1yZXNvdXJjZXMnXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IGFybkZvckRlcGxveW1lbnRDb25maWcgfSBmcm9tICcuLi91dGlscyc7XG5pbXBvcnQgeyBJTGFtYmRhRGVwbG95bWVudENvbmZpZyB9IGZyb20gJy4vZGVwbG95bWVudC1jb25maWcnO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gQ3VzdG9tTGFtYmRhRGVwbG95bWVudENvbmZpZ1R5cGUge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgQ0FOQVJZID0gJ0NhbmFyeScsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBMSU5FQVIgPSAnTGluZWFyJ1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBDdXN0b21MYW1iZGFEZXBsb3ltZW50Q29uZmlnUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSB0eXBlOiBDdXN0b21MYW1iZGFEZXBsb3ltZW50Q29uZmlnVHlwZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHBlcmNlbnRhZ2U6IG51bWJlcjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGludGVydmFsOiBEdXJhdGlvbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGRlcGxveW1lbnRDb25maWdOYW1lPzogc3RyaW5nO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEN1c3RvbUxhbWJkYURlcGxveW1lbnRDb25maWcgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElMYW1iZGFEZXBsb3ltZW50Q29uZmlnIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHJlYWRvbmx5IGRlcGxveW1lbnRDb25maWdOYW1lOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHJlYWRvbmx5IGRlcGxveW1lbnRDb25maWdBcm46IHN0cmluZztcbiAgICBwdWJsaWMgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEN1c3RvbUxhbWJkYURlcGxveW1lbnRDb25maWdQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgICAgICB0aGlzLnZhbGlkYXRlUGFyYW1ldGVycyhwcm9wcyk7XG4gICAgICAgIC8vIEluIHRoaXMgc2VjdGlvbiB3ZSBtYWtlIHRoZSBhcmd1bWVudCBmb3IgdGhlIEFXUyBBUEkgY2FsbFxuICAgICAgICBjb25zdCBkZXBsb3ltZW50VHlwZSA9ICdUaW1lQmFzZWQnICsgcHJvcHMudHlwZS50b1N0cmluZygpO1xuICAgICAgICBjb25zdCBpbnRlcnZhbE1pbnV0ZXMgPSBwcm9wcy5pbnRlcnZhbC50b01pbnV0ZXMoKS50b1N0cmluZygpO1xuICAgICAgICBjb25zdCBwZXJjZW50YWdlID0gcHJvcHMucGVyY2VudGFnZS50b1N0cmluZygpO1xuICAgICAgICBsZXQgcm91dGluZ0NvbmZpZzsgLy8gVGhlIGFyZ3VtZW50IHRvIHRoZSBBV1MgQVBJIGNhbGxcbiAgICAgICAgaWYgKHByb3BzLnR5cGUgPT0gQ3VzdG9tTGFtYmRhRGVwbG95bWVudENvbmZpZ1R5cGUuQ0FOQVJZKSB7XG4gICAgICAgICAgICByb3V0aW5nQ29uZmlnID0ge1xuICAgICAgICAgICAgICAgIHR5cGU6IGRlcGxveW1lbnRUeXBlLFxuICAgICAgICAgICAgICAgIHRpbWVCYXNlZENhbmFyeToge1xuICAgICAgICAgICAgICAgICAgICBjYW5hcnlJbnRlcnZhbDogaW50ZXJ2YWxNaW51dGVzLFxuICAgICAgICAgICAgICAgICAgICBjYW5hcnlQZXJjZW50YWdlOiBwZXJjZW50YWdlLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9O1xuICAgICAgICB9XG4gICAgICAgIGVsc2UgaWYgKHByb3BzLnR5cGUgPT0gQ3VzdG9tTGFtYmRhRGVwbG95bWVudENvbmZpZ1R5cGUuTElORUFSKSB7XG4gICAgICAgICAgICByb3V0aW5nQ29uZmlnID0ge1xuICAgICAgICAgICAgICAgIHR5cGU6IGRlcGxveW1lbnRUeXBlLFxuICAgICAgICAgICAgICAgIHRpbWVCYXNlZExpbmVhcjoge1xuICAgICAgICAgICAgICAgICAgICBsaW5lYXJJbnRlcnZhbDogaW50ZXJ2YWxNaW51dGVzLFxuICAgICAgICAgICAgICAgICAgICBsaW5lYXJQZXJjZW50YWdlOiBwZXJjZW50YWdlLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9O1xuICAgICAgICB9XG4gICAgICAgIC8vIEdlbmVyYXRlcyB0aGUgbmFtZSBvZiB0aGUgZGVwbG95bWVudCBjb25maWcuIEl0J3MgYWxzbyB3aGF0IHlvdSdsbCBzZWUgaW4gdGhlIEFXUyBjb25zb2xlXG4gICAgICAgIC8vIFRoZSBuYW1lIG9mIHRoZSBjb25maWcgaXMgPGNvbnN0cnVjdCB1bmlxdWUgaWQ+LkxhbWJkYTxkZXBsb3ltZW50IHR5cGU+PHBlcmNlbnRhZ2U+UGVyY2VudDxpbnRlcnZhbD5NaW51dGVzXG4gICAgICAgIC8vIFVubGVzcyB0aGUgdXNlciBwcm92aWRlcyBhbiBleHBsaWNpdCBuYW1lXG4gICAgICAgIHRoaXMuZGVwbG95bWVudENvbmZpZ05hbWUgPSBwcm9wcy5kZXBsb3ltZW50Q29uZmlnTmFtZSAhPT0gdW5kZWZpbmVkXG4gICAgICAgICAgICA/IHByb3BzLmRlcGxveW1lbnRDb25maWdOYW1lXG4gICAgICAgICAgICA6IGAke05hbWVzLnVuaXF1ZUlkKHRoaXMpfS5MYW1iZGEke3Byb3BzLnR5cGV9JHtwcm9wcy5wZXJjZW50YWdlfVBlcmNlbnQke3Byb3BzLnR5cGUgPT09IEN1c3RvbUxhbWJkYURlcGxveW1lbnRDb25maWdUeXBlLkxJTkVBUlxuICAgICAgICAgICAgICAgID8gJ0V2ZXJ5J1xuICAgICAgICAgICAgICAgIDogJyd9JHtwcm9wcy5pbnRlcnZhbC50b01pbnV0ZXMoKX1NaW51dGVzYDtcbiAgICAgICAgdGhpcy5kZXBsb3ltZW50Q29uZmlnQXJuID0gYXJuRm9yRGVwbG95bWVudENvbmZpZyh0aGlzLmRlcGxveW1lbnRDb25maWdOYW1lKTtcbiAgICAgICAgLy8gVGhlIEFXUyBDdXN0b20gUmVzb3VyY2UgdGhhdCBjYWxscyBDb2RlRGVwbG95IHRvIGNyZWF0ZSBhbmQgZGVsZXRlIHRoZSByZXNvdXJjZVxuICAgICAgICBuZXcgQXdzQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0RlcGxveW1lbnRDb25maWcnLCB7XG4gICAgICAgICAgICBvbkNyZWF0ZToge1xuICAgICAgICAgICAgICAgIHNlcnZpY2U6ICdDb2RlRGVwbG95JyxcbiAgICAgICAgICAgICAgICBhY3Rpb246ICdjcmVhdGVEZXBsb3ltZW50Q29uZmlnJyxcbiAgICAgICAgICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICAgICAgICAgIGRlcGxveW1lbnRDb25maWdOYW1lOiB0aGlzLmRlcGxveW1lbnRDb25maWdOYW1lLFxuICAgICAgICAgICAgICAgICAgICBjb21wdXRlUGxhdGZvcm06ICdMYW1iZGEnLFxuICAgICAgICAgICAgICAgICAgICB0cmFmZmljUm91dGluZ0NvbmZpZzogcm91dGluZ0NvbmZpZyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIC8vIFRoaXMgYHJlc291cmNlTmFtZWAgaXMgdGhlIGluaXRpYWwgcGh5c2ljYWwgSUQgb2YgdGhlIGNvbmZpZ1xuICAgICAgICAgICAgICAgIHBoeXNpY2FsUmVzb3VyY2VJZDogUGh5c2ljYWxSZXNvdXJjZUlkLm9mKHRoaXMuZGVwbG95bWVudENvbmZpZ05hbWUpLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIG9uVXBkYXRlOiB7XG4gICAgICAgICAgICAgICAgc2VydmljZTogJ0NvZGVEZXBsb3knLFxuICAgICAgICAgICAgICAgIGFjdGlvbjogJ2NyZWF0ZURlcGxveW1lbnRDb25maWcnLFxuICAgICAgICAgICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgICAgICAgICAgZGVwbG95bWVudENvbmZpZ05hbWU6IHRoaXMuZGVwbG95bWVudENvbmZpZ05hbWUsXG4gICAgICAgICAgICAgICAgICAgIGNvbXB1dGVQbGF0Zm9ybTogJ0xhbWJkYScsXG4gICAgICAgICAgICAgICAgICAgIHRyYWZmaWNSb3V0aW5nQ29uZmlnOiByb3V0aW5nQ29uZmlnLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgLy8gSWYgYHJlc291cmNlTmFtZWAgaXMgZGlmZmVyZW50IGZyb20gdGhlIGxhc3Qgc3RhY2sgdXBkYXRlIChvciBjcmVhdGlvbiksXG4gICAgICAgICAgICAgICAgLy8gdGhlIG9sZCBjb25maWcgZ2V0cyBkZWxldGVkIGFuZCB0aGUgbmV3IG9uZSBpcyBjcmVhdGVkXG4gICAgICAgICAgICAgICAgcGh5c2ljYWxSZXNvdXJjZUlkOiBQaHlzaWNhbFJlc291cmNlSWQub2YodGhpcy5kZXBsb3ltZW50Q29uZmlnTmFtZSksXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgb25EZWxldGU6IHtcbiAgICAgICAgICAgICAgICBzZXJ2aWNlOiAnQ29kZURlcGxveScsXG4gICAgICAgICAgICAgICAgYWN0aW9uOiAnZGVsZXRlRGVwbG95bWVudENvbmZpZycsXG4gICAgICAgICAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgICAgICAgICAgICBkZXBsb3ltZW50Q29uZmlnTmFtZTogdGhpcy5kZXBsb3ltZW50Q29uZmlnTmFtZSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHBvbGljeTogQXdzQ3VzdG9tUmVzb3VyY2VQb2xpY3kuZnJvbVNka0NhbGxzKHtcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IEF3c0N1c3RvbVJlc291cmNlUG9saWN5LkFOWV9SRVNPVVJDRSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLy8gVmFsaWRhdGUgdGhlIGlucHV0cy4gVGhlIHBlcmNlbnRhZ2UvaW50ZXJ2YWwgbGltaXRzIGNvbWUgZnJvbSBDb2RlRGVwbG95XG4gICAgcHJpdmF0ZSB2YWxpZGF0ZVBhcmFtZXRlcnMocHJvcHM6IEN1c3RvbUxhbWJkYURlcGxveW1lbnRDb25maWdQcm9wcyk6IHZvaWQge1xuICAgICAgICBpZiAoISgxIDw9IHByb3BzLnBlcmNlbnRhZ2UgJiYgcHJvcHMucGVyY2VudGFnZSA8PSA5OSkpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBkZXBsb3ltZW50IGNvbmZpZyBwZXJjZW50YWdlIFwiJHtwcm9wcy5wZXJjZW50YWdlLnRvU3RyaW5nKCl9XCIuIFxcXG4gICAgICAgIFN0ZXAgcGVyY2VudGFnZSBtdXN0IGJlIGFuIGludGVnZXIgYmV0d2VlbiAxIGFuZCA5OS5gKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocHJvcHMuaW50ZXJ2YWwudG9NaW51dGVzKCkgPiAyODgwKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgZGVwbG95bWVudCBjb25maWcgaW50ZXJ2YWwgXCIke3Byb3BzLmludGVydmFsLnRvU3RyaW5nKCl9XCIuIFxcXG4gICAgICAgIFRyYWZmaWMgc2hpZnRpbmcgaW50ZXJ2YWxzIG11c3QgYmUgcG9zaXRpdmUgaW50ZWdlcnMgdXAgdG8gMjg4MCAoMiBkYXlzKS5gKTtcbiAgICAgICAgfVxuICAgIH1cbn1cbiJdfQ==