"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FilterGroup = exports.EventAction = exports.Source = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const source_types_1 = require("./source-types");
/**
 * (experimental) Source provider definition for a CodeBuild Project.
 *
 * @experimental
 */
class Source {
    /**
     * @experimental
     */
    constructor(props) {
        /**
         * @experimental
         */
        this.badgeSupported = false;
        this.identifier = props.identifier;
    }
    /**
     * @experimental
     */
    static s3(props) {
        return new S3Source(props);
    }
    /**
     * @experimental
     */
    static codeCommit(props) {
        return new CodeCommitSource(props);
    }
    /**
     * @experimental
     */
    static gitHub(props) {
        return new GitHubSource(props);
    }
    /**
     * @experimental
     */
    static gitHubEnterprise(props) {
        return new GitHubEnterpriseSource(props);
    }
    /**
     * @experimental
     */
    static bitBucket(props) {
        return new BitBucketSource(props);
    }
    /**
     * (experimental) Called by the project when the source is added so that the source can perform binding operations on the source.
     *
     * For example, it can grant permissions to the
     * code build project to read from the S3 bucket.
     *
     * @experimental
     */
    bind(_scope, _project) {
        return {
            sourceProperty: {
                sourceIdentifier: this.identifier,
                type: this.type,
            },
        };
    }
}
exports.Source = Source;
/**
 * A common superclass of all build sources that are backed by Git.
 */
class GitSource extends Source {
    constructor(props) {
        super(props);
        this.cloneDepth = props.cloneDepth;
        this.branchOrRef = props.branchOrRef;
        this.fetchSubmodules = props.fetchSubmodules;
    }
    bind(_scope, _project) {
        const superConfig = super.bind(_scope, _project);
        return {
            sourceVersion: this.branchOrRef,
            sourceProperty: {
                ...superConfig.sourceProperty,
                gitCloneDepth: this.cloneDepth,
                gitSubmodulesConfig: this.fetchSubmodules ? {
                    fetchSubmodules: this.fetchSubmodules,
                } : undefined,
            },
        };
    }
}
/**
 * (experimental) The types of webhook event actions.
 *
 * @experimental
 */
var EventAction;
(function (EventAction) {
    EventAction["PUSH"] = "PUSH";
    EventAction["PULL_REQUEST_CREATED"] = "PULL_REQUEST_CREATED";
    EventAction["PULL_REQUEST_UPDATED"] = "PULL_REQUEST_UPDATED";
    EventAction["PULL_REQUEST_MERGED"] = "PULL_REQUEST_MERGED";
    EventAction["PULL_REQUEST_REOPENED"] = "PULL_REQUEST_REOPENED";
})(EventAction = exports.EventAction || (exports.EventAction = {}));
var WebhookFilterTypes;
(function (WebhookFilterTypes) {
    WebhookFilterTypes["FILE_PATH"] = "FILE_PATH";
    WebhookFilterTypes["COMMIT_MESSAGE"] = "COMMIT_MESSAGE";
    WebhookFilterTypes["HEAD_REF"] = "HEAD_REF";
    WebhookFilterTypes["ACTOR_ACCOUNT_ID"] = "ACTOR_ACCOUNT_ID";
    WebhookFilterTypes["BASE_REF"] = "BASE_REF";
})(WebhookFilterTypes || (WebhookFilterTypes = {}));
/**
 * (experimental) An object that represents a group of filter conditions for a webhook.
 *
 * Every condition in a given FilterGroup must be true in order for the whole group to be true.
 * You construct instances of it by calling the {@link #inEventOf} static factory method,
 * and then calling various `andXyz` instance methods to create modified instances of it
 * (this class is immutable).
 *
 * You pass instances of this class to the `webhookFilters` property when constructing a source.
 *
 * @experimental
 */
class FilterGroup {
    constructor(actions, filters) {
        if (actions.size === 0) {
            throw new Error('A filter group must contain at least one event action');
        }
        this.actions = actions;
        this.filters = filters;
    }
    /**
     * (experimental) Creates a new event FilterGroup that triggers on any of the provided actions.
     *
     * @param actions the actions to trigger the webhook on.
     * @experimental
     */
    static inEventOf(...actions) {
        return new FilterGroup(new Set(actions), []);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the event must affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression).
     * @experimental
     */
    andBranchIs(branchName) {
        return this.addHeadBranchFilter(branchName, true);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the event must not affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression).
     * @experimental
     */
    andBranchIsNot(branchName) {
        return this.addHeadBranchFilter(branchName, false);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the event must affect a head commit with the given message.
     *
     * @param commitMessage the commit message (can be a regular expression).
     * @experimental
     */
    andCommitMessageIs(commitMessage) {
        return this.addCommitMessageFilter(commitMessage, true);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the event must not affect a head commit with the given message.
     *
     * @param commitMessage the commit message (can be a regular expression).
     * @experimental
     */
    andCommitMessageIsNot(commitMessage) {
        return this.addCommitMessageFilter(commitMessage, false);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the event must affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression).
     * @experimental
     */
    andTagIs(tagName) {
        return this.addHeadTagFilter(tagName, true);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the event must not affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression).
     * @experimental
     */
    andTagIsNot(tagName) {
        return this.addHeadTagFilter(tagName, false);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the event must affect a Git reference (ie., a branch or a tag) that matches the given pattern.
     *
     * @param pattern a regular expression.
     * @experimental
     */
    andHeadRefIs(pattern) {
        return this.addHeadRefFilter(pattern, true);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the event must not affect a Git reference (ie., a branch or a tag) that matches the given pattern.
     *
     * @param pattern a regular expression.
     * @experimental
     */
    andHeadRefIsNot(pattern) {
        return this.addHeadRefFilter(pattern, false);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the account ID of the actor initiating the event must match the given pattern.
     *
     * @param pattern a regular expression.
     * @experimental
     */
    andActorAccountIs(pattern) {
        return this.addActorAccountId(pattern, true);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the account ID of the actor initiating the event must not match the given pattern.
     *
     * @param pattern a regular expression.
     * @experimental
     */
    andActorAccountIsNot(pattern) {
        return this.addActorAccountId(pattern, false);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the Pull Request that is the source of the event must target the given base branch.
     *
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression).
     * @experimental
     */
    andBaseBranchIs(branchName) {
        return this.addBaseBranchFilter(branchName, true);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the Pull Request that is the source of the event must not target the given base branch.
     *
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression).
     * @experimental
     */
    andBaseBranchIsNot(branchName) {
        return this.addBaseBranchFilter(branchName, false);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the Pull Request that is the source of the event must target the given Git reference.
     *
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression.
     * @experimental
     */
    andBaseRefIs(pattern) {
        return this.addBaseRefFilter(pattern, true);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the Pull Request that is the source of the event must not target the given Git reference.
     *
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression.
     * @experimental
     */
    andBaseRefIsNot(pattern) {
        return this.addBaseRefFilter(pattern, false);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the push that is the source of the event must affect a file that matches the given pattern.
     *
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub and GitHubEnterprise sources.
     *
     * @param pattern a regular expression.
     * @experimental
     */
    andFilePathIs(pattern) {
        return this.addFilePathFilter(pattern, true);
    }
    /**
     * (experimental) Create a new FilterGroup with an added condition: the push that is the source of the event must not affect a file that matches the given pattern.
     *
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub and GitHubEnterprise sources.
     *
     * @param pattern a regular expression.
     * @experimental
     */
    andFilePathIsNot(pattern) {
        return this.addFilePathFilter(pattern, false);
    }
    /** @internal */
    get _actions() {
        return set2Array(this.actions);
    }
    /** @internal */
    get _filters() {
        return this.filters.slice();
    }
    /** @internal */
    _toJson() {
        const eventFilter = {
            type: 'EVENT',
            pattern: set2Array(this.actions).join(', '),
        };
        return [eventFilter].concat(this.filters);
    }
    addCommitMessageFilter(commitMessage, include) {
        return this.addFilter(WebhookFilterTypes.COMMIT_MESSAGE, commitMessage, include);
    }
    addHeadBranchFilter(branchName, include) {
        return this.addHeadRefFilter(`refs/heads/${branchName}`, include);
    }
    addHeadTagFilter(tagName, include) {
        return this.addHeadRefFilter(`refs/tags/${tagName}`, include);
    }
    addHeadRefFilter(refName, include) {
        return this.addFilter(WebhookFilterTypes.HEAD_REF, refName, include);
    }
    addActorAccountId(accountId, include) {
        return this.addFilter(WebhookFilterTypes.ACTOR_ACCOUNT_ID, accountId, include);
    }
    addBaseBranchFilter(branchName, include) {
        return this.addBaseRefFilter(`refs/heads/${branchName}`, include);
    }
    addBaseRefFilter(refName, include) {
        if (this.actions.has(EventAction.PUSH)) {
            throw new Error('A base reference condition cannot be added if a Group contains a PUSH event action');
        }
        return this.addFilter(WebhookFilterTypes.BASE_REF, refName, include);
    }
    addFilePathFilter(pattern, include) {
        return this.addFilter(WebhookFilterTypes.FILE_PATH, pattern, include);
    }
    addFilter(type, pattern, include) {
        return new FilterGroup(this.actions, this.filters.concat([{
                type,
                pattern,
                excludeMatchedPattern: include ? undefined : true,
            }]));
    }
}
exports.FilterGroup = FilterGroup;
/**
 * A common superclass of all third-party build sources that are backed by Git.
 */
class ThirdPartyGitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.webhook = props.webhook;
        this.reportBuildStatus = props.reportBuildStatus === undefined ? true : props.reportBuildStatus;
        this.webhookFilters = props.webhookFilters || [];
    }
    bind(_scope, _project) {
        const anyFilterGroupsProvided = this.webhookFilters.length > 0;
        const webhook = this.webhook === undefined ? (anyFilterGroupsProvided ? true : undefined) : this.webhook;
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                reportBuildStatus: this.reportBuildStatus,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: webhook === undefined ? undefined : {
                webhook,
                filterGroups: anyFilterGroupsProvided ? this.webhookFilters.map(fg => fg._toJson()) : undefined,
            },
        };
    }
}
/**
 * CodeCommit Source definition for a CodeBuild project.
 */
class CodeCommitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.type = source_types_1.CODECOMMIT_SOURCE_TYPE;
        this.repo = props.repository;
    }
    bind(_scope, project) {
        // https://docs.aws.amazon.com/codebuild/latest/userguide/setting-up.html
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['codecommit:GitPull'],
            resources: [this.repo.repositoryArn],
        }));
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.repo.repositoryCloneUrlHttp,
            },
            sourceVersion: superConfig.sourceVersion,
        };
    }
}
/**
 * S3 bucket definition for a CodeBuild project.
 */
class S3Source extends Source {
    constructor(props) {
        super(props);
        this.type = source_types_1.S3_SOURCE_TYPE;
        this.bucket = props.bucket;
        this.path = props.path;
        this.version = props.version;
    }
    bind(_scope, project) {
        this.bucket.grantRead(project);
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: `${this.bucket.bucketName}/${this.path}`,
            },
            sourceVersion: this.version,
        };
    }
}
/**
 * GitHub Source definition for a CodeBuild project.
 */
class GitHubSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_SOURCE_TYPE;
        this.httpsCloneUrl = `https://github.com/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, project) {
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * GitHub Enterprise Source definition for a CodeBuild project.
 */
class GitHubEnterpriseSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_ENTERPRISE_SOURCE_TYPE;
        this.httpsCloneUrl = props.httpsCloneUrl;
        this.ignoreSslErrors = props.ignoreSslErrors;
    }
    bind(_scope, _project) {
        if (this.hasCommitMessageFilterAndPrEvent()) {
            throw new Error('COMMIT_MESSAGE filters cannot be used with GitHub Enterprise Server pull request events');
        }
        if (this.hasFilePathFilterAndPrEvent()) {
            throw new Error('FILE_PATH filters cannot be used with GitHub Enterprise Server pull request events');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
                insecureSsl: this.ignoreSslErrors,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    hasCommitMessageFilterAndPrEvent() {
        return this.webhookFilters.some(fg => (fg._filters.some(fp => fp.type === WebhookFilterTypes.COMMIT_MESSAGE) &&
            this.hasPrEvent(fg._actions)));
    }
    hasFilePathFilterAndPrEvent() {
        return this.webhookFilters.some(fg => (fg._filters.some(fp => fp.type === WebhookFilterTypes.FILE_PATH) &&
            this.hasPrEvent(fg._actions)));
    }
    hasPrEvent(actions) {
        return actions.includes(EventAction.PULL_REQUEST_CREATED ||
            EventAction.PULL_REQUEST_MERGED ||
            EventAction.PULL_REQUEST_REOPENED ||
            EventAction.PULL_REQUEST_UPDATED);
    }
}
/**
 * BitBucket Source definition for a CodeBuild project.
 */
class BitBucketSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.BITBUCKET_SOURCE_TYPE;
        this.httpsCloneUrl = `https://bitbucket.org/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, _project) {
        // BitBucket sources don't support the PULL_REQUEST_REOPENED event action
        if (this.anyWebhookFilterContainsPrReopenedEventAction()) {
            throw new Error('BitBucket sources do not support the PULL_REQUEST_REOPENED webhook event action');
        }
        // they also don't support file path conditions
        if (this.anyWebhookFilterContainsFilePathConditions()) {
            throw new Error('BitBucket sources do not support file path conditions for webhook filters');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    anyWebhookFilterContainsPrReopenedEventAction() {
        return this.webhookFilters.findIndex(fg => {
            return fg._actions.findIndex(a => a === EventAction.PULL_REQUEST_REOPENED) !== -1;
        }) !== -1;
    }
    anyWebhookFilterContainsFilePathConditions() {
        return this.webhookFilters.findIndex(fg => {
            return fg._filters.findIndex(f => f.type === WebhookFilterTypes.FILE_PATH) !== -1;
        }) !== -1;
    }
}
function set2Array(set) {
    const ret = [];
    set.forEach(el => ret.push(el));
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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