"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildEnvironmentVariableType = exports.WindowsBuildImage = exports.WindowsImageType = exports.LinuxBuildImage = exports.ImagePullPrincipalType = exports.ComputeType = exports.Project = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const aws_ecr_assets_1 = require("../../aws-ecr-assets"); // Automatically re-written from '@aws-cdk/aws-ecr-assets'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const build_spec_1 = require("./build-spec");
const cache_1 = require("./cache");
const codebuild_generated_1 = require("./codebuild.generated");
const codepipeline_artifacts_1 = require("./codepipeline-artifacts");
const no_artifacts_1 = require("./no-artifacts");
const no_source_1 = require("./no-source");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const report_group_utils_1 = require("./report-group-utils");
const source_types_1 = require("./source-types");
/**
 * Represents a reference to a CodeBuild Project.
 *
 * If you're managing the Project alongside the rest of your CDK resources,
 * use the {@link Project} class.
 *
 * If you want to reference an already existing Project
 * (or one defined in a different CDK Stack),
 * use the {@link import} method.
 */
class ProjectBase extends core_1.Resource {
    /**
     * (experimental) Access the Connections object.
     *
     * Will fail if this Project does not have a VPC set.
     *
     * @experimental
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Only VPC-associated Projects have security groups to manage. Supply the "vpc" parameter when creating the Project');
        }
        return this._connections;
    }
    /**
     * (experimental) Add a permission only if there's a policy attached.
     *
     * @param statement The permissions statement to add.
     * @experimental
     */
    addToRolePolicy(statement) {
        if (this.role) {
            this.role.addToPolicy(statement);
        }
    }
    /**
     * (experimental) Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @experimental
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codebuild'],
            detail: {
                'project-name': [this.projectName],
            },
        });
        return rule;
    }
    /**
     * (experimental) Defines a CloudWatch event rule triggered when the build project state changes.
     *
     * You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *     const rule = project.onStateChange('OnBuildStarted', { target });
     *     rule.addEventPattern({
     *       detail: {
     *         'build-status': [
     *           "IN_PROGRESS",
     *           "SUCCEEDED",
     *           "FAILED",
     *           "STOPPED"
     *         ]
     *       }
     *     });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @experimental
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build State Change'],
        });
        return rule;
    }
    /**
     * (experimental) Defines a CloudWatch event rule that triggers upon phase change of this build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @experimental
     */
    onPhaseChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build Phase Change'],
        });
        return rule;
    }
    /**
     * (experimental) Defines an event rule which triggers when a build starts.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @experimental
     */
    onBuildStarted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['IN_PROGRESS'],
            },
        });
        return rule;
    }
    /**
     * (experimental) Defines an event rule which triggers when a build fails.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @experimental
     */
    onBuildFailed(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['FAILED'],
            },
        });
        return rule;
    }
    /**
     * (experimental) Defines an event rule which triggers when a build completes successfully.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @experimental
     */
    onBuildSucceeded(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['SUCCEEDED'],
            },
        });
        return rule;
    }
    /**
     * @param metricName The name of the metric.
     * @param props Customization properties.
     * @returns a CloudWatch metric associated with this build project.
     * @experimental
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/CodeBuild',
            metricName,
            dimensions: { ProjectName: this.projectName },
            ...props,
        }).attachTo(this);
    }
    /**
     * (experimental) Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @experimental
     */
    metricBuilds(props) {
        return this.metric('Builds', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * (experimental) Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     * @experimental
     */
    metricDuration(props) {
        return this.metric('Duration', {
            statistic: 'avg',
            ...props,
        });
    }
    /**
     * (experimental) Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @experimental
     */
    metricSucceededBuilds(props) {
        return this.metric('SucceededBuilds', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * (experimental) Measures the number of builds that failed because of client error or because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @experimental
     */
    metricFailedBuilds(props) {
        return this.metric('FailedBuilds', {
            statistic: 'sum',
            ...props,
        });
    }
}
/**
 * (experimental) A representation of a CodeBuild Project.
 *
 * @experimental
 */
class Project extends ProjectBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.projectName,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        this.buildImage = (props.environment && props.environment.buildImage) || LinuxBuildImage.STANDARD_1_0;
        // let source "bind" to the project. this usually involves granting permissions
        // for the code build role to interact with the source.
        this.source = props.source || new no_source_1.NoSource();
        const sourceConfig = this.source.bind(this, this);
        if (props.badge && !this.source.badgeSupported) {
            throw new Error(`Badge is not supported for source type ${this.source.type}`);
        }
        const artifacts = props.artifacts
            ? props.artifacts
            : (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE
                ? new codepipeline_artifacts_1.CodePipelineArtifacts()
                : new no_artifacts_1.NoArtifacts());
        const artifactsConfig = artifacts.bind(this, this);
        const cache = props.cache || cache_1.Cache.none();
        // give the caching strategy the option to grant permissions to any required resources
        cache._bind(this);
        // Inject download commands for asset if requested
        const environmentVariables = props.environmentVariables || {};
        const buildSpec = props.buildSpec;
        if (this.source.type === source_types_1.NO_SOURCE_TYPE && (buildSpec === undefined || !buildSpec.isImmediate)) {
            throw new Error("If the Project's source is NoSource, you need to provide a concrete buildSpec");
        }
        this._secondarySources = [];
        this._secondarySourceVersions = [];
        this._fileSystemLocations = [];
        for (const secondarySource of props.secondarySources || []) {
            this.addSecondarySource(secondarySource);
        }
        this._secondaryArtifacts = [];
        for (const secondaryArtifact of props.secondaryArtifacts || []) {
            this.addSecondaryArtifact(secondaryArtifact);
        }
        this.validateCodePipelineSettings(artifacts);
        for (const fileSystemLocation of props.fileSystemLocations || []) {
            this.addFileSystemLocation(fileSystemLocation);
        }
        const resource = new codebuild_generated_1.CfnProject(this, 'Resource', {
            description: props.description,
            source: {
                ...sourceConfig.sourceProperty,
                buildSpec: buildSpec && buildSpec.toBuildSpec(),
            },
            artifacts: artifactsConfig.artifactsProperty,
            serviceRole: this.role.roleArn,
            environment: this.renderEnvironment(props.environment, environmentVariables),
            fileSystemLocations: core_1.Lazy.anyValue({ produce: () => this.renderFileSystemLocations() }),
            // lazy, because we have a setter for it in setEncryptionKey
            // The 'alias/aws/s3' default is necessary because leaving the `encryptionKey` field
            // empty will not remove existing encryptionKeys during an update (ref. t/D17810523)
            encryptionKey: core_1.Lazy.stringValue({ produce: () => this._encryptionKey ? this._encryptionKey.keyArn : 'alias/aws/s3' }),
            badgeEnabled: props.badge,
            cache: cache._toCloudFormation(),
            name: this.physicalName,
            timeoutInMinutes: props.timeout && props.timeout.toMinutes(),
            secondarySources: core_1.Lazy.anyValue({ produce: () => this.renderSecondarySources() }),
            secondarySourceVersions: core_1.Lazy.anyValue({ produce: () => this.renderSecondarySourceVersions() }),
            secondaryArtifacts: core_1.Lazy.anyValue({ produce: () => this.renderSecondaryArtifacts() }),
            triggers: sourceConfig.buildTriggers,
            sourceVersion: sourceConfig.sourceVersion,
            vpcConfig: this.configureVpc(props),
        });
        this.addVpcRequiredPermissions(props, resource);
        this.projectArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'codebuild',
            resource: 'project',
            resourceName: this.physicalName,
        });
        this.projectName = this.getResourceNameAttribute(resource.ref);
        this.addToRolePolicy(this.createLoggingPermission());
        // add permissions to create and use test report groups
        // with names starting with the project's name,
        // unless the customer explicitly opts out of it
        if (props.grantReportGroupPermissions !== false) {
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'codebuild:CreateReportGroup',
                    'codebuild:CreateReport',
                    'codebuild:UpdateReport',
                    'codebuild:BatchPutTestCases',
                    'codebuild:BatchPutCodeCoverages',
                ],
                resources: [report_group_utils_1.renderReportGroupArn(this, `${this.projectName}-*`)],
            }));
        }
        if (props.encryptionKey) {
            this.encryptionKey = props.encryptionKey;
        }
        // bind
        const bindFunction = this.buildImage.bind;
        if (bindFunction) {
            bindFunction.call(this.buildImage, this, this, {});
        }
    }
    /**
     * @experimental
     */
    static fromProjectArn(scope, id, projectArn) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.projectArn = projectArn;
                this.projectName = core_1.Stack.of(scope).parseArn(projectArn).resourceName;
                this.role = undefined;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Import a Project defined either outside the CDK, or in a different CDK Stack (and exported using the {@link export} method).
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param projectName the name of the project to import.
     * @returns a reference to the existing Project
     * @experimental
     * @note if you're importing a CodeBuild Project for use
     * in a CodePipeline, make sure the existing Project
     * has permissions to access the S3 Bucket of that Pipeline -
     * otherwise, builds in that Pipeline will always fail.
     */
    static fromProjectName(scope, id, projectName) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.role = undefined;
                this.projectArn = core_1.Stack.of(this).formatArn({
                    service: 'codebuild',
                    resource: 'project',
                    resourceName: projectName,
                });
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
                this.projectName = projectName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Convert the environment variables map of string to {@link BuildEnvironmentVariable}, which is the customer-facing type, to a list of {@link CfnProject.EnvironmentVariableProperty}, which is the representation of environment variables in CloudFormation.
     *
     * @param environmentVariables the map of string to environment variables.
     * @returns an array of {@link CfnProject.EnvironmentVariableProperty} instances
     * @experimental
     */
    static serializeEnvVariables(environmentVariables) {
        return Object.keys(environmentVariables).map(name => ({
            name,
            type: environmentVariables[name].type || BuildEnvironmentVariableType.PLAINTEXT,
            value: environmentVariables[name].value,
        }));
    }
    /**
     * (experimental) Adds a secondary source to the Project.
     *
     * @param secondarySource the source to add as a secondary source.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     * @experimental
     */
    addSecondarySource(secondarySource) {
        if (!secondarySource.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary sources');
        }
        const secondarySourceConfig = secondarySource.bind(this, this);
        this._secondarySources.push(secondarySourceConfig.sourceProperty);
        if (secondarySourceConfig.sourceVersion) {
            this._secondarySourceVersions.push({
                sourceIdentifier: secondarySource.identifier,
                sourceVersion: secondarySourceConfig.sourceVersion,
            });
        }
    }
    /**
     * (experimental) Adds a fileSystemLocation to the Project.
     *
     * @param fileSystemLocation the fileSystemLocation to add.
     * @experimental
     */
    addFileSystemLocation(fileSystemLocation) {
        const fileSystemConfig = fileSystemLocation.bind(this, this);
        this._fileSystemLocations.push(fileSystemConfig.location);
    }
    /**
     * (experimental) Adds a secondary artifact to the Project.
     *
     * @param secondaryArtifact the artifact to add as a secondary artifact.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     * @experimental
     */
    addSecondaryArtifact(secondaryArtifact) {
        if (!secondaryArtifact.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary artifacts');
        }
        this._secondaryArtifacts.push(secondaryArtifact.bind(this, this).artifactsProperty);
    }
    /**
     * (experimental) A callback invoked when the given project is added to a CodePipeline.
     *
     * @param _scope the construct the binding is taking place in.
     * @param options additional options for the binding.
     * @experimental
     */
    bindToCodePipeline(_scope, options) {
        // work around a bug in CodeBuild: it ignores the KMS key set on the pipeline,
        // and always uses its own, project-level key
        if (options.artifactBucket.encryptionKey && !this._encryptionKey) {
            // we cannot safely do this assignment if the key is of type kms.Key,
            // and belongs to a stack in a different account or region than the project
            // (that would cause an illegal reference, as KMS keys don't have physical names)
            const keyStack = core_1.Stack.of(options.artifactBucket.encryptionKey);
            const projectStack = core_1.Stack.of(this);
            if (!(options.artifactBucket.encryptionKey instanceof kms.Key &&
                (keyStack.account !== projectStack.account || keyStack.region !== projectStack.region))) {
                this.encryptionKey = options.artifactBucket.encryptionKey;
            }
        }
    }
    /**
     * (experimental) Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @experimental
     * @override true
     */
    validate() {
        const ret = new Array();
        if (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) {
            if (this._secondarySources.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary sources. ' +
                    "Use the CodeBuild Pipeline Actions' `extraInputs` property instead");
            }
            if (this._secondaryArtifacts.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary artifacts. ' +
                    "Use the CodeBuild Pipeline Actions' `outputs` property instead");
            }
        }
        return ret;
    }
    set encryptionKey(encryptionKey) {
        this._encryptionKey = encryptionKey;
        encryptionKey.grantEncryptDecrypt(this);
    }
    createLoggingPermission() {
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: `/aws/codebuild/${this.projectName}`,
        });
        const logGroupStarArn = `${logGroupArn}:*`;
        return new iam.PolicyStatement({
            resources: [logGroupArn, logGroupStarArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        });
    }
    renderEnvironment(env = {}, projectVars = {}) {
        var _a, _b;
        const vars = {};
        const containerVars = env.environmentVariables || {};
        // first apply environment variables from the container definition
        for (const name of Object.keys(containerVars)) {
            vars[name] = containerVars[name];
        }
        // now apply project-level vars
        for (const name of Object.keys(projectVars)) {
            vars[name] = projectVars[name];
        }
        const hasEnvironmentVars = Object.keys(vars).length > 0;
        const errors = this.buildImage.validate(env);
        if (errors.length > 0) {
            throw new Error('Invalid CodeBuild environment: ' + errors.join('\n'));
        }
        const imagePullPrincipalType = this.buildImage.imagePullPrincipalType === ImagePullPrincipalType.CODEBUILD
            ? ImagePullPrincipalType.CODEBUILD
            : ImagePullPrincipalType.SERVICE_ROLE;
        if (this.buildImage.repository) {
            if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
                this.buildImage.repository.grantPull(this);
            }
            else {
                const statement = new iam.PolicyStatement({
                    principals: [new iam.ServicePrincipal('codebuild.amazonaws.com')],
                    actions: ['ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage', 'ecr:BatchCheckLayerAvailability'],
                });
                statement.sid = 'CodeBuild';
                this.buildImage.repository.addToResourcePolicy(statement);
            }
        }
        if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
            (_a = this.buildImage.secretsManagerCredentials) === null || _a === void 0 ? void 0 : _a.grantRead(this);
        }
        const secret = this.buildImage.secretsManagerCredentials;
        return {
            type: this.buildImage.type,
            image: this.buildImage.imageId,
            imagePullCredentialsType: imagePullPrincipalType,
            registryCredential: secret
                ? {
                    credentialProvider: 'SECRETS_MANAGER',
                    // Secrets must be referenced by either the full ARN (with SecretsManager suffix), or by name.
                    // "Partial" ARNs (without the suffix) will fail a validation regex at deploy-time.
                    credential: (_b = secret.secretFullArn) !== null && _b !== void 0 ? _b : secret.secretName,
                }
                : undefined,
            privilegedMode: env.privileged || false,
            computeType: env.computeType || this.buildImage.defaultComputeType,
            environmentVariables: hasEnvironmentVars ? Project.serializeEnvVariables(vars) : undefined,
        };
    }
    renderFileSystemLocations() {
        return this._fileSystemLocations.length === 0
            ? undefined
            : this._fileSystemLocations;
    }
    renderSecondarySources() {
        return this._secondarySources.length === 0
            ? undefined
            : this._secondarySources;
    }
    renderSecondarySourceVersions() {
        return this._secondarySourceVersions.length === 0
            ? undefined
            : this._secondarySourceVersions;
    }
    renderSecondaryArtifacts() {
        return this._secondaryArtifacts.length === 0
            ? undefined
            : this._secondaryArtifacts;
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * codebuild creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroups || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if ((props.securityGroups && props.securityGroups.length > 0) && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic generated security group for CodeBuild ' + core_1.Names.uniqueId(this),
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        return {
            vpcId: props.vpc.vpcId,
            subnets: props.vpc.selectSubnets(props.subnetSelection).subnetIds,
            securityGroupIds: this.connections.securityGroups.map(s => s.securityGroupId),
        };
    }
    addVpcRequiredPermissions(props, project) {
        if (!props.vpc || !this.role) {
            return;
        }
        this.role.addToPolicy(new iam.PolicyStatement({
            resources: [`arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:network-interface/*`],
            actions: ['ec2:CreateNetworkInterfacePermission'],
            conditions: {
                StringEquals: {
                    'ec2:Subnet': props.vpc
                        .selectSubnets(props.subnetSelection).subnetIds
                        .map(si => `arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:subnet/${si}`),
                    'ec2:AuthorizedService': 'codebuild.amazonaws.com',
                },
            },
        }));
        const policy = new iam.Policy(this, 'PolicyDocument', {
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        'ec2:CreateNetworkInterface',
                        'ec2:DescribeNetworkInterfaces',
                        'ec2:DeleteNetworkInterface',
                        'ec2:DescribeSubnets',
                        'ec2:DescribeSecurityGroups',
                        'ec2:DescribeDhcpOptions',
                        'ec2:DescribeVpcs',
                    ],
                }),
            ],
        });
        this.role.attachInlinePolicy(policy);
        // add an explicit dependency between the EC2 Policy and this Project -
        // otherwise, creating the Project fails, as it requires these permissions
        // to be already attached to the Project's Role
        project.node.addDependency(policy);
    }
    validateCodePipelineSettings(artifacts) {
        const sourceType = this.source.type;
        const artifactsType = artifacts.type;
        if ((sourceType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE ||
            artifactsType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) &&
            (sourceType !== artifactsType)) {
            throw new Error('Both source and artifacts must be set to CodePipeline');
        }
    }
}
exports.Project = Project;
/**
 * (experimental) Build machine compute type.
 *
 * @experimental
 */
var ComputeType;
(function (ComputeType) {
    ComputeType["SMALL"] = "BUILD_GENERAL1_SMALL";
    ComputeType["MEDIUM"] = "BUILD_GENERAL1_MEDIUM";
    ComputeType["LARGE"] = "BUILD_GENERAL1_LARGE";
    ComputeType["X2_LARGE"] = "BUILD_GENERAL1_2XLARGE";
})(ComputeType = exports.ComputeType || (exports.ComputeType = {}));
/**
 * (experimental) The type of principal CodeBuild will use to pull your build Docker image.
 *
 * @experimental
 */
var ImagePullPrincipalType;
(function (ImagePullPrincipalType) {
    ImagePullPrincipalType["CODEBUILD"] = "CODEBUILD";
    ImagePullPrincipalType["SERVICE_ROLE"] = "SERVICE_ROLE";
})(ImagePullPrincipalType = exports.ImagePullPrincipalType || (exports.ImagePullPrincipalType = {}));
class ArmBuildImage {
    constructor(imageId) {
        this.type = 'ARM_CONTAINER';
        this.defaultComputeType = ComputeType.LARGE;
        this.imagePullPrincipalType = ImagePullPrincipalType.CODEBUILD;
        this.imageId = imageId;
    }
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== ComputeType.LARGE) {
            ret.push(`ARM images only support ComputeType '${ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
/**
 * (experimental) A CodeBuild image running Linux.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - LinuxBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 * - LinuxBuildImage.fromAsset(parent, id, props)
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 * @experimental
 */
class LinuxBuildImage {
    constructor(props) {
        /**
         * (experimental) The type of build environment.
         *
         * @experimental
         */
        this.type = 'LINUX_CONTAINER';
        /**
         * (experimental) The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
         *
         * @experimental
         */
        this.defaultComputeType = ComputeType.SMALL;
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Linux build image from a Docker Hub image.
     * @experimental
     */
    static fromDockerRegistry(name, options = {}) {
        return new LinuxBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
        });
    }
    /**
     * @param repository The ECR repository.
     * @param tag Image tag (default "latest").
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     * @experimental
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new LinuxBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository,
        });
    }
    /**
     * (experimental) Uses an Docker image asset as a Linux build image.
     *
     * @experimental
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new LinuxBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository: asset.repository,
        });
    }
    /**
     * (experimental) Uses a Docker image provided by CodeBuild.
     *
     * @param id The image identifier.
     * @returns A Docker image provided by CodeBuild.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     * @experimental
     * @example
     *
     * 'aws/codebuild/standard:4.0'
     */
    static fromCodeBuildImageId(id) {
        return LinuxBuildImage.codeBuildImage(id);
    }
    static codeBuildImage(name) {
        return new LinuxBuildImage({
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
        });
    }
    /**
     * (experimental) Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @experimental
     */
    validate(_) {
        return [];
    }
    /**
     * (experimental) Make a buildspec to run the indicated script.
     *
     * @experimental
     */
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxBuildImage = LinuxBuildImage;
/**
 * @experimental
 */
LinuxBuildImage.STANDARD_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:1.0');
/**
 * @experimental
 */
LinuxBuildImage.STANDARD_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:2.0');
/**
 * @experimental
 */
LinuxBuildImage.STANDARD_3_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:3.0');
/**
 * (experimental) The `aws/codebuild/standard:4.0` build image.
 *
 * @experimental
 */
LinuxBuildImage.STANDARD_4_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:4.0');
/**
 * @experimental
 */
LinuxBuildImage.AMAZON_LINUX_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:1.0');
/**
 * @experimental
 */
LinuxBuildImage.AMAZON_LINUX_2_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:2.0');
/**
 * (experimental) The Amazon Linux 2 x86_64 standard image, version `3.0`.
 *
 * @experimental
 */
LinuxBuildImage.AMAZON_LINUX_2_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:3.0');
/**
 * @experimental
 */
LinuxBuildImage.AMAZON_LINUX_2_ARM = new ArmBuildImage('aws/codebuild/amazonlinux2-aarch64-standard:1.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_BASE = LinuxBuildImage.codeBuildImage('aws/codebuild/ubuntu-base:14.04');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_24_4_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:24.4.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_26_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:26.1.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOCKER_17_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:17.09.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOCKER_18_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:18.09.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_10 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.10');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.11');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_8 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-8');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_9 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-9');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-11');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_14_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.14.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.1.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_8_11_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:8.11.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_6_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:6.3.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PHP_5_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:5.6');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PHP_7_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PHP_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.7.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_6_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.6.5');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_5_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.5.2');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_4_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.4.5');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_3_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.3.6');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_2_7_12 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:2.7.12');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.3');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.3.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_2_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.2.5');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.1');
/**
 * (experimental) Environment type for Windows Docker images.
 *
 * @experimental
 */
var WindowsImageType;
(function (WindowsImageType) {
    WindowsImageType["STANDARD"] = "WINDOWS_CONTAINER";
    WindowsImageType["SERVER_2019"] = "WINDOWS_SERVER_2019_CONTAINER";
})(WindowsImageType = exports.WindowsImageType || (exports.WindowsImageType = {}));
/**
 * (experimental) A CodeBuild image running Windows.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - WindowsBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }, imageType])
 * - WindowsBuildImage.fromEcrRepository(repo[, tag, imageType])
 * - WindowsBuildImage.fromAsset(parent, id, props, [, imageType])
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 * @experimental
 */
class WindowsBuildImage {
    constructor(props) {
        var _a;
        /**
         * (experimental) The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
         *
         * @experimental
         */
        this.defaultComputeType = ComputeType.MEDIUM;
        this.type = ((_a = props.imageType) !== null && _a !== void 0 ? _a : WindowsImageType.STANDARD).toString();
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Windows build image from a Docker Hub image.
     * @experimental
     */
    static fromDockerRegistry(name, options = {}, imageType = WindowsImageType.STANDARD) {
        return new WindowsBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
        });
    }
    /**
     * @param repository The ECR repository.
     * @param tag Image tag (default "latest").
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     * @experimental
     */
    static fromEcrRepository(repository, tag = 'latest', imageType = WindowsImageType.STANDARD) {
        return new WindowsBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository,
        });
    }
    /**
     * (experimental) Uses an Docker image asset as a Windows build image.
     *
     * @experimental
     */
    static fromAsset(scope, id, props, imageType = WindowsImageType.STANDARD) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new WindowsBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository: asset.repository,
        });
    }
    /**
     * (experimental) Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @experimental
     */
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType === ComputeType.SMALL) {
            ret.push('Windows images do not support the Small ComputeType');
        }
        return ret;
    }
    /**
     * (experimental) Make a buildspec to run the indicated script.
     *
     * @experimental
     */
    runScriptBuildspec(entrypoint) {
        return build_spec_1.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    // Would love to do downloading here and executing in the next step,
                    // but I don't know how to propagate the value of $TEMPDIR.
                    //
                    // Punting for someone who knows PowerShell well enough.
                    commands: [],
                },
                build: {
                    commands: [
                        'Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName',
                        `aws s3 cp s3://$env:${run_script_linux_build_spec_1.S3_BUCKET_ENV}/$env:${run_script_linux_build_spec_1.S3_KEY_ENV} $TEMPDIR\\scripts.zip`,
                        'New-Item -ItemType Directory -Path $TEMPDIR\\scriptdir',
                        'Expand-Archive -Path $TEMPDIR/scripts.zip -DestinationPath $TEMPDIR\\scriptdir',
                        '$env:SCRIPT_DIR = "$TEMPDIR\\scriptdir"',
                        `& $TEMPDIR\\scriptdir\\${entrypoint}`,
                    ],
                },
            },
        });
    }
}
exports.WindowsBuildImage = WindowsBuildImage;
/**
 * (deprecated) Corresponds to the standard CodeBuild image `aws/codebuild/windows-base:1.0`.
 *
 * @deprecated `WindowsBuildImage.WINDOWS_BASE_2_0` should be used instead.
 */
WindowsBuildImage.WIN_SERVER_CORE_2016_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * (experimental) The standard CodeBuild image `aws/codebuild/windows-base:2.0`, which is based off Windows Server Core 2016.
 *
 * @experimental
 */
WindowsBuildImage.WINDOWS_BASE_2_0 = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * (experimental) The standard CodeBuild image `aws/codebuild/windows-base:2019-1.0`, which is based off Windows Server Core 2019.
 *
 * @experimental
 */
WindowsBuildImage.WIN_SERVER_CORE_2019_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2019-1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
    imageType: WindowsImageType.SERVER_2019,
});
/**
 * @experimental
 */
var BuildEnvironmentVariableType;
(function (BuildEnvironmentVariableType) {
    BuildEnvironmentVariableType["PLAINTEXT"] = "PLAINTEXT";
    BuildEnvironmentVariableType["PARAMETER_STORE"] = "PARAMETER_STORE";
    BuildEnvironmentVariableType["SECRETS_MANAGER"] = "SECRETS_MANAGER";
})(BuildEnvironmentVariableType = exports.BuildEnvironmentVariableType || (exports.BuildEnvironmentVariableType = {}));
//# sourceMappingURL=data:application/json;base64,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