"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LinuxGpuBuildImage = void 0;
const ecr = require("../../aws-ecr"); // Automatically re-written from '@aws-cdk/aws-ecr'
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const region_info_1 = require("../../region-info"); // Automatically re-written from '@aws-cdk/region-info'
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const project_1 = require("./project");
const mappingName = 'AwsDeepLearningContainersRepositoriesAccounts';
/**
 * (experimental) A CodeBuild GPU image running Linux.
 *
 * This class has public constants that represent the most popular GPU images from AWS Deep Learning Containers.
 *
 * @see https://aws.amazon.com/releasenotes/available-deep-learning-containers-images
 * @experimental
 */
class LinuxGpuBuildImage {
    constructor(repositoryName, tag, account) {
        this.repositoryName = repositoryName;
        this.account = account;
        /**
         * (experimental) The type of build environment.
         *
         * @experimental
         */
        this.type = 'LINUX_GPU_CONTAINER';
        /**
         * (experimental) The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
         *
         * @experimental
         */
        this.defaultComputeType = project_1.ComputeType.LARGE;
        /**
         * (experimental) The type of principal that CodeBuild will use to pull this build Docker image.
         *
         * @experimental
         */
        this.imagePullPrincipalType = project_1.ImagePullPrincipalType.SERVICE_ROLE;
        this.accountExpression = account !== null && account !== void 0 ? account : core.Fn.findInMap(mappingName, core.Aws.REGION, 'repositoryAccount');
        this.imageId = `${this.accountExpression}.dkr.ecr.${core.Aws.REGION}.${core.Aws.URL_SUFFIX}/${repositoryName}:${tag}`;
    }
    /**
     * (experimental) Returns a Linux GPU build image from AWS Deep Learning Containers.
     *
     * @param repositoryName the name of the repository, for example "pytorch-inference".
     * @param tag the tag of the image, for example "1.5.0-gpu-py36-cu101-ubuntu16.04".
     * @param account the AWS account ID where the DLC repository for this region is hosted in.
     * @see https://aws.amazon.com/releasenotes/available-deep-learning-containers-images
     * @experimental
     */
    static awsDeepLearningContainersImage(repositoryName, tag, account) {
        return new LinuxGpuBuildImage(repositoryName, tag, account);
    }
    /**
     * (experimental) Function that allows the build image access to the construct tree.
     *
     * @experimental
     */
    bind(scope, project, _options) {
        if (!this.account) {
            const scopeStack = core.Stack.of(scope);
            // Unfortunately, the account IDs of the DLC repositories are not the same in all regions.
            // Because of that, use a (singleton) Mapping to find the correct account
            if (!scopeStack.node.tryFindChild(mappingName)) {
                const mapping = {};
                // get the accounts from the region-info module
                const region2Accounts = region_info_1.RegionInfo.regionMap(region_info_1.FactName.DLC_REPOSITORY_ACCOUNT);
                for (const [region, account] of Object.entries(region2Accounts)) {
                    mapping[region] = { repositoryAccount: account };
                }
                new core.CfnMapping(scopeStack, mappingName, { mapping });
            }
        }
        const repository = ecr.Repository.fromRepositoryAttributes(scope, 'AwsDlcRepositoryCodeBuild', {
            repositoryName: this.repositoryName,
            repositoryArn: ecr.Repository.arnForLocalRepository(this.repositoryName, scope, this.accountExpression),
        });
        repository.grantPull(project);
        return {};
    }
    /**
     * (experimental) Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @experimental
     */
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== project_1.ComputeType.LARGE) {
            ret.push(`GPU images only support ComputeType '${project_1.ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    /**
     * (experimental) Make a buildspec to run the indicated script.
     *
     * @experimental
     */
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxGpuBuildImage = LinuxGpuBuildImage;
/**
 * (experimental) Tensorflow 1.14.0 GPU image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_14_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.14.0-gpu-py36-cu100-ubuntu16.04');
/**
 * (experimental) Tensorflow 1.15.0 GPU image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.15.0-gpu-py36-cu100-ubuntu18.04');
/**
 * (experimental) Tensorflow 1.15.2 GPU training image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_2_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.15.2-gpu-py37-cu100-ubuntu18.04');
/**
 * (experimental) Tensorflow 1.15.2 GPU inference image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_2_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-inference', '1.15.2-gpu-py36-cu100-ubuntu18.04');
/**
 * (experimental) Tensorflow 2.0.0 GPU image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_0_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.0.0-gpu-py36-cu100-ubuntu18.04');
/**
 * (experimental) Tensorflow 2.0.1 GPU image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_0_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.0.1-gpu-py36-cu100-ubuntu18.04');
/**
 * (experimental) Tensorflow 2.1.0 GPU training image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.1.0-gpu-py36-cu101-ubuntu18.04');
/**
 * (experimental) Tensorflow 2.1.0 GPU inference image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-inference', '2.1.0-gpu-py36-cu101-ubuntu18.04');
/**
 * (experimental) Tensorflow 2.2.0 GPU training image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_2_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.2.0-gpu-py37-cu101-ubuntu18.04');
/**
 * (experimental) PyTorch 1.2.0 GPU image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_2_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.2.0-gpu-py36-cu100-ubuntu16.04');
/**
 * (experimental) PyTorch 1.3.1 GPU image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_3_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.3.1-gpu-py36-cu101-ubuntu16.04');
/**
 * (experimental) PyTorch 1.4.0 GPU training image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_4_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.4.0-gpu-py36-cu101-ubuntu16.04');
/**
 * (experimental) PyTorch 1.4.0 GPU inference image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_4_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-inference', '1.4.0-gpu-py36-cu101-ubuntu16.04');
/**
 * (experimental) PyTorch 1.5.0 GPU training image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_5_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.5.0-gpu-py36-cu101-ubuntu16.04');
/**
 * (experimental) PyTorch 1.5.0 GPU inference image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_5_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-inference', '1.5.0-gpu-py36-cu101-ubuntu16.04');
/**
 * (experimental) MXNet 1.4.1 GPU image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_MXNET_1_4_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('mxnet-training', '1.4.1-gpu-py36-cu100-ubuntu16.04');
/**
 * (experimental) MXNet 1.6.0 GPU image from AWS Deep Learning Containers.
 *
 * @experimental
 */
LinuxGpuBuildImage.DLC_MXNET_1_6_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('mxnet-training', '1.6.0-gpu-py36-cu101-ubuntu16.04');
//# sourceMappingURL=data:application/json;base64,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