"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricSet = exports.allMetricsGraphJson = void 0;
const drop_empty_object_at_the_end_of_an_array_token_1 = require("./drop-empty-object-at-the-end-of-an-array-token");
const env_tokens_1 = require("./env-tokens");
const metric_util_1 = require("./metric-util");
const object_1 = require("./object");
/**
 * Return the JSON structure which represents these metrics in a graph.
 *
 * Depending on the metric type (stat or expression), one `Metric` object
 * can render to multiple time series.
 *
 * - Top-level metrics will be rendered visibly, additionally added metrics will
 *   be rendered invisibly.
 * - IDs used in math expressions need to be either globally unique, or refer to the same
 *   metric object.
 *
 * This will be called by GraphWidget, no need for clients to call this.
 */
function allMetricsGraphJson(left, right) {
    // Add metrics to a set which will automatically expand them recursively,
    // making sure to retain conflicting the visible one on conflicting metrics objects.
    const mset = new MetricSet();
    mset.addTopLevel('left', ...left);
    mset.addTopLevel('right', ...right);
    // Render all metrics from the set.
    return mset.entries.map(entry => new drop_empty_object_at_the_end_of_an_array_token_1.DropEmptyObjectAtTheEndOfAnArray(metricGraphJson(entry.metric, entry.tag, entry.id)));
}
exports.allMetricsGraphJson = allMetricsGraphJson;
function metricGraphJson(metric, yAxis, id) {
    const config = metric.toMetricConfig();
    const ret = [];
    const options = { ...config.renderingProperties };
    metric_util_1.dispatchMetric(metric, {
        withStat(stat) {
            ret.push(stat.namespace, stat.metricName);
            // Dimensions
            for (const dim of (stat.dimensions || [])) {
                ret.push(dim.name, dim.value);
            }
            // Metric attributes that are rendered to graph options
            if (stat.account) {
                options.accountId = env_tokens_1.accountIfDifferentFromStack(stat.account);
            }
            if (stat.region) {
                options.region = env_tokens_1.regionIfDifferentFromStack(stat.region);
            }
            if (stat.period && stat.period.toSeconds() !== 300) {
                options.period = stat.period.toSeconds();
            }
            if (stat.statistic && stat.statistic !== 'Average') {
                options.stat = stat.statistic;
            }
        },
        withExpression(expr) {
            options.expression = expr.expression;
        },
    });
    // Options
    if (!yAxis) {
        options.visible = false;
    }
    if (yAxis !== 'left') {
        options.yAxis = yAxis;
    }
    if (id) {
        options.id = id;
    }
    // If math expressions don't have a label (or an ID), they'll render with an unelegant
    // autogenerated id ("metric_alias0"). Our ids may in the future also be autogenerated,
    // so if an ME doesn't have a label, use its toString() as the label (renders the expression).
    if (options.visible !== false && options.expression && !options.label) {
        options.label = metric.toString();
    }
    const renderedOpts = object_1.dropUndefined(options);
    if (Object.keys(renderedOpts).length !== 0) {
        ret.push(renderedOpts);
    }
    return ret;
}
/**
 * Contain a set of metrics, expanding math expressions
 *
 * "Primary" metrics (added via a top-level call) can be tagged with an additional value.
 */
class MetricSet {
    constructor() {
        this.metrics = new Array();
        this.metricById = new Map();
        this.metricByKey = new Map();
    }
    /**
     * Add the given set of metrics to this set
     */
    addTopLevel(tag, ...metrics) {
        for (const metric of metrics) {
            this.addOne(metric, tag);
        }
    }
    /**
     * Access all the accumulated timeseries entries
     */
    get entries() {
        return this.metrics;
    }
    /**
     * Add a metric into the set
     *
     * The id may not be the same as a previous metric added, unless it's the same metric.
     *
     * It can be made visible, in which case the new "metric" object replaces the old
     * one (and the new ones "renderingPropertieS" will be honored instead of the old
     * one's).
     */
    addOne(metric, tag, id) {
        const key = metric_util_1.metricKey(metric);
        let existingEntry;
        // Try lookup existing by id if we have one
        if (id) {
            existingEntry = this.metricById.get(id);
            if (existingEntry && metric_util_1.metricKey(existingEntry.metric) !== key) {
                throw new Error(`Cannot have two different metrics share the same id ('${id}') in one Alarm or Graph. Rename one of them.`);
            }
        }
        if (!existingEntry) {
            // Try lookup by metric if we didn't find one by id
            existingEntry = this.metricByKey.get(key);
            // If the one we found already has an id, it must be different from the id
            // we're trying to add and we want to add a new metric. Pretend we didn't
            // find one.
            if ((existingEntry === null || existingEntry === void 0 ? void 0 : existingEntry.id) && id) {
                existingEntry = undefined;
            }
        }
        // Create a new entry if we didn't find one so far
        let entry;
        if (existingEntry) {
            entry = existingEntry;
        }
        else {
            entry = { metric };
            this.metrics.push(entry);
            this.metricByKey.set(key, entry);
        }
        // If it didn't have an id but now we do, add one
        if (!entry.id && id) {
            entry.id = id;
            this.metricById.set(id, entry);
        }
        // If it didn't have a tag but now we do, add one
        if (!entry.tag && tag) {
            entry.tag = tag;
        }
        // Recurse and add children
        const conf = metric.toMetricConfig();
        if (conf.mathExpression) {
            for (const [subId, subMetric] of Object.entries(conf.mathExpression.usingMetrics)) {
                this.addOne(subMetric, undefined, subId);
            }
        }
    }
}
exports.MetricSet = MetricSet;
//# sourceMappingURL=data:application/json;base64,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