"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alarm = exports.TreatMissingData = exports.ComparisonOperator = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const alarm_base_1 = require("./alarm-base");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
const metric_util_1 = require("./private/metric-util");
const object_1 = require("./private/object");
const rendering_1 = require("./private/rendering");
const statistic_1 = require("./private/statistic");
/**
 * (experimental) Comparison operator for evaluating alarms.
 *
 * @experimental
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    ComparisonOperator["GREATER_THAN_OR_EQUAL_TO_THRESHOLD"] = "GreaterThanOrEqualToThreshold";
    ComparisonOperator["GREATER_THAN_THRESHOLD"] = "GreaterThanThreshold";
    ComparisonOperator["LESS_THAN_THRESHOLD"] = "LessThanThreshold";
    ComparisonOperator["LESS_THAN_OR_EQUAL_TO_THRESHOLD"] = "LessThanOrEqualToThreshold";
    ComparisonOperator["LESS_THAN_LOWER_OR_GREATER_THAN_UPPER_THRESHOLD"] = "LessThanLowerOrGreaterThanUpperThreshold";
    ComparisonOperator["GREATER_THAN_UPPER_THRESHOLD"] = "GreaterThanUpperThreshold";
    ComparisonOperator["LESS_THAN_LOWER_THRESHOLD"] = "LessThanLowerThreshold";
})(ComparisonOperator = exports.ComparisonOperator || (exports.ComparisonOperator = {}));
const OPERATOR_SYMBOLS = {
    GreaterThanOrEqualToThreshold: '>=',
    GreaterThanThreshold: '>',
    LessThanThreshold: '<',
    LessThanOrEqualToThreshold: '<=',
};
/**
 * (experimental) Specify how missing data points are treated during alarm evaluation.
 *
 * @experimental
 */
var TreatMissingData;
(function (TreatMissingData) {
    TreatMissingData["BREACHING"] = "breaching";
    TreatMissingData["NOT_BREACHING"] = "notBreaching";
    TreatMissingData["IGNORE"] = "ignore";
    TreatMissingData["MISSING"] = "missing";
})(TreatMissingData = exports.TreatMissingData || (exports.TreatMissingData = {}));
/**
 * (experimental) An alarm on a CloudWatch metric.
 *
 * @experimental
 */
class Alarm extends alarm_base_1.AlarmBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.alarmName,
        });
        const comparisonOperator = props.comparisonOperator || ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD;
        // Render metric, process potential overrides from the alarm
        // (It would be preferable if the statistic etc. was worked into the metric,
        // but hey we're allowing overrides...)
        const metricProps = this.renderMetric(props.metric);
        if (props.period) {
            metricProps.period = props.period.toSeconds();
        }
        if (props.statistic) {
            // Will overwrite both fields if present
            Object.assign(metricProps, {
                statistic: renderIfSimpleStatistic(props.statistic),
                extendedStatistic: renderIfExtendedStatistic(props.statistic),
            });
        }
        const alarm = new cloudwatch_generated_1.CfnAlarm(this, 'Resource', {
            // Meta
            alarmDescription: props.alarmDescription,
            alarmName: this.physicalName,
            // Evaluation
            comparisonOperator,
            threshold: props.threshold,
            datapointsToAlarm: props.datapointsToAlarm,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            evaluationPeriods: props.evaluationPeriods,
            treatMissingData: props.treatMissingData,
            // Actions
            actionsEnabled: props.actionsEnabled,
            alarmActions: core_1.Lazy.listValue({ produce: () => this.alarmActionArns }),
            insufficientDataActions: core_1.Lazy.listValue({ produce: (() => this.insufficientDataActionArns) }),
            okActions: core_1.Lazy.listValue({ produce: () => this.okActionArns }),
            // Metric
            ...metricProps,
        });
        this.alarmArn = this.getResourceArnAttribute(alarm.attrArn, {
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.alarmName = this.getResourceNameAttribute(alarm.ref);
        this.metric = props.metric;
        const datapoints = props.datapointsToAlarm || props.evaluationPeriods;
        this.annotation = {
            // eslint-disable-next-line max-len
            label: `${this.metric} ${OPERATOR_SYMBOLS[comparisonOperator]} ${props.threshold} for ${datapoints} datapoints within ${describePeriod(props.evaluationPeriods * metric_util_1.metricPeriod(props.metric).toSeconds())}`,
            value: props.threshold,
        };
    }
    /**
     * (experimental) Import an existing CloudWatch alarm provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param alarmArn Alarm ARN (i.e. arn:aws:cloudwatch:<region>:<account-id>:alarm:Foo).
     * @experimental
     */
    static fromAlarmArn(scope, id, alarmArn) {
        class Import extends alarm_base_1.AlarmBase {
            constructor() {
                super(...arguments);
                this.alarmArn = alarmArn;
                this.alarmName = core_1.Stack.of(scope).parseArn(alarmArn, ':').resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Turn this alarm into a horizontal annotation.
     *
     * This is useful if you want to represent an Alarm in a non-AlarmWidget.
     * An `AlarmWidget` can directly show an alarm, but it can only show a
     * single alarm and no other metrics. Instead, you can convert the alarm to
     * a HorizontalAnnotation and add it as an annotation to another graph.
     *
     * This might be useful if:
     *
     * - You want to show multiple alarms inside a single graph, for example if
     *    you have both a "small margin/long period" alarm as well as a
     *    "large margin/short period" alarm.
     *
     * - You want to show an Alarm line in a graph with multiple metrics in it.
     *
     * @experimental
     */
    toAnnotation() {
        return this.annotation;
    }
    renderMetric(metric) {
        const self = this;
        return metric_util_1.dispatchMetric(metric, {
            withStat(st) {
                var _a;
                self.validateMetricStat(st, metric);
                return object_1.dropUndefined({
                    dimensions: st.dimensions,
                    namespace: st.namespace,
                    metricName: st.metricName,
                    period: (_a = st.period) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                    statistic: renderIfSimpleStatistic(st.statistic),
                    extendedStatistic: renderIfExtendedStatistic(st.statistic),
                    unit: st.unitFilter,
                });
            },
            withExpression() {
                // Expand the math expression metric into a set
                const mset = new rendering_1.MetricSet();
                mset.addTopLevel(true, metric);
                let eid = 0;
                function uniqueMetricId() {
                    return `expr_${++eid}`;
                }
                return {
                    metrics: mset.entries.map(entry => metric_util_1.dispatchMetric(entry.metric, {
                        withStat(stat, conf) {
                            var _a;
                            self.validateMetricStat(stat, entry.metric);
                            return {
                                metricStat: {
                                    metric: {
                                        metricName: stat.metricName,
                                        namespace: stat.namespace,
                                        dimensions: stat.dimensions,
                                    },
                                    period: stat.period.toSeconds(),
                                    stat: stat.statistic,
                                    unit: stat.unitFilter,
                                },
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                        withExpression(expr, conf) {
                            var _a;
                            const hasSubmetrics = mathExprHasSubmetrics(expr);
                            if (hasSubmetrics) {
                                assertSubmetricsCount(expr);
                            }
                            return {
                                expression: expr.expression,
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                period: hasSubmetrics ? undefined : expr.period,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                    })),
                };
            },
        });
    }
    /**
     * Validate that if a region and account are in the given stat config, they match the Alarm
     */
    validateMetricStat(stat, metric) {
        const stack = core_1.Stack.of(this);
        if (definitelyDifferent(stat.region, stack.region)) {
            throw new Error(`Cannot create an Alarm in region '${stack.region}' based on metric '${metric}' in '${stat.region}'`);
        }
        if (definitelyDifferent(stat.account, stack.account)) {
            throw new Error(`Cannot create an Alarm in account '${stack.account}' based on metric '${metric}' in '${stat.account}'`);
        }
    }
}
exports.Alarm = Alarm;
function definitelyDifferent(x, y) {
    return x && !core_1.Token.isUnresolved(y) && x !== y;
}
/**
 * Return a human readable string for this period
 *
 * We know the seconds are always one of a handful of allowed values.
 */
function describePeriod(seconds) {
    if (seconds === 60) {
        return '1 minute';
    }
    if (seconds === 1) {
        return '1 second';
    }
    if (seconds > 60) {
        return (seconds / 60) + ' minutes';
    }
    return seconds + ' seconds';
}
function renderIfSimpleStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    return undefined;
}
function renderIfExtendedStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'percentile') {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return statistic.toLowerCase();
    }
    return undefined;
}
function mathExprHasSubmetrics(expr) {
    return Object.keys(expr.usingMetrics).length > 0;
}
function assertSubmetricsCount(expr) {
    if (Object.keys(expr.usingMetrics).length > 10) {
        // https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/AlarmThatSendsEmail.html#alarms-on-metric-math-expressions
        throw new Error('Alarms on math expressions cannot contain more than 10 individual metrics');
    }
    ;
}
//# sourceMappingURL=data:application/json;base64,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