"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataResourceType = exports.Trail = exports.ReadWriteType = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudtrail_generated_1 = require("./cloudtrail.generated");
/**
 * (experimental) Types of events that CloudTrail can log.
 *
 * @experimental
 */
var ReadWriteType;
(function (ReadWriteType) {
    ReadWriteType["READ_ONLY"] = "ReadOnly";
    ReadWriteType["WRITE_ONLY"] = "WriteOnly";
    ReadWriteType["ALL"] = "All";
    ReadWriteType["NONE"] = "None";
})(ReadWriteType = exports.ReadWriteType || (exports.ReadWriteType = {}));
/**
 * (experimental) Cloud trail allows you to log events that happen in your AWS account For example:.
 *
 * import { CloudTrail } from '@aws-cdk/aws-cloudtrail'
 *
 * const cloudTrail = new CloudTrail(this, 'MyTrail');
 *
 * NOTE the above example creates an UNENCRYPTED bucket by default,
 * If you are required to use an Encrypted bucket you can supply a preconfigured bucket
 * via TrailProps
 *
 * @experimental
 */
class Trail extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id, {
            physicalName: props.trailName,
        });
        this.eventSelectors = [];
        const cloudTrailPrincipal = new iam.ServicePrincipal('cloudtrail.amazonaws.com');
        this.s3bucket = props.bucket || new s3.Bucket(this, 'S3', { encryption: s3.BucketEncryption.UNENCRYPTED });
        this.s3bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.s3bucket.bucketArn],
            actions: ['s3:GetBucketAcl'],
            principals: [cloudTrailPrincipal],
        }));
        this.s3bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.s3bucket.arnForObjects(`${props.s3KeyPrefix ? `${props.s3KeyPrefix}/` : ''}AWSLogs/${core_1.Stack.of(this).account}/*`)],
            actions: ['s3:PutObject'],
            principals: [cloudTrailPrincipal],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        this.topic = props.snsTopic;
        if (this.topic) {
            this.topic.grantPublish(cloudTrailPrincipal);
        }
        let logsRole;
        if (props.sendToCloudWatchLogs) {
            if (props.cloudWatchLogGroup) {
                this.logGroup = props.cloudWatchLogGroup;
            }
            else {
                this.logGroup = new logs.LogGroup(this, 'LogGroup', {
                    retention: (_a = props.cloudWatchLogsRetention) !== null && _a !== void 0 ? _a : logs.RetentionDays.ONE_YEAR,
                });
            }
            logsRole = new iam.Role(this, 'LogsRole', { assumedBy: cloudTrailPrincipal });
            logsRole.addToPolicy(new iam.PolicyStatement({
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: [this.logGroup.logGroupArn],
            }));
        }
        if (props.managementEvents) {
            let managementEvent;
            if (props.managementEvents === ReadWriteType.NONE) {
                managementEvent = {
                    includeManagementEvents: false,
                };
            }
            else {
                managementEvent = {
                    includeManagementEvents: true,
                    readWriteType: props.managementEvents,
                };
            }
            this.eventSelectors.push(managementEvent);
        }
        if (props.kmsKey && props.encryptionKey) {
            throw new Error('Both kmsKey and encryptionKey must not be specified. Use only encryptionKey');
        }
        // TODO: not all regions support validation. Use service configuration data to fail gracefully
        const trail = new cloudtrail_generated_1.CfnTrail(this, 'Resource', {
            isLogging: true,
            enableLogFileValidation: props.enableFileValidation == null ? true : props.enableFileValidation,
            isMultiRegionTrail: props.isMultiRegionTrail == null ? true : props.isMultiRegionTrail,
            includeGlobalServiceEvents: props.includeGlobalServiceEvents == null ? true : props.includeGlobalServiceEvents,
            trailName: this.physicalName,
            kmsKeyId: (_c = (_b = props.encryptionKey) === null || _b === void 0 ? void 0 : _b.keyArn) !== null && _c !== void 0 ? _c : (_d = props.kmsKey) === null || _d === void 0 ? void 0 : _d.keyArn,
            s3BucketName: this.s3bucket.bucketName,
            s3KeyPrefix: props.s3KeyPrefix,
            cloudWatchLogsLogGroupArn: (_e = this.logGroup) === null || _e === void 0 ? void 0 : _e.logGroupArn,
            cloudWatchLogsRoleArn: logsRole === null || logsRole === void 0 ? void 0 : logsRole.roleArn,
            snsTopicName: (_f = this.topic) === null || _f === void 0 ? void 0 : _f.topicName,
            eventSelectors: this.eventSelectors,
        });
        this.trailArn = this.getResourceArnAttribute(trail.attrArn, {
            service: 'cloudtrail',
            resource: 'trail',
            resourceName: this.physicalName,
        });
        this.trailSnsTopicArn = trail.attrSnsTopicArn;
        // Add a dependency on the bucket policy being updated, CloudTrail will test this upon creation.
        if (this.s3bucket.policy) {
            trail.node.addDependency(this.s3bucket.policy);
        }
        // If props.sendToCloudWatchLogs is set to true then the trail needs to depend on the created logsRole
        // so that it can create the log stream for the log group. This ensures the logsRole is created and propagated
        // before the trail tries to create the log stream.
        if (logsRole !== undefined) {
            trail.node.addDependency(logsRole);
        }
    }
    /**
     * (experimental) Create an event rule for when an event is recorded by any Trail in the account.
     *
     * Note that the event doesn't necessarily have to come from this Trail, it can
     * be captured from any one.
     *
     * Be sure to filter the event further down using an event pattern.
     *
     * @experimental
     */
    static onEvent(scope, id, options = {}) {
        const rule = new events.Rule(scope, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            detailType: ['AWS API Call via CloudTrail'],
        });
        return rule;
    }
    /**
     * (experimental) When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     *
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds an Event Selector for filtering events that match either S3 or Lambda function operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param dataResourceValues the list of data resource ARNs to include in logging (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     * @experimental
     */
    addEventSelector(dataResourceType, dataResourceValues, options = {}) {
        if (dataResourceValues.length > 250) {
            throw new Error('A maximum of 250 data elements can be in one event selector');
        }
        if (this.eventSelectors.length > 5) {
            throw new Error('A maximum of 5 event selectors are supported per trail.');
        }
        this.eventSelectors.push({
            dataResources: [{
                    type: dataResourceType,
                    values: dataResourceValues,
                }],
            includeManagementEvents: options.includeManagementEvents,
            readWriteType: options.readWriteType,
        });
    }
    /**
     * (experimental) When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     *
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds a Lambda Data Event Selector for filtering events that match Lambda function operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param handlers the list of lambda function handlers whose data events should be logged (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     * @experimental
     */
    addLambdaEventSelector(handlers, options = {}) {
        if (handlers.length === 0) {
            return;
        }
        const dataResourceValues = handlers.map((h) => h.functionArn);
        return this.addEventSelector(DataResourceType.LAMBDA_FUNCTION, dataResourceValues, options);
    }
    /**
     * (experimental) Log all Lamda data events for all lambda functions the account.
     *
     * @default false
     * @see https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html
     * @experimental
     */
    logAllLambdaDataEvents(options = {}) {
        return this.addEventSelector(DataResourceType.LAMBDA_FUNCTION, [`arn:${this.stack.partition}:lambda`], options);
    }
    /**
     * (experimental) When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     *
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds an S3 Data Event Selector for filtering events that match S3 operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param s3Selector the list of S3 bucket with optional prefix to include in logging (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     * @experimental
     */
    addS3EventSelector(s3Selector, options = {}) {
        if (s3Selector.length === 0) {
            return;
        }
        const dataResourceValues = s3Selector.map((sel) => { var _a; return `${sel.bucket.bucketArn}/${(_a = sel.objectPrefix) !== null && _a !== void 0 ? _a : ''}`; });
        return this.addEventSelector(DataResourceType.S3_OBJECT, dataResourceValues, options);
    }
    /**
     * (experimental) Log all S3 data events for all objects for all buckets in the account.
     *
     * @default false
     * @see https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html
     * @experimental
     */
    logAllS3DataEvents(options = {}) {
        return this.addEventSelector(DataResourceType.S3_OBJECT, [`arn:${this.stack.partition}:s3:::`], options);
    }
    /**
     * (deprecated) Create an event rule for when an event is recorded by any Trail in the account.
     *
     * Note that the event doesn't necessarily have to come from this Trail, it can
     * be captured from any one.
     *
     * Be sure to filter the event further down using an event pattern.
     *
     * @deprecated - use Trail.onEvent()
     */
    onCloudTrailEvent(id, options = {}) {
        return Trail.onEvent(this, id, options);
    }
}
exports.Trail = Trail;
/**
 * (experimental) Resource type for a data event.
 *
 * @experimental
 */
var DataResourceType;
(function (DataResourceType) {
    DataResourceType["LAMBDA_FUNCTION"] = "AWS::Lambda::Function";
    DataResourceType["S3_OBJECT"] = "AWS::S3::Object";
})(DataResourceType = exports.DataResourceType || (exports.DataResourceType = {}));
//# sourceMappingURL=data:application/json;base64,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