"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheBehavior = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cache_policy_1 = require("../cache-policy");
const distribution_1 = require("../distribution");
/**
 * Allows configuring a variety of CloudFront functionality for a given URL path pattern.
 *
 * Note: This really should simply by called 'Behavior', but this name is already taken by the legacy
 * CloudFrontWebDistribution implementation.
 */
class CacheBehavior {
    constructor(originId, props) {
        this.props = props;
        this.originId = originId;
        this.validateEdgeLambdas(props.edgeLambdas);
        this.grantEdgeLambdaFunctionExecutionRole(props.edgeLambdas);
    }
    /**
     * Creates and returns the CloudFormation representation of this behavior.
     * This renders as a "CacheBehaviorProperty" regardless of if this is a default
     * cache behavior or not, as the two are identical except that the pathPattern
     * is omitted for the default cache behavior.
     *
     * @internal
     */
    _renderBehavior() {
        var _a, _b, _c, _d, _e, _f;
        return {
            pathPattern: this.props.pathPattern,
            targetOriginId: this.originId,
            allowedMethods: (_a = this.props.allowedMethods) === null || _a === void 0 ? void 0 : _a.methods,
            cachedMethods: (_b = this.props.cachedMethods) === null || _b === void 0 ? void 0 : _b.methods,
            cachePolicyId: ((_c = this.props.cachePolicy) !== null && _c !== void 0 ? _c : cache_policy_1.CachePolicy.CACHING_OPTIMIZED).cachePolicyId,
            compress: (_d = this.props.compress) !== null && _d !== void 0 ? _d : true,
            originRequestPolicyId: (_e = this.props.originRequestPolicy) === null || _e === void 0 ? void 0 : _e.originRequestPolicyId,
            smoothStreaming: this.props.smoothStreaming,
            viewerProtocolPolicy: (_f = this.props.viewerProtocolPolicy) !== null && _f !== void 0 ? _f : distribution_1.ViewerProtocolPolicy.ALLOW_ALL,
            lambdaFunctionAssociations: this.props.edgeLambdas
                ? this.props.edgeLambdas.map(edgeLambda => ({
                    lambdaFunctionArn: edgeLambda.functionVersion.edgeArn,
                    eventType: edgeLambda.eventType.toString(),
                    includeBody: edgeLambda.includeBody,
                }))
                : undefined,
        };
    }
    validateEdgeLambdas(edgeLambdas) {
        const includeBodyEventTypes = [distribution_1.LambdaEdgeEventType.ORIGIN_REQUEST, distribution_1.LambdaEdgeEventType.VIEWER_REQUEST];
        if (edgeLambdas && edgeLambdas.some(lambda => lambda.includeBody && !includeBodyEventTypes.includes(lambda.eventType))) {
            throw new Error('\'includeBody\' can only be true for ORIGIN_REQUEST or VIEWER_REQUEST event types.');
        }
    }
    grantEdgeLambdaFunctionExecutionRole(edgeLambdas) {
        if (!edgeLambdas || edgeLambdas.length === 0) {
            return;
        }
        edgeLambdas.forEach((edgeLambda) => {
            const role = edgeLambda.functionVersion.role;
            if (role && role instanceof iam.Role && role.assumeRolePolicy) {
                role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                    actions: ['sts:AssumeRole'],
                    principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                }));
            }
        });
    }
}
exports.CacheBehavior = CacheBehavior;
//# sourceMappingURL=data:application/json;base64,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