"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginRequestQueryStringBehavior = exports.OriginRequestHeaderBehavior = exports.OriginRequestCookieBehavior = exports.OriginRequestPolicy = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * (experimental) A Origin Request Policy configuration.
 *
 * @experimental
 * @resource AWS::CloudFront::OriginRequestPolicy
 */
class OriginRequestPolicy extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d;
        super(scope, id, {
            physicalName: props.originRequestPolicyName,
        });
        const originRequestPolicyName = (_a = props.originRequestPolicyName) !== null && _a !== void 0 ? _a : core_1.Names.uniqueId(this);
        if (!core_1.Token.isUnresolved(originRequestPolicyName) && !originRequestPolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'originRequestPolicyName' can only include '-', '_', and alphanumeric characters, got: '${props.originRequestPolicyName}'`);
        }
        const cookies = (_b = props.cookieBehavior) !== null && _b !== void 0 ? _b : OriginRequestCookieBehavior.none();
        const headers = (_c = props.headerBehavior) !== null && _c !== void 0 ? _c : OriginRequestHeaderBehavior.none();
        const queryStrings = (_d = props.queryStringBehavior) !== null && _d !== void 0 ? _d : OriginRequestQueryStringBehavior.none();
        const resource = new cloudfront_generated_1.CfnOriginRequestPolicy(this, 'Resource', {
            originRequestPolicyConfig: {
                name: originRequestPolicyName,
                comment: props.comment,
                cookiesConfig: {
                    cookieBehavior: cookies.behavior,
                    cookies: cookies.cookies,
                },
                headersConfig: {
                    headerBehavior: headers.behavior,
                    headers: headers.headers,
                },
                queryStringsConfig: {
                    queryStringBehavior: queryStrings.behavior,
                    queryStrings: queryStrings.queryStrings,
                },
            },
        });
        this.originRequestPolicyId = resource.ref;
    }
    /**
     * (experimental) Imports a Origin Request Policy from its id.
     *
     * @experimental
     */
    static fromOriginRequestPolicyId(scope, id, originRequestPolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.originRequestPolicyId = originRequestPolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed origin request policy. */
    static fromManagedOriginRequestPolicy(managedOriginRequestPolicyId) {
        return new class {
            constructor() {
                this.originRequestPolicyId = managedOriginRequestPolicyId;
            }
        }();
    }
}
exports.OriginRequestPolicy = OriginRequestPolicy;
/**
 * (experimental) This policy includes only the User-Agent and Referer headers.
 *
 * It doesn’t include any query strings or cookies.
 *
 * @experimental
 */
OriginRequestPolicy.USER_AGENT_REFERER_HEADERS = OriginRequestPolicy.fromManagedOriginRequestPolicy('acba4595-bd28-49b8-b9fe-13317c0390fa');
/**
 * (experimental) This policy includes the header that enables cross-origin resource sharing (CORS) requests when the origin is a custom origin.
 *
 * @experimental
 */
OriginRequestPolicy.CORS_CUSTOM_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('59781a5b-3903-41f3-afcb-af62929ccde1');
/**
 * (experimental) This policy includes the headers that enable cross-origin resource sharing (CORS) requests when the origin is an Amazon S3 bucket.
 *
 * @experimental
 */
OriginRequestPolicy.CORS_S3_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('88a5eaf4-2fd4-4709-b370-b4c650ea3fcf');
/**
 * (experimental) This policy includes all values (query strings, headers, and cookies) in the viewer request.
 *
 * @experimental
 */
OriginRequestPolicy.ALL_VIEWER = OriginRequestPolicy.fromManagedOriginRequestPolicy('216adef6-5c7f-47e4-b989-5492eafa07d3');
/**
 * (experimental) This policy is designed for use with an origin that is an AWS Elemental MediaTailor endpoint.
 *
 * @experimental
 */
OriginRequestPolicy.ELEMENTAL_MEDIA_TAILOR = OriginRequestPolicy.fromManagedOriginRequestPolicy('775133bc-15f2-49f9-abea-afb2e0bf67d2');
/**
 * (experimental) Ddetermines whether any cookies in viewer requests (and if so, which cookies) are included in requests that CloudFront sends to the origin.
 *
 * @experimental
 */
class OriginRequestCookieBehavior {
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
    /**
     * (experimental) Cookies in viewer requests are not included in requests that CloudFront sends to the origin.
     *
     * Any cookies that are listed in a CachePolicy are still included in origin requests.
     *
     * @experimental
     */
    static none() { return new OriginRequestCookieBehavior('none'); }
    /**
     * (experimental) All cookies in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static all() { return new OriginRequestCookieBehavior('all'); }
    /**
     * (experimental) Only the provided `cookies` are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new OriginRequestCookieBehavior('whitelist', cookies);
    }
}
exports.OriginRequestCookieBehavior = OriginRequestCookieBehavior;
/**
 * (experimental) Determines whether any HTTP headers (and if so, which headers) are included in requests that CloudFront sends to the origin.
 *
 * @experimental
 */
class OriginRequestHeaderBehavior {
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
    /**
     * (experimental) HTTP headers are not included in requests that CloudFront sends to the origin.
     *
     * Any headers that are listed in a CachePolicy are still included in origin requests.
     *
     * @experimental
     */
    static none() { return new OriginRequestHeaderBehavior('none'); }
    /**
     * (experimental) All HTTP headers in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * Additionally, any additional CloudFront headers provided are included; the additional headers are added by CloudFront.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-cloudfront-headers.html
     * @experimental
     */
    static all(...cloudfrontHeaders) {
        if (cloudfrontHeaders.length > 0) {
            if (!cloudfrontHeaders.every(header => header.startsWith('CloudFront-'))) {
                throw new Error('additional CloudFront headers passed to `OriginRequestHeaderBehavior.all()` must begin with \'CloudFront-\'');
            }
            return new OriginRequestHeaderBehavior('allViewerAndWhitelistCloudFront', cloudfrontHeaders);
        }
        else {
            return new OriginRequestHeaderBehavior('allViewer');
        }
    }
    /**
     * (experimental) Listed headers are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        return new OriginRequestHeaderBehavior('whitelist', headers);
    }
}
exports.OriginRequestHeaderBehavior = OriginRequestHeaderBehavior;
/**
 * (experimental) Determines whether any URL query strings in viewer requests (and if so, which query strings) are included in requests that CloudFront sends to the origin.
 *
 * @experimental
 */
class OriginRequestQueryStringBehavior {
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
    /**
     * (experimental) Query strings in viewer requests are not included in requests that CloudFront sends to the origin.
     *
     * Any query strings that are listed in a CachePolicy are still included in origin requests.
     *
     * @experimental
     */
    static none() { return new OriginRequestQueryStringBehavior('none'); }
    /**
     * (experimental) All query strings in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static all() { return new OriginRequestQueryStringBehavior('all'); }
    /**
     * (experimental) Only the provided `queryStrings` are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new OriginRequestQueryStringBehavior('whitelist', queryStrings);
    }
}
exports.OriginRequestQueryStringBehavior = OriginRequestQueryStringBehavior;
//# sourceMappingURL=data:application/json;base64,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