"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Origin = void 0;
const cloudfront = require("../../aws-cloudfront"); // Automatically re-written from '@aws-cdk/aws-cloudfront'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const http_origin_1 = require("./http-origin");
/**
 * (experimental) An Origin that is backed by an S3 bucket.
 *
 * If the bucket is configured for website hosting, this origin will be configured to use the bucket as an
 * HTTP server origin and will use the bucket's configured website redirects and error handling. Otherwise,
 * the origin is created as a bucket origin and will use CloudFront's redirect and error handling.
 *
 * @experimental
 */
class S3Origin {
    /**
     * @experimental
     */
    constructor(bucket, props = {}) {
        this.origin = bucket.isWebsite ?
            new http_origin_1.HttpOrigin(bucket.bucketWebsiteDomainName, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY,
                ...props,
            }) :
            new S3BucketOrigin(bucket, props);
    }
    /**
     * (experimental) The method called when a given Origin is added (for the first time) to a Distribution.
     *
     * @experimental
     */
    bind(scope, options) {
        return this.origin.bind(scope, options);
    }
}
exports.S3Origin = S3Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 */
class S3BucketOrigin extends cloudfront.OriginBase {
    constructor(bucket, { originAccessIdentity, ...props }) {
        super(bucket.bucketRegionalDomainName, props);
        this.bucket = bucket;
        if (originAccessIdentity) {
            this.originAccessIdentity = originAccessIdentity;
        }
    }
    bind(scope, options) {
        if (!this.originAccessIdentity) {
            // Using a bucket from another stack creates a cyclic reference with
            // the bucket taking a dependency on the generated S3CanonicalUserId when `grantRead` is called,
            // and the distribution having a dependency on the bucket's domain name.
            // Fix this by parenting the OAI in the bucket's stack when cross-stack usage is detected.
            const bucketStack = cdk.Stack.of(this.bucket);
            const bucketInDifferentStack = bucketStack !== cdk.Stack.of(scope);
            const oaiScope = bucketInDifferentStack ? bucketStack : scope;
            const oaiId = bucketInDifferentStack ? `${cdk.Names.uniqueId(scope)}S3Origin` : 'S3Origin';
            this.originAccessIdentity = new cloudfront.OriginAccessIdentity(oaiScope, oaiId, {
                comment: `Identity for ${options.originId}`,
            });
            this.bucket.grantRead(this.originAccessIdentity);
        }
        return super.bind(scope, options);
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiczMtb3JpZ2luLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiczMtb3JpZ2luLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLG1EQUFtRCxDQUFDLDBEQUEwRDtBQUU5RyxrQ0FBa0MsQ0FBQyxnREFBZ0Q7QUFDbkYsK0NBQTJDOzs7Ozs7Ozs7O0FBOEIzQyxNQUFhLFFBQVE7Ozs7SUFFakIsWUFBWSxNQUFrQixFQUFFLFFBQXVCLEVBQUU7UUFDckQsSUFBSSxDQUFDLE1BQU0sR0FBRyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDNUIsSUFBSSx3QkFBVSxDQUFDLE1BQU0sQ0FBQyx1QkFBdUIsRUFBRTtnQkFDM0MsY0FBYyxFQUFFLFVBQVUsQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTO2dCQUN6RCxHQUFHLEtBQUs7YUFDWCxDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksY0FBYyxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQztJQUMxQyxDQUFDOzs7Ozs7SUFDTSxJQUFJLENBQUMsS0FBb0IsRUFBRSxPQUFxQztRQUNuRSxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztJQUM1QyxDQUFDO0NBQ0o7QUFiRCw0QkFhQztBQUNEOzs7O0dBSUc7QUFDSCxNQUFNLGNBQWUsU0FBUSxVQUFVLENBQUMsVUFBVTtJQUU5QyxZQUE2QixNQUFrQixFQUFFLEVBQUUsb0JBQW9CLEVBQUUsR0FBRyxLQUFLLEVBQWlCO1FBQzlGLEtBQUssQ0FBQyxNQUFNLENBQUMsd0JBQXdCLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFEckIsV0FBTSxHQUFOLE1BQU0sQ0FBWTtRQUUzQyxJQUFJLG9CQUFvQixFQUFFO1lBQ3RCLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxvQkFBb0IsQ0FBQztTQUNwRDtJQUNMLENBQUM7SUFDTSxJQUFJLENBQUMsS0FBb0IsRUFBRSxPQUFxQztRQUNuRSxJQUFJLENBQUMsSUFBSSxDQUFDLG9CQUFvQixFQUFFO1lBQzVCLG9FQUFvRTtZQUNwRSxnR0FBZ0c7WUFDaEcsd0VBQXdFO1lBQ3hFLDBGQUEwRjtZQUMxRixNQUFNLFdBQVcsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDOUMsTUFBTSxzQkFBc0IsR0FBRyxXQUFXLEtBQUssR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDbkUsTUFBTSxRQUFRLEdBQUcsc0JBQXNCLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO1lBQzlELE1BQU0sS0FBSyxHQUFHLHNCQUFzQixDQUFDLENBQUMsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQztZQUMzRixJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxVQUFVLENBQUMsb0JBQW9CLENBQUMsUUFBUSxFQUFFLEtBQUssRUFBRTtnQkFDN0UsT0FBTyxFQUFFLGdCQUFnQixPQUFPLENBQUMsUUFBUSxFQUFFO2FBQzlDLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1NBQ3BEO1FBQ0QsT0FBTyxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBQ1Msb0JBQW9CO1FBQzFCLE9BQU8sRUFBRSxvQkFBb0IsRUFBRSxxQ0FBcUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLHdCQUF3QixFQUFFLEVBQUUsQ0FBQztJQUMvSCxDQUFDO0NBQ0oiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjbG91ZGZyb250IGZyb20gXCIuLi8uLi9hd3MtY2xvdWRmcm9udFwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3VkZnJvbnQnXG5pbXBvcnQgKiBhcyBzMyBmcm9tIFwiLi4vLi4vYXdzLXMzXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtczMnXG5pbXBvcnQgKiBhcyBjZGsgZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBIdHRwT3JpZ2luIH0gZnJvbSAnLi9odHRwLW9yaWdpbic7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgUzNPcmlnaW5Qcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBvcmlnaW5QYXRoPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBvcmlnaW5BY2Nlc3NJZGVudGl0eT86IGNsb3VkZnJvbnQuSU9yaWdpbkFjY2Vzc0lkZW50aXR5O1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBTM09yaWdpbiBpbXBsZW1lbnRzIGNsb3VkZnJvbnQuSU9yaWdpbiB7XG4gICAgcHJpdmF0ZSByZWFkb25seSBvcmlnaW46IGNsb3VkZnJvbnQuSU9yaWdpbjtcbiAgICBjb25zdHJ1Y3RvcihidWNrZXQ6IHMzLklCdWNrZXQsIHByb3BzOiBTM09yaWdpblByb3BzID0ge30pIHtcbiAgICAgICAgdGhpcy5vcmlnaW4gPSBidWNrZXQuaXNXZWJzaXRlID9cbiAgICAgICAgICAgIG5ldyBIdHRwT3JpZ2luKGJ1Y2tldC5idWNrZXRXZWJzaXRlRG9tYWluTmFtZSwge1xuICAgICAgICAgICAgICAgIHByb3RvY29sUG9saWN5OiBjbG91ZGZyb250Lk9yaWdpblByb3RvY29sUG9saWN5LkhUVFBfT05MWSxcbiAgICAgICAgICAgICAgICAuLi5wcm9wcyxcbiAgICAgICAgICAgIH0pIDpcbiAgICAgICAgICAgIG5ldyBTM0J1Y2tldE9yaWdpbihidWNrZXQsIHByb3BzKTtcbiAgICB9XG4gICAgcHVibGljIGJpbmQoc2NvcGU6IGNkay5Db25zdHJ1Y3QsIG9wdGlvbnM6IGNsb3VkZnJvbnQuT3JpZ2luQmluZE9wdGlvbnMpOiBjbG91ZGZyb250Lk9yaWdpbkJpbmRDb25maWcge1xuICAgICAgICByZXR1cm4gdGhpcy5vcmlnaW4uYmluZChzY29wZSwgb3B0aW9ucyk7XG4gICAgfVxufVxuLyoqXG4gKiBBbiBPcmlnaW4gc3BlY2lmaWMgdG8gYSBTMyBidWNrZXQgKG5vdCBjb25maWd1cmVkIGZvciB3ZWJzaXRlIGhvc3RpbmcpLlxuICpcbiAqIENvbnRhaW5zIGFkZGl0aW9uYWwgbG9naWMgYXJvdW5kIGJ1Y2tldCBwZXJtaXNzaW9ucyBhbmQgb3JpZ2luIGFjY2VzcyBpZGVudGl0aWVzLlxuICovXG5jbGFzcyBTM0J1Y2tldE9yaWdpbiBleHRlbmRzIGNsb3VkZnJvbnQuT3JpZ2luQmFzZSB7XG4gICAgcHJpdmF0ZSBvcmlnaW5BY2Nlc3NJZGVudGl0eSE6IGNsb3VkZnJvbnQuSU9yaWdpbkFjY2Vzc0lkZW50aXR5O1xuICAgIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgYnVja2V0OiBzMy5JQnVja2V0LCB7IG9yaWdpbkFjY2Vzc0lkZW50aXR5LCAuLi5wcm9wcyB9OiBTM09yaWdpblByb3BzKSB7XG4gICAgICAgIHN1cGVyKGJ1Y2tldC5idWNrZXRSZWdpb25hbERvbWFpbk5hbWUsIHByb3BzKTtcbiAgICAgICAgaWYgKG9yaWdpbkFjY2Vzc0lkZW50aXR5KSB7XG4gICAgICAgICAgICB0aGlzLm9yaWdpbkFjY2Vzc0lkZW50aXR5ID0gb3JpZ2luQWNjZXNzSWRlbnRpdHk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgcHVibGljIGJpbmQoc2NvcGU6IGNkay5Db25zdHJ1Y3QsIG9wdGlvbnM6IGNsb3VkZnJvbnQuT3JpZ2luQmluZE9wdGlvbnMpOiBjbG91ZGZyb250Lk9yaWdpbkJpbmRDb25maWcge1xuICAgICAgICBpZiAoIXRoaXMub3JpZ2luQWNjZXNzSWRlbnRpdHkpIHtcbiAgICAgICAgICAgIC8vIFVzaW5nIGEgYnVja2V0IGZyb20gYW5vdGhlciBzdGFjayBjcmVhdGVzIGEgY3ljbGljIHJlZmVyZW5jZSB3aXRoXG4gICAgICAgICAgICAvLyB0aGUgYnVja2V0IHRha2luZyBhIGRlcGVuZGVuY3kgb24gdGhlIGdlbmVyYXRlZCBTM0Nhbm9uaWNhbFVzZXJJZCB3aGVuIGBncmFudFJlYWRgIGlzIGNhbGxlZCxcbiAgICAgICAgICAgIC8vIGFuZCB0aGUgZGlzdHJpYnV0aW9uIGhhdmluZyBhIGRlcGVuZGVuY3kgb24gdGhlIGJ1Y2tldCdzIGRvbWFpbiBuYW1lLlxuICAgICAgICAgICAgLy8gRml4IHRoaXMgYnkgcGFyZW50aW5nIHRoZSBPQUkgaW4gdGhlIGJ1Y2tldCdzIHN0YWNrIHdoZW4gY3Jvc3Mtc3RhY2sgdXNhZ2UgaXMgZGV0ZWN0ZWQuXG4gICAgICAgICAgICBjb25zdCBidWNrZXRTdGFjayA9IGNkay5TdGFjay5vZih0aGlzLmJ1Y2tldCk7XG4gICAgICAgICAgICBjb25zdCBidWNrZXRJbkRpZmZlcmVudFN0YWNrID0gYnVja2V0U3RhY2sgIT09IGNkay5TdGFjay5vZihzY29wZSk7XG4gICAgICAgICAgICBjb25zdCBvYWlTY29wZSA9IGJ1Y2tldEluRGlmZmVyZW50U3RhY2sgPyBidWNrZXRTdGFjayA6IHNjb3BlO1xuICAgICAgICAgICAgY29uc3Qgb2FpSWQgPSBidWNrZXRJbkRpZmZlcmVudFN0YWNrID8gYCR7Y2RrLk5hbWVzLnVuaXF1ZUlkKHNjb3BlKX1TM09yaWdpbmAgOiAnUzNPcmlnaW4nO1xuICAgICAgICAgICAgdGhpcy5vcmlnaW5BY2Nlc3NJZGVudGl0eSA9IG5ldyBjbG91ZGZyb250Lk9yaWdpbkFjY2Vzc0lkZW50aXR5KG9haVNjb3BlLCBvYWlJZCwge1xuICAgICAgICAgICAgICAgIGNvbW1lbnQ6IGBJZGVudGl0eSBmb3IgJHtvcHRpb25zLm9yaWdpbklkfWAsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIHRoaXMuYnVja2V0LmdyYW50UmVhZCh0aGlzLm9yaWdpbkFjY2Vzc0lkZW50aXR5KTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gc3VwZXIuYmluZChzY29wZSwgb3B0aW9ucyk7XG4gICAgfVxuICAgIHByb3RlY3RlZCByZW5kZXJTM09yaWdpbkNvbmZpZygpOiBjbG91ZGZyb250LkNmbkRpc3RyaWJ1dGlvbi5TM09yaWdpbkNvbmZpZ1Byb3BlcnR5IHwgdW5kZWZpbmVkIHtcbiAgICAgICAgcmV0dXJuIHsgb3JpZ2luQWNjZXNzSWRlbnRpdHk6IGBvcmlnaW4tYWNjZXNzLWlkZW50aXR5L2Nsb3VkZnJvbnQvJHt0aGlzLm9yaWdpbkFjY2Vzc0lkZW50aXR5Lm9yaWdpbkFjY2Vzc0lkZW50aXR5TmFtZX1gIH07XG4gICAgfVxufVxuIl19