"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheck = exports.ScalingProcess = exports.ScalingEvents = exports.ScalingEvent = exports.UpdateType = exports.AutoScalingGroup = exports.GroupMetric = exports.GroupMetrics = exports.UpdatePolicy = exports.Signals = exports.Monitoring = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const elbv2 = require("../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
const lifecycle_hook_1 = require("./lifecycle-hook");
const scheduled_action_1 = require("./scheduled-action");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
const volume_1 = require("./volume");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * (experimental) The monitoring mode for instances launched in an autoscaling group.
 *
 * @experimental
 */
var Monitoring;
(function (Monitoring) {
    Monitoring[Monitoring["BASIC"] = 0] = "BASIC";
    Monitoring[Monitoring["DETAILED"] = 1] = "DETAILED";
})(Monitoring = exports.Monitoring || (exports.Monitoring = {}));
/**
 * (experimental) Configure whether the AutoScalingGroup waits for signals.
 *
 * If you do configure waiting for signals, you should make sure the instances
 * invoke `cfn-signal` somewhere in their UserData to signal that they have
 * started up (either successfully or unsuccessfully).
 *
 * Signals are used both during intial creation and subsequent updates.
 *
 * @experimental
 */
class Signals {
    /**
     * (experimental) Wait for the desiredCapacity of the AutoScalingGroup amount of signals to have been received.
     *
     * If no desiredCapacity has been configured, wait for minCapacity signals intead.
     *
     * This number is used during initial creation and during replacing updates.
     * During rolling updates, all updated instances must send a signal.
     *
     * @experimental
     */
    static waitForAll(options = {}) {
        validatePercentage(options.minSuccessPercentage);
        return new class extends Signals {
            renderCreationPolicy(renderOptions) {
                var _a;
                return this.doRender(options, (_a = renderOptions.desiredCapacity) !== null && _a !== void 0 ? _a : renderOptions.minCapacity);
            }
        }();
    }
    /**
     * (experimental) Wait for the minCapacity of the AutoScalingGroup amount of signals to have been received.
     *
     * This number is used during initial creation and during replacing updates.
     * During rolling updates, all updated instances must send a signal.
     *
     * @experimental
     */
    static waitForMinCapacity(options = {}) {
        validatePercentage(options.minSuccessPercentage);
        return new class extends Signals {
            renderCreationPolicy(renderOptions) {
                return this.doRender(options, renderOptions.minCapacity);
            }
        }();
    }
    /**
     * (experimental) Wait for a specific amount of signals to have been received.
     *
     * You should send one signal per instance, so this represents the number of
     * instances to wait for.
     *
     * This number is used during initial creation and during replacing updates.
     * During rolling updates, all updated instances must send a signal.
     *
     * @experimental
     */
    static waitForCount(count, options = {}) {
        validatePercentage(options.minSuccessPercentage);
        return new class extends Signals {
            renderCreationPolicy() {
                return this.doRender(options, count);
            }
        }();
    }
    /**
     * (experimental) Helper to render the actual creation policy, as the logic between them is quite similar.
     *
     * @experimental
     */
    doRender(options, count) {
        var _a;
        const minSuccessfulInstancesPercent = validatePercentage(options.minSuccessPercentage);
        return {
            ...options.minSuccessPercentage !== undefined ? { autoScalingCreationPolicy: { minSuccessfulInstancesPercent } } : {},
            resourceSignal: {
                count,
                timeout: (_a = options.timeout) === null || _a === void 0 ? void 0 : _a.toIsoString(),
            },
        };
    }
}
exports.Signals = Signals;
/**
 * (experimental) How existing instances should be updated.
 *
 * @experimental
 */
class UpdatePolicy {
    /**
     * (experimental) Create a new AutoScalingGroup and switch over to it.
     *
     * @experimental
     */
    static replacingUpdate() {
        return new class extends UpdatePolicy {
            _renderUpdatePolicy() {
                return {
                    autoScalingReplacingUpdate: { willReplace: true },
                };
            }
        }();
    }
    /**
     * (experimental) Replace the instances in the AutoScalingGroup one by one, or in batches.
     *
     * @experimental
     */
    static rollingUpdate(options = {}) {
        const minSuccessPercentage = validatePercentage(options.minSuccessPercentage);
        return new class extends UpdatePolicy {
            _renderUpdatePolicy(renderOptions) {
                var _a, _b, _c, _d, _e, _f, _g, _h, _j;
                return {
                    autoScalingRollingUpdate: {
                        maxBatchSize: options.maxBatchSize,
                        minInstancesInService: options.minInstancesInService,
                        suspendProcesses: (_a = options.suspendProcesses) !== null && _a !== void 0 ? _a : DEFAULT_SUSPEND_PROCESSES,
                        minSuccessfulInstancesPercent: minSuccessPercentage !== null && minSuccessPercentage !== void 0 ? minSuccessPercentage : (_c = (_b = renderOptions.creationPolicy) === null || _b === void 0 ? void 0 : _b.autoScalingCreationPolicy) === null || _c === void 0 ? void 0 : _c.minSuccessfulInstancesPercent,
                        waitOnResourceSignals: ((_d = options.waitOnResourceSignals) !== null && _d !== void 0 ? _d : ((_e = renderOptions.creationPolicy) === null || _e === void 0 ? void 0 : _e.resourceSignal) !== undefined) ? true : undefined,
                        pauseTime: (_g = (_f = options.pauseTime) === null || _f === void 0 ? void 0 : _f.toIsoString()) !== null && _g !== void 0 ? _g : (_j = (_h = renderOptions.creationPolicy) === null || _h === void 0 ? void 0 : _h.resourceSignal) === null || _j === void 0 ? void 0 : _j.timeout,
                    },
                };
            }
        }();
    }
}
exports.UpdatePolicy = UpdatePolicy;
/**
 * (experimental) A set of group metrics.
 *
 * @experimental
 */
class GroupMetrics {
    /**
     * @experimental
     */
    constructor(...metrics) {
        /**
         * @internal
         */
        this._metrics = new Set();
        metrics === null || metrics === void 0 ? void 0 : metrics.forEach(metric => this._metrics.add(metric));
    }
    /**
     * (experimental) Report all group metrics.
     *
     * @experimental
     */
    static all() {
        return new GroupMetrics();
    }
}
exports.GroupMetrics = GroupMetrics;
/**
 * (experimental) Group metrics that an Auto Scaling group sends to Amazon CloudWatch.
 *
 * @experimental
 */
class GroupMetric {
    /**
     * @experimental
     */
    constructor(name) {
        this.name = name;
    }
}
exports.GroupMetric = GroupMetric;
/**
 * (experimental) The minimum size of the Auto Scaling group.
 *
 * @experimental
 */
GroupMetric.MIN_SIZE = new GroupMetric('GroupMinSize');
/**
 * (experimental) The maximum size of the Auto Scaling group.
 *
 * @experimental
 */
GroupMetric.MAX_SIZE = new GroupMetric('GroupMaxSize');
/**
 * (experimental) The number of instances that the Auto Scaling group attempts to maintain.
 *
 * @experimental
 */
GroupMetric.DESIRED_CAPACITY = new GroupMetric('GroupDesiredCapacity');
/**
 * (experimental) The number of instances that are running as part of the Auto Scaling group This metric does not include instances that are pending or terminating.
 *
 * @experimental
 */
GroupMetric.IN_SERVICE_INSTANCES = new GroupMetric('GroupInServiceInstances');
/**
 * (experimental) The number of instances that are pending A pending instance is not yet in service, this metric does not include instances that are in service or terminating.
 *
 * @experimental
 */
GroupMetric.PENDING_INSTANCES = new GroupMetric('GroupPendingInstances');
/**
 * (experimental) The number of instances that are in a Standby state Instances in this state are still running but are not actively in service.
 *
 * @experimental
 */
GroupMetric.STANDBY_INSTANCES = new GroupMetric('GroupStandbyInstances');
/**
 * (experimental) The number of instances that are in the process of terminating This metric does not include instances that are in service or pending.
 *
 * @experimental
 */
GroupMetric.TERMINATING_INSTANCES = new GroupMetric('GroupTerminatingInstances');
/**
 * (experimental) The total number of instances in the Auto Scaling group This metric identifies the number of instances that are in service, pending, and terminating.
 *
 * @experimental
 */
GroupMetric.TOTAL_INSTANCES = new GroupMetric('GroupTotalInstances');
class AutoScalingGroupBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
    }
    /**
     * (experimental) Send a message to either an SQS queue or SNS topic when instances launch or terminate.
     *
     * @experimental
     */
    addLifecycleHook(id, props) {
        return new lifecycle_hook_1.LifecycleHook(this, `LifecycleHook${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * (experimental) Scale out or in based on time.
     *
     * @experimental
     */
    scaleOnSchedule(id, props) {
        return new scheduled_action_1.ScheduledAction(this, `ScheduledAction${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * (experimental) Scale out or in to achieve a target CPU utilization.
     *
     * @experimental
     */
    scaleOnCpuUtilization(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_CPU_UTILIZATION,
            targetValue: props.targetUtilizationPercent,
            ...props,
        });
    }
    /**
     * (experimental) Scale out or in to achieve a target network ingress rate.
     *
     * @experimental
     */
    scaleOnIncomingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_IN,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * (experimental) Scale out or in to achieve a target network egress rate.
     *
     * @experimental
     */
    scaleOnOutgoingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_OUT,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * (experimental) Scale out or in to achieve a target request handling rate.
     *
     * The AutoScalingGroup must have been attached to an Application Load Balancer
     * in order to be able to call this.
     *
     * @experimental
     */
    scaleOnRequestCount(id, props) {
        if (this.albTargetGroup === undefined) {
            throw new Error('Attach the AutoScalingGroup to a non-imported Application Load Balancer before calling scaleOnRequestCount()');
        }
        const resourceLabel = `${this.albTargetGroup.firstLoadBalancerFullName}/${this.albTargetGroup.targetGroupFullName}`;
        const policy = new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET,
            targetValue: props.targetRequestsPerSecond,
            resourceLabel,
            ...props,
        });
        policy.node.addDependency(this.albTargetGroup.loadBalancerAttached);
        return policy;
    }
    /**
     * (experimental) Scale out or in in order to keep a metric around a target value.
     *
     * @experimental
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            customMetric: props.metric,
            ...props,
        });
    }
    /**
     * (experimental) Scale out or in, in response to a metric.
     *
     * @experimental
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, autoScalingGroup: this });
    }
    addUserData(..._commands) {
        // do nothing
    }
}
/**
 * (experimental) A Fleet represents a managed set of EC2 instances.
 *
 * The Fleet models a number of AutoScalingGroups, a launch configuration, a
 * security group and an instance role.
 *
 * It allows adding arbitrary commands to the startup scripts of the instances
 * in the fleet.
 *
 * The ASG spans the availability zones specified by vpcSubnets, falling back to
 * the Vpc default strategy if not specified.
 *
 * @experimental
 */
class AutoScalingGroup extends AutoScalingGroupBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.autoScalingGroupName,
        });
        this.securityGroups = [];
        this.loadBalancerNames = [];
        this.targetGroupArns = [];
        this.groupMetrics = [];
        this.notifications = [];
        if (props.initOptions && !props.init) {
            throw new Error('Setting \'initOptions\' requires that \'init\' is also set');
        }
        this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'InstanceSecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: props.allowAllOutbound !== false,
        });
        this.connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        if (props.groupMetrics) {
            this.groupMetrics.push(...props.groupMetrics);
        }
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_a = props.userData) !== null && _a !== void 0 ? _a : imageConfig.userData;
        const userDataToken = core_1.Lazy.stringValue({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.listValue({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const launchConfig = new autoscaling_generated_1.CfnLaunchConfiguration(this, 'LaunchConfig', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            instanceMonitoring: (props.instanceMonitoring !== undefined ? (props.instanceMonitoring === Monitoring.DETAILED) : undefined),
            securityGroups: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            associatePublicIpAddress: props.associatePublicIpAddress,
            spotPrice: props.spotPrice,
            blockDeviceMappings: (props.blockDevices !== undefined ?
                synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined),
        });
        launchConfig.node.addDependency(this.role);
        // desiredCapacity just reflects what the user has supplied.
        const desiredCapacity = props.desiredCapacity;
        const minCapacity = props.minCapacity !== undefined ? props.minCapacity : 1;
        const maxCapacity = props.maxCapacity !== undefined ? props.maxCapacity :
            desiredCapacity !== undefined ? desiredCapacity : Math.max(minCapacity, 1);
        core_1.withResolved(minCapacity, maxCapacity, (min, max) => {
            if (min > max) {
                throw new Error(`minCapacity (${min}) should be <= maxCapacity (${max})`);
            }
        });
        core_1.withResolved(desiredCapacity, minCapacity, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Should have minCapacity (${min}) <= desiredCapacity (${desired})`);
            }
        });
        core_1.withResolved(desiredCapacity, maxCapacity, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (max < desired) {
                throw new Error(`Should have desiredCapacity (${desired}) <= maxCapacity (${max})`);
            }
        });
        if (desiredCapacity !== undefined) {
            core_1.Annotations.of(this).addWarning('desiredCapacity has been configured. Be aware this will reset the size of your AutoScalingGroup on every deployment. See https://github.com/aws/aws-cdk/issues/5215');
        }
        this.maxInstanceLifetime = props.maxInstanceLifetime;
        if (this.maxInstanceLifetime &&
            (this.maxInstanceLifetime.toSeconds() < 604800 || this.maxInstanceLifetime.toSeconds() > 31536000)) {
            throw new Error('maxInstanceLifetime must be between 7 and 365 days (inclusive)');
        }
        if (props.notificationsTopic && props.notifications) {
            throw new Error('Cannot set \'notificationsTopic\' and \'notifications\', \'notificationsTopic\' is deprecated use \'notifications\' instead');
        }
        if (props.notificationsTopic) {
            this.notifications = [{
                    topic: props.notificationsTopic,
                }];
        }
        if (props.notifications) {
            this.notifications = props.notifications.map(nc => {
                var _a;
                return ({
                    topic: nc.topic,
                    scalingEvents: (_a = nc.scalingEvents) !== null && _a !== void 0 ? _a : ScalingEvents.ALL,
                });
            });
        }
        const { subnetIds, hasPublic } = props.vpc.selectSubnets(props.vpcSubnets);
        const asgProps = {
            autoScalingGroupName: this.physicalName,
            cooldown: props.cooldown !== undefined ? props.cooldown.toSeconds().toString() : undefined,
            minSize: core_1.Tokenization.stringifyNumber(minCapacity),
            maxSize: core_1.Tokenization.stringifyNumber(maxCapacity),
            desiredCapacity: desiredCapacity !== undefined ? core_1.Tokenization.stringifyNumber(desiredCapacity) : undefined,
            launchConfigurationName: launchConfig.ref,
            loadBalancerNames: core_1.Lazy.listValue({ produce: () => this.loadBalancerNames }, { omitEmpty: true }),
            targetGroupArns: core_1.Lazy.listValue({ produce: () => this.targetGroupArns }, { omitEmpty: true }),
            notificationConfigurations: this.renderNotificationConfiguration(),
            metricsCollection: core_1.Lazy.anyValue({ produce: () => this.renderMetricsCollection() }),
            vpcZoneIdentifier: subnetIds,
            healthCheckType: props.healthCheck && props.healthCheck.type,
            healthCheckGracePeriod: props.healthCheck && props.healthCheck.gracePeriod && props.healthCheck.gracePeriod.toSeconds(),
            maxInstanceLifetime: this.maxInstanceLifetime ? this.maxInstanceLifetime.toSeconds() : undefined,
        };
        if (!hasPublic && props.associatePublicIpAddress) {
            throw new Error("To set 'associatePublicIpAddress: true' you must select Public subnets (vpcSubnets: { subnetType: SubnetType.PUBLIC })");
        }
        this.autoScalingGroup = new autoscaling_generated_1.CfnAutoScalingGroup(this, 'ASG', asgProps);
        this.osType = imageConfig.osType;
        this.autoScalingGroupName = this.getResourceNameAttribute(this.autoScalingGroup.ref),
            this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                service: 'autoscaling',
                resource: 'autoScalingGroup:*:autoScalingGroupName',
                resourceName: this.autoScalingGroupName,
            });
        this.node.defaultChild = this.autoScalingGroup;
        this.applyUpdatePolicies(props, { desiredCapacity, minCapacity });
        if (props.init) {
            this.applyCloudFormationInit(props.init, props.initOptions);
        }
        this.spotPrice = props.spotPrice;
    }
    /**
     * @experimental
     */
    static fromAutoScalingGroupName(scope, id, autoScalingGroupName) {
        class Import extends AutoScalingGroupBase {
            constructor() {
                super(...arguments);
                this.autoScalingGroupName = autoScalingGroupName;
                this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                    service: 'autoscaling',
                    resource: 'autoScalingGroup:*:autoScalingGroupName',
                    resourceName: this.autoScalingGroupName,
                });
                this.osType = ec2.OperatingSystemType.UNKNOWN;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Add the security group to all instances via the launch configuration security groups array.
     *
     * @param securityGroup : The security group to add.
     * @experimental
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * (experimental) Attach to a classic load balancer.
     *
     * @experimental
     */
    attachToClassicLB(loadBalancer) {
        this.loadBalancerNames.push(loadBalancer.loadBalancerName);
    }
    /**
     * (experimental) Attach to ELBv2 Application Target Group.
     *
     * @experimental
     */
    attachToApplicationTargetGroup(targetGroup) {
        if (this.albTargetGroup !== undefined) {
            throw new Error('Cannot add AutoScalingGroup to 2nd Target Group');
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        if (targetGroup instanceof elbv2.ApplicationTargetGroup) {
            // Copy onto self if it's a concrete type. We need this for autoscaling
            // based on request count, which we cannot do with an imported TargetGroup.
            this.albTargetGroup = targetGroup;
        }
        targetGroup.registerConnectable(this);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * (experimental) Attach to ELBv2 Application Target Group.
     *
     * @experimental
     */
    attachToNetworkTargetGroup(targetGroup) {
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * (experimental) Add command to the startup script of fleet instances.
     *
     * The command must be in the scripting language supported by the fleet's OS (i.e. Linux/Windows).
     * Does nothing for imported ASGs.
     *
     * @experimental
     */
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * (experimental) Adds a statement to the IAM role assumed by instances of this fleet.
     *
     * @experimental
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * (experimental) Use a CloudFormation Init configuration at instance startup.
     *
     * This does the following:
     *
     * - Attaches the CloudFormation Init metadata to the AutoScalingGroup resource.
     * - Add commands to the UserData to run `cfn-init` and `cfn-signal`.
     * - Update the instance's CreationPolicy to wait for `cfn-init` to finish
     *    before reporting success.
     *
     * @experimental
     */
    applyCloudFormationInit(init, options = {}) {
        var _a;
        if (!((_a = this.autoScalingGroup.cfnOptions.creationPolicy) === null || _a === void 0 ? void 0 : _a.resourceSignal)) {
            throw new Error('When applying CloudFormationInit, you must also configure signals by supplying \'signals\' at instantiation time.');
        }
        init.attach(this.autoScalingGroup, {
            platform: this.osType,
            instanceRole: this.role,
            userData: this.userData,
            configSets: options.configSets,
            embedFingerprint: options.embedFingerprint,
            printLog: options.printLog,
            ignoreFailures: options.ignoreFailures,
        });
    }
    /**
     * Apply CloudFormation update policies for the AutoScalingGroup
     */
    applyUpdatePolicies(props, signalOptions) {
        var _a;
        // Make sure people are not using the old and new properties together
        const oldProps = [
            'updateType',
            'rollingUpdateConfiguration',
            'resourceSignalCount',
            'resourceSignalTimeout',
            'replacingUpdateMinSuccessfulInstancesPercent',
        ];
        for (const prop of oldProps) {
            if ((props.signals || props.updatePolicy) && props[prop] !== undefined) {
                throw new Error(`Cannot set 'signals'/'updatePolicy' and '${prop}' together. Prefer 'signals'/'updatePolicy'`);
            }
        }
        // Reify updatePolicy to `rollingUpdate` default in case it is combined with `init`
        props = {
            ...props,
            updatePolicy: (_a = props.updatePolicy) !== null && _a !== void 0 ? _a : (props.init ? UpdatePolicy.rollingUpdate() : undefined),
        };
        if (props.signals || props.updatePolicy) {
            this.applyNewSignalUpdatePolicies(props, signalOptions);
        }
        else {
            this.applyLegacySignalUpdatePolicies(props);
        }
        // The following is technically part of the "update policy" but it's also a completely
        // separate aspect of rolling/replacing update, so it's just its own top-level property.
        // Default is 'true' because that's what you're most likely to want
        if (props.ignoreUnmodifiedSizeProperties !== false) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: true },
            };
        }
        if (props.signals && !props.init) {
            // To be able to send a signal using `cfn-init`, the execution role needs
            // `cloudformation:SignalResource`. Normally the binding of CfnInit would
            // grant that permissions and another one, but if the user wants to use
            // `signals` without `init`, add the permissions here.
            //
            // If they call `applyCloudFormationInit()` after construction, nothing bad
            // happens either, we'll just have a duplicate statement which doesn't hurt.
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: ['cloudformation:SignalResource'],
                resources: [core_1.Aws.STACK_ID],
            }));
        }
    }
    /**
     * Use 'signals' and 'updatePolicy' to determine the creation and update policies
     */
    applyNewSignalUpdatePolicies(props, signalOptions) {
        var _a, _b;
        this.autoScalingGroup.cfnOptions.creationPolicy = (_a = props.signals) === null || _a === void 0 ? void 0 : _a.renderCreationPolicy(signalOptions);
        this.autoScalingGroup.cfnOptions.updatePolicy = (_b = props.updatePolicy) === null || _b === void 0 ? void 0 : _b._renderUpdatePolicy({
            creationPolicy: this.autoScalingGroup.cfnOptions.creationPolicy,
        });
    }
    applyLegacySignalUpdatePolicies(props) {
        if (props.updateType === UpdateType.REPLACING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingReplacingUpdate: {
                    willReplace: true,
                },
            };
            if (props.replacingUpdateMinSuccessfulInstancesPercent !== undefined) {
                // Yes, this goes on CreationPolicy, not as a process parameter to ReplacingUpdate.
                // It's a little confusing, but the docs seem to explicitly state it will only be used
                // during the update?
                //
                // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html
                this.autoScalingGroup.cfnOptions.creationPolicy = {
                    ...this.autoScalingGroup.cfnOptions.creationPolicy,
                    autoScalingCreationPolicy: {
                        minSuccessfulInstancesPercent: validatePercentage(props.replacingUpdateMinSuccessfulInstancesPercent),
                    },
                };
            }
        }
        else if (props.updateType === UpdateType.ROLLING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingRollingUpdate: renderRollingUpdateConfig(props.rollingUpdateConfiguration),
            };
        }
        if (props.resourceSignalCount !== undefined || props.resourceSignalTimeout !== undefined) {
            this.autoScalingGroup.cfnOptions.creationPolicy = {
                ...this.autoScalingGroup.cfnOptions.creationPolicy,
                resourceSignal: {
                    count: props.resourceSignalCount,
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toISOString(),
                },
            };
        }
    }
    renderNotificationConfiguration() {
        if (this.notifications.length === 0) {
            return undefined;
        }
        return this.notifications.map(notification => ({
            topicArn: notification.topic.topicArn,
            notificationTypes: notification.scalingEvents ? notification.scalingEvents._types : ScalingEvents.ALL._types,
        }));
    }
    renderMetricsCollection() {
        if (this.groupMetrics.length === 0) {
            return undefined;
        }
        return this.groupMetrics.map(group => {
            var _a;
            return ({
                granularity: '1Minute',
                metrics: ((_a = group._metrics) === null || _a === void 0 ? void 0 : _a.size) !== 0 ? [...group._metrics].map(m => m.name) : undefined,
            });
        });
    }
}
exports.AutoScalingGroup = AutoScalingGroup;
/**
 * (deprecated) The type of update to perform on instances in this AutoScalingGroup.
 *
 * @deprecated Use UpdatePolicy instead
 */
var UpdateType;
(function (UpdateType) {
    UpdateType["NONE"] = "None";
    UpdateType["REPLACING_UPDATE"] = "Replace";
    UpdateType["ROLLING_UPDATE"] = "RollingUpdate";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
/**
 * (experimental) Fleet scaling events.
 *
 * @experimental
 */
var ScalingEvent;
(function (ScalingEvent) {
    ScalingEvent["INSTANCE_LAUNCH"] = "autoscaling:EC2_INSTANCE_LAUNCH";
    ScalingEvent["INSTANCE_TERMINATE"] = "autoscaling:EC2_INSTANCE_TERMINATE";
    ScalingEvent["INSTANCE_TERMINATE_ERROR"] = "autoscaling:EC2_INSTANCE_TERMINATE_ERROR";
    ScalingEvent["INSTANCE_LAUNCH_ERROR"] = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR";
    ScalingEvent["TEST_NOTIFICATION"] = "autoscaling:TEST_NOTIFICATION";
})(ScalingEvent = exports.ScalingEvent || (exports.ScalingEvent = {}));
/**
 * (experimental) A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS or create a custom group by instantiating a `NotificationTypes` object, e.g: `new NotificationTypes(`NotificationType.INSTANCE_LAUNCH`)`.
 *
 * @experimental
 */
class ScalingEvents {
    /**
     * @experimental
     */
    constructor(...types) {
        this._types = types;
    }
}
exports.ScalingEvents = ScalingEvents;
/**
 * (experimental) Fleet scaling errors.
 *
 * @experimental
 */
ScalingEvents.ERRORS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * (experimental) All fleet scaling events.
 *
 * @experimental
 */
ScalingEvents.ALL = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * (experimental) Fleet scaling launch events.
 *
 * @experimental
 */
ScalingEvents.LAUNCH_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR);
/**
 * @experimental
 */
var ScalingProcess;
(function (ScalingProcess) {
    ScalingProcess["LAUNCH"] = "Launch";
    ScalingProcess["TERMINATE"] = "Terminate";
    ScalingProcess["HEALTH_CHECK"] = "HealthCheck";
    ScalingProcess["REPLACE_UNHEALTHY"] = "ReplaceUnhealthy";
    ScalingProcess["AZ_REBALANCE"] = "AZRebalance";
    ScalingProcess["ALARM_NOTIFICATION"] = "AlarmNotification";
    ScalingProcess["SCHEDULED_ACTIONS"] = "ScheduledActions";
    ScalingProcess["ADD_TO_LOAD_BALANCER"] = "AddToLoadBalancer";
})(ScalingProcess = exports.ScalingProcess || (exports.ScalingProcess = {}));
// Recommended list of processes to suspend from here:
// https://aws.amazon.com/premiumsupport/knowledge-center/auto-scaling-group-rolling-updates/
const DEFAULT_SUSPEND_PROCESSES = [ScalingProcess.HEALTH_CHECK, ScalingProcess.REPLACE_UNHEALTHY, ScalingProcess.AZ_REBALANCE,
    ScalingProcess.ALARM_NOTIFICATION, ScalingProcess.SCHEDULED_ACTIONS];
/**
 * (experimental) Health check settings.
 *
 * @experimental
 */
class HealthCheck {
    constructor(type, gracePeriod) {
        this.type = type;
        this.gracePeriod = gracePeriod;
    }
    /**
     * (experimental) Use EC2 for health checks.
     *
     * @param options EC2 health check options.
     * @experimental
     */
    static ec2(options = {}) {
        return new HealthCheck(HealthCheckType.EC2, options.grace);
    }
    /**
     * (experimental) Use ELB for health checks.
     *
     * It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.
     *
     * @param options ELB health check options.
     * @experimental
     */
    static elb(options) {
        return new HealthCheck(HealthCheckType.ELB, options.grace);
    }
}
exports.HealthCheck = HealthCheck;
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["EC2"] = "EC2";
    HealthCheckType["ELB"] = "ELB";
})(HealthCheckType || (HealthCheckType = {}));
/**
 * Render the rolling update configuration into the appropriate object
 */
function renderRollingUpdateConfig(config = {}) {
    var _a;
    const waitOnResourceSignals = config.minSuccessfulInstancesPercent !== undefined ? true : false;
    const pauseTime = config.pauseTime || (waitOnResourceSignals ? core_1.Duration.minutes(5) : core_1.Duration.seconds(0));
    return {
        maxBatchSize: config.maxBatchSize,
        minInstancesInService: config.minInstancesInService,
        minSuccessfulInstancesPercent: validatePercentage(config.minSuccessfulInstancesPercent),
        waitOnResourceSignals,
        pauseTime: pauseTime && pauseTime.toISOString(),
        suspendProcesses: (_a = config.suspendProcesses) !== null && _a !== void 0 ? _a : DEFAULT_SUSPEND_PROCESSES,
    };
}
function validatePercentage(x) {
    if (x === undefined || (0 <= x && x <= 100)) {
        return x;
    }
    throw new Error(`Expected: a percentage 0..100, got: ${x}`);
}
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (volume === volume_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
            return {
                deviceName,
                noDevice: true,
            };
        }
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === volume_1.EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== volume_1.EbsDeviceVolumeType.IO1) {
                core_1.Annotations.of(construct).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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