"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateArbitraryIntervals = void 0;
/**
 * Arbitrary (valid) array of intervals
 *
 * There are many invalid combinations of interval arrays, so we have
 * to be very specific about generating arrays that are valid. We do this
 * by taking a full, valid interval schedule and progressively stripping parts
 * away from it.
 *
 * Some of the changes may change its meaning, but we take care to never leave
 * a schedule with insufficient information so that the parser will error out.
 */
function generateArbitraryIntervals(mrng) {
    const ret = new Array();
    const absolute = mrng.nextBoolean();
    // Ascending or descending scaling
    const factor = (mrng.nextBoolean() ? 1 : -1) * (absolute ? 10 : 1);
    const bias = absolute ? 50 : 0;
    // Begin with a full schedule
    ret.push({ lower: 0, upper: 10, change: -2 * factor + bias });
    ret.push({ lower: 10, upper: 20, change: -1 * factor + bias });
    ret.push({ lower: 20, upper: 60, change: 0 + bias });
    ret.push({ lower: 60, upper: 80, change: 0 + bias });
    ret.push({ lower: 80, upper: 90, change: 1 * factor + bias });
    ret.push({ lower: 90, upper: Infinity, change: 2 * factor + bias });
    // Take away parts from this. First we see if we do something to the 0-change alarms.
    // The actions can be: remove it OR turn it into a regular change value.
    const noChanges = ret.filter(x => x.change === bias);
    if (!absolute) {
        if (mrng.nextBoolean()) {
            if (mrng.nextBoolean()) {
                ret.splice(ret.indexOf(noChanges[0]), 1);
            }
            else {
                noChanges[0] = { ...noChanges[0], change: -1 * factor + bias };
            }
        }
        if (mrng.nextBoolean()) {
            if (mrng.nextBoolean()) {
                ret.splice(ret.indexOf(noChanges[1]), 1);
            }
            else {
                noChanges[1] = { ...noChanges[1], change: 1 * factor + bias };
            }
        }
    }
    else {
        // In absolute mode both have to get the same treatment at the same time
        // otherwise we'll end up with a timeline with two gaps
        if (mrng.nextBoolean()) {
            ret.splice(ret.indexOf(noChanges[0]), 1);
            ret.splice(ret.indexOf(noChanges[1]), 1);
        }
        else {
            noChanges[0] = { ...noChanges[0], change: -1 * factor + bias };
            noChanges[1] = { ...noChanges[1], change: 1 * factor + bias };
        }
    }
    // We might also take away either the bottom or the upper half
    if (mrng.nextInt(0, 2) === 0) {
        const signToStrip = mrng.nextBoolean() ? -1 : 1;
        let ix = ret.findIndex(x => Math.sign(x.change - bias) === signToStrip);
        while (ix >= 0) {
            ret.splice(ix, 1);
            ix = ret.findIndex(x => Math.sign(x.change - bias) === signToStrip);
        }
    }
    // Then we're going to arbitrarily get rid of bounds in the most naive way possible
    const iterations = mrng.nextInt(0, 10);
    for (let iter = 0; iter < iterations; iter++) {
        const i = mrng.nextInt(0, ret.length - 1);
        if (mrng.nextBoolean()) {
            // scrap lower bound
            // okay if current interval has an upper bound AND the preceding interval has an upper bound
            if (ret[i].upper !== undefined && (i === 0 || ret[i - 1].upper !== undefined)) {
                ret[i] = { ...ret[i], lower: undefined };
            }
        }
        else {
            // scrap upper bound
            // okay if current interval has a lower bound AND the succeeding interval has a lower bound
            if (ret[i].lower !== undefined && (i === ret.length - 1 || ret[i + 1].lower !== undefined)) {
                ret[i] = { ...ret[i], upper: undefined };
            }
        }
    }
    return { absolute, intervals: ret };
}
exports.generateArbitraryIntervals = generateArbitraryIntervals;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC11dGlscy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QtdXRpbHMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0E7Ozs7Ozs7Ozs7R0FVRztBQUNILFNBQWdCLDBCQUEwQixDQUFDLElBQXNCO0lBQzdELE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxFQUE4QixDQUFDO0lBQ3BELE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUNwQyxrQ0FBa0M7SUFDbEMsTUFBTSxNQUFNLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNuRSxNQUFNLElBQUksR0FBRyxRQUFRLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQy9CLDZCQUE2QjtJQUM3QixHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxFQUFFLENBQUMsRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUMsR0FBRyxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUM5RCxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUMsR0FBRyxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUMvRCxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFLE1BQU0sRUFBRSxDQUFDLEdBQUcsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUNyRCxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFLE1BQU0sRUFBRSxDQUFDLEdBQUcsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUNyRCxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFLE1BQU0sRUFBRSxDQUFDLEdBQUcsTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLENBQUM7SUFDOUQsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxNQUFNLEVBQUUsQ0FBQyxHQUFHLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQ3BFLHFGQUFxRjtJQUNyRix3RUFBd0U7SUFDeEUsTUFBTSxTQUFTLEdBQUcsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLEtBQUssSUFBSSxDQUFDLENBQUM7SUFDckQsSUFBSSxDQUFDLFFBQVEsRUFBRTtRQUNYLElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRSxFQUFFO1lBQ3BCLElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRSxFQUFFO2dCQUNwQixHQUFHLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7YUFDNUM7aUJBQ0k7Z0JBQ0QsU0FBUyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQyxHQUFHLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQzthQUNsRTtTQUNKO1FBQ0QsSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFLEVBQUU7WUFDcEIsSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFLEVBQUU7Z0JBQ3BCLEdBQUcsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQzthQUM1QztpQkFDSTtnQkFDRCxTQUFTLENBQUMsQ0FBQyxDQUFDLEdBQUcsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxNQUFNLEVBQUUsQ0FBQyxHQUFHLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQzthQUNqRTtTQUNKO0tBQ0o7U0FDSTtRQUNELHdFQUF3RTtRQUN4RSx1REFBdUQ7UUFDdkQsSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFLEVBQUU7WUFDcEIsR0FBRyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ3pDLEdBQUcsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztTQUM1QzthQUNJO1lBQ0QsU0FBUyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQyxHQUFHLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQztZQUMvRCxTQUFTLENBQUMsQ0FBQyxDQUFDLEdBQUcsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxNQUFNLEVBQUUsQ0FBQyxHQUFHLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQztTQUNqRTtLQUNKO0lBQ0QsOERBQThEO0lBQzlELElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEtBQUssQ0FBQyxFQUFFO1FBQzFCLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNoRCxJQUFJLEVBQUUsR0FBRyxHQUFHLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQyxLQUFLLFdBQVcsQ0FBQyxDQUFDO1FBQ3hFLE9BQU8sRUFBRSxJQUFJLENBQUMsRUFBRTtZQUNaLEdBQUcsQ0FBQyxNQUFNLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2xCLEVBQUUsR0FBRyxHQUFHLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQyxLQUFLLFdBQVcsQ0FBQyxDQUFDO1NBQ3ZFO0tBQ0o7SUFDRCxtRkFBbUY7SUFDbkYsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDdkMsS0FBSyxJQUFJLElBQUksR0FBRyxDQUFDLEVBQUUsSUFBSSxHQUFHLFVBQVUsRUFBRSxJQUFJLEVBQUUsRUFBRTtRQUMxQyxNQUFNLENBQUMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsRUFBRSxHQUFHLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQzFDLElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRSxFQUFFO1lBQ3BCLG9CQUFvQjtZQUNwQiw0RkFBNEY7WUFDNUYsSUFBSSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxLQUFLLFNBQVMsSUFBSSxDQUFDLENBQUMsS0FBSyxDQUFDLElBQUksR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxLQUFLLEtBQUssU0FBUyxDQUFDLEVBQUU7Z0JBQzNFLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxFQUFFLEdBQUcsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxTQUFTLEVBQUUsQ0FBQzthQUM1QztTQUNKO2FBQ0k7WUFDRCxvQkFBb0I7WUFDcEIsMkZBQTJGO1lBQzNGLElBQUksR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssS0FBSyxTQUFTLElBQUksQ0FBQyxDQUFDLEtBQUssR0FBRyxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxLQUFLLEtBQUssU0FBUyxDQUFDLEVBQUU7Z0JBQ3hGLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxFQUFFLEdBQUcsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxTQUFTLEVBQUUsQ0FBQzthQUM1QztTQUNKO0tBQ0o7SUFDRCxPQUFPLEVBQUUsUUFBUSxFQUFFLFNBQVMsRUFBRSxHQUFHLEVBQUUsQ0FBQztBQUN4QyxDQUFDO0FBM0VELGdFQTJFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGFwcHNjYWxpbmcgZnJvbSAnLi4vbGliJztcbi8qKlxuICogQXJiaXRyYXJ5ICh2YWxpZCkgYXJyYXkgb2YgaW50ZXJ2YWxzXG4gKlxuICogVGhlcmUgYXJlIG1hbnkgaW52YWxpZCBjb21iaW5hdGlvbnMgb2YgaW50ZXJ2YWwgYXJyYXlzLCBzbyB3ZSBoYXZlXG4gKiB0byBiZSB2ZXJ5IHNwZWNpZmljIGFib3V0IGdlbmVyYXRpbmcgYXJyYXlzIHRoYXQgYXJlIHZhbGlkLiBXZSBkbyB0aGlzXG4gKiBieSB0YWtpbmcgYSBmdWxsLCB2YWxpZCBpbnRlcnZhbCBzY2hlZHVsZSBhbmQgcHJvZ3Jlc3NpdmVseSBzdHJpcHBpbmcgcGFydHNcbiAqIGF3YXkgZnJvbSBpdC5cbiAqXG4gKiBTb21lIG9mIHRoZSBjaGFuZ2VzIG1heSBjaGFuZ2UgaXRzIG1lYW5pbmcsIGJ1dCB3ZSB0YWtlIGNhcmUgdG8gbmV2ZXIgbGVhdmVcbiAqIGEgc2NoZWR1bGUgd2l0aCBpbnN1ZmZpY2llbnQgaW5mb3JtYXRpb24gc28gdGhhdCB0aGUgcGFyc2VyIHdpbGwgZXJyb3Igb3V0LlxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2VuZXJhdGVBcmJpdHJhcnlJbnRlcnZhbHMobXJuZzogSVJhbmRvbUdlbmVyYXRvcik6IEFyYml0cmFyeUludGVydmFscyB7XG4gICAgY29uc3QgcmV0ID0gbmV3IEFycmF5PGFwcHNjYWxpbmcuU2NhbGluZ0ludGVydmFsPigpO1xuICAgIGNvbnN0IGFic29sdXRlID0gbXJuZy5uZXh0Qm9vbGVhbigpO1xuICAgIC8vIEFzY2VuZGluZyBvciBkZXNjZW5kaW5nIHNjYWxpbmdcbiAgICBjb25zdCBmYWN0b3IgPSAobXJuZy5uZXh0Qm9vbGVhbigpID8gMSA6IC0xKSAqIChhYnNvbHV0ZSA/IDEwIDogMSk7XG4gICAgY29uc3QgYmlhcyA9IGFic29sdXRlID8gNTAgOiAwO1xuICAgIC8vIEJlZ2luIHdpdGggYSBmdWxsIHNjaGVkdWxlXG4gICAgcmV0LnB1c2goeyBsb3dlcjogMCwgdXBwZXI6IDEwLCBjaGFuZ2U6IC0yICogZmFjdG9yICsgYmlhcyB9KTtcbiAgICByZXQucHVzaCh7IGxvd2VyOiAxMCwgdXBwZXI6IDIwLCBjaGFuZ2U6IC0xICogZmFjdG9yICsgYmlhcyB9KTtcbiAgICByZXQucHVzaCh7IGxvd2VyOiAyMCwgdXBwZXI6IDYwLCBjaGFuZ2U6IDAgKyBiaWFzIH0pO1xuICAgIHJldC5wdXNoKHsgbG93ZXI6IDYwLCB1cHBlcjogODAsIGNoYW5nZTogMCArIGJpYXMgfSk7XG4gICAgcmV0LnB1c2goeyBsb3dlcjogODAsIHVwcGVyOiA5MCwgY2hhbmdlOiAxICogZmFjdG9yICsgYmlhcyB9KTtcbiAgICByZXQucHVzaCh7IGxvd2VyOiA5MCwgdXBwZXI6IEluZmluaXR5LCBjaGFuZ2U6IDIgKiBmYWN0b3IgKyBiaWFzIH0pO1xuICAgIC8vIFRha2UgYXdheSBwYXJ0cyBmcm9tIHRoaXMuIEZpcnN0IHdlIHNlZSBpZiB3ZSBkbyBzb21ldGhpbmcgdG8gdGhlIDAtY2hhbmdlIGFsYXJtcy5cbiAgICAvLyBUaGUgYWN0aW9ucyBjYW4gYmU6IHJlbW92ZSBpdCBPUiB0dXJuIGl0IGludG8gYSByZWd1bGFyIGNoYW5nZSB2YWx1ZS5cbiAgICBjb25zdCBub0NoYW5nZXMgPSByZXQuZmlsdGVyKHggPT4geC5jaGFuZ2UgPT09IGJpYXMpO1xuICAgIGlmICghYWJzb2x1dGUpIHtcbiAgICAgICAgaWYgKG1ybmcubmV4dEJvb2xlYW4oKSkge1xuICAgICAgICAgICAgaWYgKG1ybmcubmV4dEJvb2xlYW4oKSkge1xuICAgICAgICAgICAgICAgIHJldC5zcGxpY2UocmV0LmluZGV4T2Yobm9DaGFuZ2VzWzBdKSwgMSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICBub0NoYW5nZXNbMF0gPSB7IC4uLm5vQ2hhbmdlc1swXSwgY2hhbmdlOiAtMSAqIGZhY3RvciArIGJpYXMgfTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICBpZiAobXJuZy5uZXh0Qm9vbGVhbigpKSB7XG4gICAgICAgICAgICBpZiAobXJuZy5uZXh0Qm9vbGVhbigpKSB7XG4gICAgICAgICAgICAgICAgcmV0LnNwbGljZShyZXQuaW5kZXhPZihub0NoYW5nZXNbMV0pLCAxKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgICAgIG5vQ2hhbmdlc1sxXSA9IHsgLi4ubm9DaGFuZ2VzWzFdLCBjaGFuZ2U6IDEgKiBmYWN0b3IgKyBiaWFzIH07XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICB9XG4gICAgZWxzZSB7XG4gICAgICAgIC8vIEluIGFic29sdXRlIG1vZGUgYm90aCBoYXZlIHRvIGdldCB0aGUgc2FtZSB0cmVhdG1lbnQgYXQgdGhlIHNhbWUgdGltZVxuICAgICAgICAvLyBvdGhlcndpc2Ugd2UnbGwgZW5kIHVwIHdpdGggYSB0aW1lbGluZSB3aXRoIHR3byBnYXBzXG4gICAgICAgIGlmIChtcm5nLm5leHRCb29sZWFuKCkpIHtcbiAgICAgICAgICAgIHJldC5zcGxpY2UocmV0LmluZGV4T2Yobm9DaGFuZ2VzWzBdKSwgMSk7XG4gICAgICAgICAgICByZXQuc3BsaWNlKHJldC5pbmRleE9mKG5vQ2hhbmdlc1sxXSksIDEpO1xuICAgICAgICB9XG4gICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgbm9DaGFuZ2VzWzBdID0geyAuLi5ub0NoYW5nZXNbMF0sIGNoYW5nZTogLTEgKiBmYWN0b3IgKyBiaWFzIH07XG4gICAgICAgICAgICBub0NoYW5nZXNbMV0gPSB7IC4uLm5vQ2hhbmdlc1sxXSwgY2hhbmdlOiAxICogZmFjdG9yICsgYmlhcyB9O1xuICAgICAgICB9XG4gICAgfVxuICAgIC8vIFdlIG1pZ2h0IGFsc28gdGFrZSBhd2F5IGVpdGhlciB0aGUgYm90dG9tIG9yIHRoZSB1cHBlciBoYWxmXG4gICAgaWYgKG1ybmcubmV4dEludCgwLCAyKSA9PT0gMCkge1xuICAgICAgICBjb25zdCBzaWduVG9TdHJpcCA9IG1ybmcubmV4dEJvb2xlYW4oKSA/IC0xIDogMTtcbiAgICAgICAgbGV0IGl4ID0gcmV0LmZpbmRJbmRleCh4ID0+IE1hdGguc2lnbih4LmNoYW5nZSAtIGJpYXMpID09PSBzaWduVG9TdHJpcCk7XG4gICAgICAgIHdoaWxlIChpeCA+PSAwKSB7XG4gICAgICAgICAgICByZXQuc3BsaWNlKGl4LCAxKTtcbiAgICAgICAgICAgIGl4ID0gcmV0LmZpbmRJbmRleCh4ID0+IE1hdGguc2lnbih4LmNoYW5nZSAtIGJpYXMpID09PSBzaWduVG9TdHJpcCk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgLy8gVGhlbiB3ZSdyZSBnb2luZyB0byBhcmJpdHJhcmlseSBnZXQgcmlkIG9mIGJvdW5kcyBpbiB0aGUgbW9zdCBuYWl2ZSB3YXkgcG9zc2libGVcbiAgICBjb25zdCBpdGVyYXRpb25zID0gbXJuZy5uZXh0SW50KDAsIDEwKTtcbiAgICBmb3IgKGxldCBpdGVyID0gMDsgaXRlciA8IGl0ZXJhdGlvbnM7IGl0ZXIrKykge1xuICAgICAgICBjb25zdCBpID0gbXJuZy5uZXh0SW50KDAsIHJldC5sZW5ndGggLSAxKTtcbiAgICAgICAgaWYgKG1ybmcubmV4dEJvb2xlYW4oKSkge1xuICAgICAgICAgICAgLy8gc2NyYXAgbG93ZXIgYm91bmRcbiAgICAgICAgICAgIC8vIG9rYXkgaWYgY3VycmVudCBpbnRlcnZhbCBoYXMgYW4gdXBwZXIgYm91bmQgQU5EIHRoZSBwcmVjZWRpbmcgaW50ZXJ2YWwgaGFzIGFuIHVwcGVyIGJvdW5kXG4gICAgICAgICAgICBpZiAocmV0W2ldLnVwcGVyICE9PSB1bmRlZmluZWQgJiYgKGkgPT09IDAgfHwgcmV0W2kgLSAxXS51cHBlciAhPT0gdW5kZWZpbmVkKSkge1xuICAgICAgICAgICAgICAgIHJldFtpXSA9IHsgLi4ucmV0W2ldLCBsb3dlcjogdW5kZWZpbmVkIH07XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAvLyBzY3JhcCB1cHBlciBib3VuZFxuICAgICAgICAgICAgLy8gb2theSBpZiBjdXJyZW50IGludGVydmFsIGhhcyBhIGxvd2VyIGJvdW5kIEFORCB0aGUgc3VjY2VlZGluZyBpbnRlcnZhbCBoYXMgYSBsb3dlciBib3VuZFxuICAgICAgICAgICAgaWYgKHJldFtpXS5sb3dlciAhPT0gdW5kZWZpbmVkICYmIChpID09PSByZXQubGVuZ3RoIC0gMSB8fCByZXRbaSArIDFdLmxvd2VyICE9PSB1bmRlZmluZWQpKSB7XG4gICAgICAgICAgICAgICAgcmV0W2ldID0geyAuLi5yZXRbaV0sIHVwcGVyOiB1bmRlZmluZWQgfTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4geyBhYnNvbHV0ZSwgaW50ZXJ2YWxzOiByZXQgfTtcbn1cbmV4cG9ydCBpbnRlcmZhY2UgSVJhbmRvbUdlbmVyYXRvciB7XG4gICAgbmV4dEJvb2xlYW4oKTogYm9vbGVhbjtcbiAgICBuZXh0SW50KG1pbjogbnVtYmVyLCBtYXg6IG51bWJlcik6IG51bWJlcjtcbn1cbmV4cG9ydCBpbnRlcmZhY2UgQXJiaXRyYXJ5SW50ZXJ2YWxzIHtcbiAgICByZWFkb25seSBhYnNvbHV0ZTogYm9vbGVhbjtcbiAgICByZWFkb25seSBpbnRlcnZhbHM6IGFwcHNjYWxpbmcuU2NhbGluZ0ludGVydmFsW107XG59XG4iXX0=