"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnumType = exports.UnionType = exports.InputType = exports.ObjectType = exports.InterfaceType = void 0;
const private_1 = require("./private");
const schema_field_1 = require("./schema-field");
/**
 * (experimental) Interface Types are abstract types that includes a certain set of fields that other types must include if they implement the interface.
 *
 * @experimental
 */
class InterfaceType {
    /**
     * @experimental
     */
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
        this.directives = props.directives;
    }
    /**
     * (experimental) Create a GraphQL Type representing this Intermediate Type.
     *
     * @param options the options to configure this attribute.
     * @experimental
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * (experimental) Generate the string of this object type.
     *
     * @experimental
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'interface',
            name: this.name,
            directives: this.directives,
            fields: Object.keys(this.definition).map((key) => {
                const field = this.definition[key];
                return `${key}${field.argsToString()}: ${field.toString()}${field.directivesToString(this.modes)}`;
            }),
            modes: this.modes,
        });
    }
    /**
     * (experimental) Add a field to this Interface Type.
     *
     * Interface Types must have both fieldName and field options.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Interface Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.InterfaceType = InterfaceType;
/**
 * (experimental) Object Types are types declared by you.
 *
 * @experimental
 */
class ObjectType extends InterfaceType {
    /**
     * @experimental
     */
    constructor(name, props) {
        var _a, _b;
        const options = {
            definition: (_b = (_a = props.interfaceTypes) === null || _a === void 0 ? void 0 : _a.reduce((def, interfaceType) => {
                return Object.assign({}, def, interfaceType.definition);
            }, props.definition)) !== null && _b !== void 0 ? _b : props.definition,
            directives: props.directives,
        };
        super(name, options);
        this.interfaceTypes = props.interfaceTypes;
        this.resolvers = [];
        Object.keys(this.definition).forEach((fieldName) => {
            const field = this.definition[fieldName];
            this.generateResolver(fieldName, field.fieldOptions);
        });
    }
    /**
     * (experimental) Add a field to this Object Type.
     *
     * Object Types must have both fieldName and field options.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Object Types must have both fieldName and field options.');
        }
        this.generateResolver(options.fieldName, options.field.fieldOptions);
        this.definition[options.fieldName] = options.field;
    }
    /**
     * (experimental) Generate the string of this object type.
     *
     * @experimental
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'type',
            name: this.name,
            interfaceTypes: this.interfaceTypes,
            directives: this.directives,
            fields: Object.keys(this.definition).map((key) => {
                const field = this.definition[key];
                return `${key}${field.argsToString()}: ${field.toString()}${field.directivesToString(this.modes)}`;
            }),
            modes: this.modes,
        });
    }
    /**
     * (experimental) Generate the resolvers linked to this Object Type.
     *
     * @experimental
     */
    generateResolver(fieldName, options) {
        if (!(options === null || options === void 0 ? void 0 : options.dataSource))
            return;
        if (!this.resolvers) {
            this.resolvers = [];
        }
        this.resolvers.push(options.dataSource.createResolver({
            typeName: this.name,
            fieldName: fieldName,
            pipelineConfig: options.pipelineConfig,
            requestMappingTemplate: options.requestMappingTemplate,
            responseMappingTemplate: options.responseMappingTemplate,
        }));
    }
}
exports.ObjectType = ObjectType;
/**
 * (experimental) Input Types are abstract types that define complex objects.
 *
 * They are used in arguments to represent
 *
 * @experimental
 */
class InputType {
    /**
     * @experimental
     */
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
    }
    /**
     * (experimental) Create a GraphQL Type representing this Input Type.
     *
     * @param options the options to configure this attribute.
     * @experimental
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * (experimental) Generate the string of this input type.
     *
     * @experimental
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'input',
            name: this.name,
            fields: Object.keys(this.definition).map((key) => `${key}${this.definition[key].argsToString()}: ${this.definition[key].toString()}`),
            modes: this.modes,
        });
    }
    /**
     * (experimental) Add a field to this Input Type.
     *
     * Input Types must have both fieldName and field options.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Input Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.InputType = InputType;
/**
 * (experimental) Union Types are abstract types that are similar to Interface Types, but they cannot to specify any common fields between types.
 *
 * Note that fields of a union type need to be object types. In other words,
 * you can't create a union type out of interfaces, other unions, or inputs.
 *
 * @experimental
 */
class UnionType {
    /**
     * @experimental
     */
    constructor(name, options) {
        this.name = name;
        this.definition = {};
        options.definition.map((def) => this.addField({ field: def.attribute() }));
    }
    /**
     * (experimental) Create a GraphQL Type representing this Union Type.
     *
     * @param options the options to configure this attribute.
     * @experimental
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * (experimental) Generate the string of this Union type.
     *
     * @experimental
     */
    toString() {
        // Return a string that appends all Object Types for this Union Type
        // i.e. 'union Example = example1 | example2'
        return Object.values(this.definition).reduce((acc, field) => `${acc} ${field.toString()} |`, `union ${this.name} =`).slice(0, -2);
    }
    /**
     * (experimental) Add a field to this Union Type.
     *
     * Input Types must have field options and the IField must be an Object Type.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        var _a;
        if (options.fieldName) {
            throw new Error('Union Types cannot be configured with the fieldName option. Use the field option instead.');
        }
        if (!options.field) {
            throw new Error('Union Types must be configured with the field option.');
        }
        if (options.field && !(options.field.intermediateType instanceof ObjectType)) {
            throw new Error('Fields for Union Types must be Object Types.');
        }
        this.definition[((_a = options.field) === null || _a === void 0 ? void 0 : _a.toString()) + 'id'] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.UnionType = UnionType;
/**
 * (experimental) Enum Types are abstract types that includes a set of fields that represent the strings this type can create.
 *
 * @experimental
 */
class EnumType {
    /**
     * @experimental
     */
    constructor(name, options) {
        this.name = name;
        this.definition = {};
        options.definition.map((fieldName) => this.addField({ fieldName }));
    }
    /**
     * (experimental) Create an GraphQL Type representing this Enum Type.
     *
     * @experimental
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * (experimental) Generate the string of this enum type.
     *
     * @experimental
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'enum',
            name: this.name,
            fields: Object.keys(this.definition),
            modes: this.modes,
        });
    }
    /**
     * (experimental) Add a field to this Enum Type.
     *
     * To add a field to this Enum Type, you must only configure
     * addField with the fieldName options.
     *
     * @param options the options to add a field.
     * @experimental
     */
    addField(options) {
        if (options.field) {
            throw new Error('Enum Type fields consist of strings. Use the fieldName option instead of the field option.');
        }
        if (!options.fieldName) {
            throw new Error('When adding a field to an Enum Type, you must configure the fieldName option.');
        }
        if (options.fieldName.indexOf(' ') > -1) {
            throw new Error(`Enum Type values cannot have whitespace. Received: ${options.fieldName}`);
        }
        this.definition[options.fieldName] = schema_field_1.GraphqlType.string();
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.EnumType = EnumType;
//# sourceMappingURL=data:application/json;base64,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