"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Type = exports.Directive = void 0;
const graphqlapi_1 = require("./graphqlapi");
/**
 * (experimental) Directives for types.
 *
 * i.e. @aws_iam or @aws_subscribe
 *
 * @experimental
 */
class Directive {
    constructor(statement, options) {
        this.statement = statement;
        this.mode = options === null || options === void 0 ? void 0 : options.mode;
        this.mutationFields = options === null || options === void 0 ? void 0 : options.mutationFields;
    }
    /**
     * (experimental) Add the @aws_iam directive.
     *
     * @experimental
     */
    static iam() {
        return new Directive('@aws_iam', { mode: graphqlapi_1.AuthorizationType.IAM });
    }
    /**
     * (experimental) Add the @aws_oidc directive.
     *
     * @experimental
     */
    static oidc() {
        return new Directive('@aws_oidc', { mode: graphqlapi_1.AuthorizationType.OIDC });
    }
    /**
     * (experimental) Add the @aws_api_key directive.
     *
     * @experimental
     */
    static apiKey() {
        return new Directive('@aws_api_key', { mode: graphqlapi_1.AuthorizationType.API_KEY });
    }
    /**
     * (experimental) Add the @aws_auth or @aws_cognito_user_pools directive.
     *
     * @param groups the groups to allow access to.
     * @experimental
     */
    static cognito(...groups) {
        if (groups.length === 0) {
            throw new Error(`Cognito authorization requires at least one Cognito group to be supplied. Received: ${groups.length}`);
        }
        // this function creates the cognito groups as a string (i.e. ["group1", "group2", "group3"])
        const stringify = (array) => {
            return array.reduce((acc, element) => `${acc}"${element}", `, '').slice(0, -2);
        };
        return new Directive(`@aws_auth(cognito_groups: [${stringify(groups)}])`, { mode: graphqlapi_1.AuthorizationType.USER_POOL });
    }
    /**
     * (experimental) Add the @aws_subscribe directive.
     *
     * Only use for top level Subscription type.
     *
     * @param mutations the mutation fields to link to.
     * @experimental
     */
    static subscribe(...mutations) {
        if (mutations.length === 0) {
            throw new Error(`Subscribe directive requires at least one mutation field to be supplied. Received: ${mutations.length}`);
        }
        // this function creates the subscribe directive as a string (i.e. ["mutation_field_1", "mutation_field_2"])
        const stringify = (array) => {
            return array.reduce((acc, mutation) => `${acc}"${mutation}", `, '').slice(0, -2);
        };
        return new Directive(`@aws_subscribe(mutations: [${stringify(mutations)}])`, { mutationFields: mutations });
    }
    /**
     * (experimental) Add a custom directive.
     *
     * @param statement - the directive statement to append.
     * @experimental
     */
    static custom(statement) {
        return new Directive(statement);
    }
    /**
     * Method called when the stringifying Directive for schema generation
     *
     * @param modes the authorization modes
     *
     * @internal
     */
    _bindToAuthModes(modes) {
        this.modes = modes;
        return this;
    }
    /**
     * (experimental) Generate the directive statement.
     *
     * @experimental
     */
    toString() {
        if (this.modes && this.mode && !this.modes.some((mode) => mode === this.mode)) {
            throw new Error(`No Authorization Type ${this.mode} declared in GraphQL Api.`);
        }
        if (this.mode === graphqlapi_1.AuthorizationType.USER_POOL && this.modes && this.modes.length > 1) {
            this.statement = this.statement.replace('@aws_auth', '@aws_cognito_user_pools');
        }
        return this.statement;
    }
}
exports.Directive = Directive;
/**
 * (experimental) Enum containing the Types that can be used to define ObjectTypes.
 *
 * @experimental
 */
var Type;
(function (Type) {
    Type["ID"] = "ID";
    Type["STRING"] = "String";
    Type["INT"] = "Int";
    Type["FLOAT"] = "Float";
    Type["BOOLEAN"] = "Boolean";
    Type["AWS_DATE"] = "AWSDate";
    Type["AWS_TIME"] = "AWSTime";
    Type["AWS_DATE_TIME"] = "AWSDateTime";
    Type["AWS_TIMESTAMP"] = "AWSTimestamp";
    Type["AWS_EMAIL"] = "AWSEmail";
    Type["AWS_JSON"] = "AWSJSON";
    Type["AWS_URL"] = "AWSURL";
    Type["AWS_PHONE"] = "AWSPhone";
    Type["AWS_IP_ADDRESS"] = "AWSIPAddress";
    Type["INTERMEDIATE"] = "INTERMEDIATE";
})(Type = exports.Type || (exports.Type = {}));
//# sourceMappingURL=data:application/json;base64,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