"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RdsDataSource = exports.LambdaDataSource = exports.HttpDataSource = exports.DynamoDbDataSource = exports.NoneDataSource = exports.BackedDataSource = exports.BaseDataSource = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appsync_generated_1 = require("./appsync.generated");
const resolver_1 = require("./resolver");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * (experimental) Abstract AppSync datasource implementation.
 *
 * Do not use directly but use subclasses for concrete datasources
 *
 * @experimental
 */
class BaseDataSource extends core_2.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props, extended) {
        var _a, _b;
        super(scope, id);
        if (extended.type !== 'NONE') {
            this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        }
        const name = (_a = props.name) !== null && _a !== void 0 ? _a : id;
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: name,
            description: props.description,
            serviceRoleArn: (_b = this.serviceRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            ...extended,
        });
        this.name = name;
        this.api = props.api;
    }
    /**
     * (experimental) creates a new resolver for this datasource and API using the given properties.
     *
     * @experimental
     */
    createResolver(props) {
        return new resolver_1.Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * (experimental) Abstract AppSync datasource implementation.
 *
 * Do not use directly but use subclasses for resource backed datasources
 *
 * @experimental
 */
class BackedDataSource extends BaseDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props, extended) {
        super(scope, id, props, extended);
        this.grantPrincipal = this.serviceRole;
    }
}
exports.BackedDataSource = BackedDataSource;
/**
 * (experimental) An AppSync dummy datasource.
 *
 * @experimental
 */
class NoneDataSource extends BaseDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'NONE',
        });
    }
}
exports.NoneDataSource = NoneDataSource;
/**
 * (experimental) An AppSync datasource backed by a DynamoDB table.
 *
 * @experimental
 */
class DynamoDbDataSource extends BackedDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this);
        }
        else {
            props.table.grantReadWriteData(this);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * (experimental) An AppSync datasource backed by a http endpoint.
 *
 * @experimental
 */
class HttpDataSource extends BaseDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        const authorizationConfig = props.authorizationConfig ? {
            authorizationType: 'AWS_IAM',
            awsIamConfig: props.authorizationConfig,
        } : undefined;
        super(scope, id, props, {
            type: 'HTTP',
            httpConfig: {
                endpoint: props.endpoint,
                authorizationConfig,
            },
        });
    }
}
exports.HttpDataSource = HttpDataSource;
/**
 * (experimental) An AppSync datasource backed by a Lambda function.
 *
 * @experimental
 */
class LambdaDataSource extends BackedDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this);
    }
}
exports.LambdaDataSource = LambdaDataSource;
/**
 * (experimental) An AppSync datasource backed by RDS.
 *
 * @experimental
 */
class RdsDataSource extends BackedDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'RELATIONAL_DATABASE',
            relationalDatabaseConfig: {
                rdsHttpEndpointConfig: {
                    awsRegion: props.databaseCluster.stack.region,
                    dbClusterIdentifier: props.databaseCluster.clusterIdentifier,
                    awsSecretStoreArn: props.secretStore.secretArn,
                },
                relationalDatabaseSourceType: 'RDS_HTTP_ENDPOINT',
            },
        });
        props.secretStore.grantRead(this);
        const clusterArn = core_1.Stack.of(this).formatArn({
            service: 'rds',
            resource: `cluster:${props.databaseCluster.clusterIdentifier}`,
        });
        // Change to grant with RDS grant becomes implemented
        aws_iam_1.Grant.addToPrincipal({
            grantee: this,
            actions: [
                'rds-data:DeleteItems',
                'rds-data:ExecuteSql',
                'rds-data:ExecuteStatement',
                'rds-data:GetItems',
                'rds-data:InsertItems',
                'rds-data:UpdateItems',
            ],
            resourceArns: [clusterArn, `${clusterArn}:*`],
            scope: this,
        });
    }
}
exports.RdsDataSource = RdsDataSource;
//# sourceMappingURL=data:application/json;base64,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