"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualService = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * (experimental) VirtualService represents a service inside an AppMesh.
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 * @experimental
 */
class VirtualService extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.stringValue({ produce: () => cdk.Names.uniqueId(this) }),
        });
        if (props.virtualNode && props.virtualRouter) {
            throw new Error('Must provide only one of virtualNode or virtualRouter for the provider');
        }
        this.mesh = props.mesh;
        // Check which provider to use node or router (or neither)
        if (props.virtualRouter) {
            this.virtualServiceProvider = this.addVirtualRouter(props.virtualRouter.virtualRouterName);
        }
        if (props.virtualNode) {
            this.virtualServiceProvider = this.addVirtualNode(props.virtualNode.virtualNodeName);
        }
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            virtualServiceName: this.physicalName,
            spec: {
                provider: this.virtualServiceProvider,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing VirtualService given an ARN.
     *
     * @experimental
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new ImportedVirtualService(scope, id, {
            virtualServiceArn,
        });
    }
    /**
     * (experimental) Import an existing VirtualService given mesh and service names.
     *
     * @experimental
     */
    static fromVirtualServiceName(scope, id, meshName, virtualServiceName) {
        return new ImportedVirtualService(scope, id, {
            meshName,
            virtualServiceName,
        });
    }
    addVirtualRouter(name) {
        return {
            virtualRouter: {
                virtualRouterName: name,
            },
        };
    }
    addVirtualNode(name) {
        return {
            virtualNode: {
                virtualNodeName: name,
            },
        };
    }
}
exports.VirtualService = VirtualService;
/**
 * Returns properties that allows a VirtualService to be imported
 */
class ImportedVirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualServiceArn) {
            this.virtualServiceArn = props.virtualServiceArn;
            this.virtualServiceName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualServiceArn).resourceName));
        }
        else if (props.virtualServiceName && props.meshName) {
            this.virtualServiceName = props.virtualServiceName;
            this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualService`,
                resourceName: this.virtualServiceName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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