"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualNode = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
const utils_1 = require("./private/utils");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualNodeBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        /**
         * @experimental
         */
        this.backends = new Array();
        /**
         * @experimental
         */
        this.listeners = new Array();
    }
    /**
     * (experimental) Add a Virtual Services that this node is expected to send outbound traffic to.
     *
     * @experimental
     */
    addBackends(...props) {
        for (const s of props) {
            this.backends.push({
                virtualService: {
                    virtualServiceName: s.virtualServiceName,
                },
            });
        }
    }
    /**
     * (experimental) Utility method to add an inbound listener for this virtual node.
     *
     * @experimental
     */
    addListeners(...listeners) {
        if (this.listeners.length + listeners.length > 1) {
            throw new Error('VirtualNode may have at most one listener');
        }
        for (const listener of listeners) {
            const portMapping = listener.portMapping || { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP };
            this.listeners.push({
                portMapping,
                healthCheck: renderHealthCheck(listener.healthCheck, portMapping),
            });
        }
    }
}
function renderHealthCheck(hc, pm) {
    var _a;
    if (hc === undefined) {
        return undefined;
    }
    if (hc.protocol === shared_interfaces_1.Protocol.TCP && hc.path) {
        throw new Error('The path property cannot be set with Protocol.TCP');
    }
    if (hc.protocol === shared_interfaces_1.Protocol.GRPC && hc.path) {
        throw new Error('The path property cannot be set with Protocol.GRPC');
    }
    const protocol = (_a = hc.protocol) !== null && _a !== void 0 ? _a : pm.protocol;
    const healthCheck = {
        healthyThreshold: hc.healthyThreshold || 2,
        intervalMillis: (hc.interval || cdk.Duration.seconds(5)).toMilliseconds(),
        path: hc.path || (protocol === shared_interfaces_1.Protocol.HTTP ? '/' : undefined),
        port: hc.port || pm.port,
        protocol: hc.protocol || pm.protocol,
        timeoutMillis: (hc.timeout || cdk.Duration.seconds(2)).toMilliseconds(),
        unhealthyThreshold: hc.unhealthyThreshold || 2,
    };
    utils_1.validateHealthChecks(healthCheck);
    return healthCheck;
}
/**
 * (experimental) VirtualNode represents a newly defined App Mesh VirtualNode.
 *
 * Any inbound traffic that your virtual node expects should be specified as a
 * listener. Any outbound traffic that your virtual node expects to reach
 * should be specified as a backend.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
 * @experimental
 */
class VirtualNode extends VirtualNodeBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.virtualNodeName || cdk.Lazy.stringValue({ produce: () => cdk.Names.uniqueId(this) }),
        });
        this.mesh = props.mesh;
        this.addBackends(...props.backends || []);
        this.addListeners(...props.listener ? [props.listener] : []);
        const accessLogging = (_a = props.accessLog) === null || _a === void 0 ? void 0 : _a.bind(this);
        const node = new appmesh_generated_1.CfnVirtualNode(this, 'Resource', {
            virtualNodeName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                backends: cdk.Lazy.anyValue({ produce: () => this.backends }, { omitEmptyArray: true }),
                listeners: cdk.Lazy.anyValue({ produce: () => this.listeners }, { omitEmptyArray: true }),
                serviceDiscovery: {
                    dns: props.dnsHostName !== undefined ? { hostname: props.dnsHostName } : undefined,
                    awsCloudMap: props.cloudMapService !== undefined ? {
                        serviceName: props.cloudMapService.serviceName,
                        namespaceName: props.cloudMapService.namespace.namespaceName,
                        attributes: renderAttributes(props.cloudMapServiceInstanceAttributes),
                    } : undefined,
                },
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualNodeAccessLog,
                } : undefined,
            },
        });
        this.virtualNodeName = this.getResourceNameAttribute(node.attrVirtualNodeName);
        this.virtualNodeArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualNode`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing VirtualNode given an ARN.
     *
     * @experimental
     */
    static fromVirtualNodeArn(scope, id, virtualNodeArn) {
        return new ImportedVirtualNode(scope, id, { virtualNodeArn });
    }
    /**
     * (experimental) Import an existing VirtualNode given its name.
     *
     * @experimental
     */
    static fromVirtualNodeName(scope, id, meshName, virtualNodeName) {
        return new ImportedVirtualNode(scope, id, {
            meshName,
            virtualNodeName,
        });
    }
}
exports.VirtualNode = VirtualNode;
function renderAttributes(attrs) {
    if (attrs === undefined) {
        return undefined;
    }
    return Object.entries(attrs).map(([key, value]) => ({ key, value }));
}
/**
 * Used to import a VirtualNode and read its properties
 */
class ImportedVirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualNodeArn) {
            this.virtualNodeArn = props.virtualNodeArn;
            this.virtualNodeName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualNodeArn).resourceName));
        }
        else if (props.virtualNodeName && props.meshName) {
            this.virtualNodeName = props.virtualNodeName;
            this.virtualNodeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualNode`,
                resourceName: this.virtualNodeName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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