"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualGateway = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
const gateway_route_1 = require("./gateway-route");
const mesh_1 = require("./mesh");
const virtual_gateway_listener_1 = require("./virtual-gateway-listener");
class VirtualGatewayBase extends cdk.Resource {
    /**
     * (experimental) Utility method to add a new GatewayRoute to the VirtualGateway.
     *
     * @experimental
     */
    addGatewayRoute(id, props) {
        return new gateway_route_1.GatewayRoute(this, id, {
            ...props,
            virtualGateway: this,
        });
    }
}
/**
 * (experimental) VirtualGateway represents a newly defined App Mesh Virtual Gateway.
 *
 * A virtual gateway allows resources that are outside of your mesh to communicate to resources that
 * are inside of your mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
 * @experimental
 */
class VirtualGateway extends VirtualGatewayBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.virtualGatewayName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        /**
         * @experimental
         */
        this.listeners = new Array();
        this.mesh = props.mesh;
        if (!props.listeners) {
            // Use listener default of http listener port 8080 if no listener is defined
            this.listeners.push(virtual_gateway_listener_1.VirtualGatewayListener.httpGatewayListener().bind(this));
        }
        else {
            props.listeners.forEach(listener => this.listeners.push(listener.bind(this)));
        }
        const accessLogging = (_a = props.accessLog) === null || _a === void 0 ? void 0 : _a.bind(this);
        const node = new appmesh_generated_1.CfnVirtualGateway(this, 'Resource', {
            virtualGatewayName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                listeners: this.listeners.map(listener => listener.listener),
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualGatewayAccessLog,
                } : undefined,
            },
        });
        this.virtualGatewayName = this.getResourceNameAttribute(node.attrVirtualGatewayName);
        this.virtualGatewayArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualGateway`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing VirtualGateway given an ARN.
     *
     * @experimental
     */
    static fromVirtualGatewayArn(scope, id, virtualGatewayArn) {
        return new ImportedVirtualGateway(scope, id, { virtualGatewayArn });
    }
}
exports.VirtualGateway = VirtualGateway;
/**
 * Used to import a VirtualGateway and read its properties
 */
class ImportedVirtualGateway extends VirtualGatewayBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualGatewayArn) {
            const meshName = cdk.Fn.select(0, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualGatewayArn).resourceName));
            this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', meshName);
            this.virtualGatewayArn = props.virtualGatewayArn;
            this.virtualGatewayName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualGatewayArn).resourceName));
        }
        else {
            throw new Error('Need virtualGatewayArn');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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