"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualGatewayListener = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const utils_1 = require("./private/utils");
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * (experimental) Represents the properties needed to define listeners for a VirtualGateway.
 *
 * @experimental
 */
class VirtualGatewayListener {
    /**
     * (experimental) Returns an HTTP Listener for a VirtualGateway.
     *
     * @experimental
     */
    static httpGatewayListener(props = {}) {
        return new HttpGatewayListener(props);
    }
    /**
     * (experimental) Returns an HTTP2 Listener for a VirtualGateway.
     *
     * @experimental
     */
    static http2GatewayListener(props = {}) {
        return new Http2GatewayListener(props);
    }
    /**
     * (experimental) Returns a GRPC Listener for a VirtualGateway.
     *
     * @experimental
     */
    static grpcGatewayListener(props = {}) {
        return new GrpcGatewayListener(props);
    }
    /**
     * @experimental
     */
    renderHealthCheck(hc) {
        if (hc.protocol === shared_interfaces_1.Protocol.TCP) {
            throw new Error('TCP health checks are not permitted for gateway listeners');
        }
        if (hc.protocol === shared_interfaces_1.Protocol.GRPC && hc.path) {
            throw new Error('The path property cannot be set with Protocol.GRPC');
        }
        const protocol = hc.protocol ? hc.protocol : this.protocol;
        const healthCheck = {
            healthyThreshold: hc.healthyThreshold || 2,
            intervalMillis: (hc.interval || cdk.Duration.seconds(5)).toMilliseconds(),
            path: hc.path || ((protocol === shared_interfaces_1.Protocol.HTTP || protocol === shared_interfaces_1.Protocol.HTTP2) ? '/' : undefined),
            port: hc.port || this.port,
            protocol: hc.protocol || this.protocol,
            timeoutMillis: (hc.timeout || cdk.Duration.seconds(2)).toMilliseconds(),
            unhealthyThreshold: hc.unhealthyThreshold || 2,
        };
        utils_1.validateHealthChecks(healthCheck);
        return healthCheck;
    }
}
exports.VirtualGatewayListener = VirtualGatewayListener;
/**
 * Represents the properties needed to define an HTTP Listener for a VirtualGateway
 */
class HttpGatewayListener extends VirtualGatewayListener {
    constructor(props = {}) {
        super();
        /**
         * Protocol the listener implements
         */
        this.protocol = shared_interfaces_1.Protocol.HTTP;
        this.port = props.port ? props.port : 8080;
        this.healthCheck = props.healthCheck;
    }
    /**
     * Called when the GatewayListener type is initialized. Can be used to enforce
     * mutual exclusivity
     */
    bind(_scope) {
        return {
            listener: {
                portMapping: {
                    port: this.port,
                    protocol: this.protocol,
                },
                healthCheck: this.healthCheck ? this.renderHealthCheck(this.healthCheck) : undefined,
            },
        };
    }
}
/**
* Represents the properties needed to define an HTTP2 Listener for a VirtualGateway
*/
class Http2GatewayListener extends HttpGatewayListener {
    constructor(props = {}) {
        super(props);
        this.protocol = shared_interfaces_1.Protocol.HTTP2;
    }
}
/**
 * Represents the properties needed to define a GRPC Listener for Virtual Gateway
 */
class GrpcGatewayListener extends VirtualGatewayListener {
    constructor(props = {}) {
        super();
        /**
         * Protocol the listener implements
         */
        this.protocol = shared_interfaces_1.Protocol.GRPC;
        this.port = props.port ? props.port : 8080;
        this.healthCheck = props.healthCheck;
    }
    /**
     * Called when the GatewayListener type is initialized. Can be used to enforce
     * mutual exclusivity
     */
    bind(_scope) {
        return {
            listener: {
                portMapping: {
                    port: this.port,
                    protocol: shared_interfaces_1.Protocol.GRPC,
                },
                healthCheck: this.healthCheck ? this.renderHealthCheck(this.healthCheck) : undefined,
            },
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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