"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route = exports.RouteType = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * (experimental) Type of route.
 *
 * @experimental
 */
var RouteType;
(function (RouteType) {
    RouteType["HTTP"] = "http";
    RouteType["TCP"] = "tcp";
})(RouteType = exports.RouteType || (exports.RouteType = {}));
/**
 * (experimental) Route represents a new or existing route attached to a VirtualRouter and Mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
 * @experimental
 */
class Route extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.routeName || cdk.Lazy.stringValue({ produce: () => cdk.Names.uniqueId(this) }),
        });
        this.weightedTargets = new Array();
        this.virtualRouter = props.virtualRouter;
        const routeType = props.routeType !== undefined ? props.routeType :
            props.prefix !== undefined ? RouteType.HTTP :
                RouteType.TCP;
        if (routeType === RouteType.HTTP) {
            this.httpRoute = this.renderHttpRoute(props);
        }
        else {
            this.tcpRoute = this.renderTcpRoute(props);
        }
        const route = new appmesh_generated_1.CfnRoute(this, 'Resource', {
            routeName: this.physicalName,
            meshName: this.virtualRouter.mesh.meshName,
            virtualRouterName: this.virtualRouter.virtualRouterName,
            spec: {
                tcpRoute: this.tcpRoute,
                httpRoute: this.httpRoute,
            },
        });
        this.routeName = this.getResourceNameAttribute(route.attrRouteName);
        this.routeArn = this.getResourceArnAttribute(route.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter/${props.virtualRouter.virtualRouterName}/route`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing route given an ARN.
     *
     * @experimental
     */
    static fromRouteArn(scope, id, routeArn) {
        return new ImportedRoute(scope, id, { routeArn });
    }
    /**
     * (experimental) Import an existing route given its name.
     *
     * @experimental
     */
    static fromRouteName(scope, id, meshName, virtualRouterName, routeName) {
        return new ImportedRoute(scope, id, { meshName, virtualRouterName, routeName });
    }
    /**
     * Utility method to add weighted route targets to an existing route
     */
    renderWeightedTargets(props) {
        for (const t of props) {
            this.weightedTargets.push({
                virtualNode: t.virtualNode.virtualNodeName,
                weight: t.weight || 1,
            });
        }
        return this.weightedTargets;
    }
    renderHttpRoute(props) {
        return {
            match: {
                prefix: props.prefix || '/',
            },
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
    renderTcpRoute(props) {
        return {
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
}
exports.Route = Route;
/**
 * Represents and imported IRoute
 */
class ImportedRoute extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.routeArn) {
            this.routeArn = props.routeArn;
            this.routeName = cdk.Fn.select(4, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.routeArn).resourceName));
        }
        else if (props.routeName && props.meshName && props.virtualRouterName) {
            this.routeName = props.routeName;
            this.routeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter/${props.virtualRouterName}/route`,
                resourceName: this.routeName,
            });
        }
        else {
            throw new Error('Need either arn or three names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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