"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateHealthChecks = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Validates health check properties, throws an error if they are misconfigured.
 *
 * @param healthCheck Healthcheck property from a Virtual Node or Virtual Gateway
 */
function validateHealthChecks(healthCheck) {
    Object.keys(healthCheck)
        .filter((key) => HEALTH_CHECK_PROPERTY_THRESHOLDS[key] &&
        typeof healthCheck[key] === 'number' &&
        !cdk.Token.isUnresolved(healthCheck[key])).map((key) => {
        const [min, max] = HEALTH_CHECK_PROPERTY_THRESHOLDS[key];
        const value = healthCheck[key];
        if (value < min) {
            throw new Error(`The value of '${key}' is below the minimum threshold (expected >=${min}, got ${value})`);
        }
        if (value > max) {
            throw new Error(`The value of '${key}' is above the maximum threshold (expected <=${max}, got ${value})`);
        }
    });
}
exports.validateHealthChecks = validateHealthChecks;
/**
 * Minimum and maximum thresholds for HeathCheck numeric properties
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_HealthCheckPolicy.html
 */
const HEALTH_CHECK_PROPERTY_THRESHOLDS = {
    healthyThreshold: [2, 10],
    intervalMillis: [5000, 300000],
    port: [1, 65535],
    timeoutMillis: [2000, 60000],
    unhealthyThreshold: [2, 10],
};
//# sourceMappingURL=data:application/json;base64,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