"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mesh = exports.MeshFilterType = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
const virtual_gateway_1 = require("./virtual-gateway");
const virtual_node_1 = require("./virtual-node");
const virtual_router_1 = require("./virtual-router");
const virtual_service_1 = require("./virtual-service");
/**
 * (experimental) A utility enum defined for the egressFilter type property, the default of DROP_ALL, allows traffic only to other resources inside the mesh, or API calls to amazon resources.
 *
 * @default DROP_ALL
 * @experimental
 */
var MeshFilterType;
(function (MeshFilterType) {
    MeshFilterType["ALLOW_ALL"] = "ALLOW_ALL";
    MeshFilterType["DROP_ALL"] = "DROP_ALL";
})(MeshFilterType = exports.MeshFilterType || (exports.MeshFilterType = {}));
/**
 * Represents a new or imported AppMesh mesh
 */
class MeshBase extends cdk.Resource {
    /**
     * (experimental) Adds a VirtualRouter to the Mesh with the given id and props.
     *
     * @experimental
     */
    addVirtualRouter(id, props = {}) {
        return new virtual_router_1.VirtualRouter(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * (experimental) Adds a VirtualService with the given id.
     *
     * @experimental
     */
    addVirtualService(id, props = {}) {
        return new virtual_service_1.VirtualService(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * (experimental) Adds a VirtualNode to the Mesh.
     *
     * @experimental
     */
    addVirtualNode(id, props = {}) {
        return new virtual_node_1.VirtualNode(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * (experimental) Adds a VirtualGateway to the Mesh.
     *
     * @experimental
     */
    addVirtualGateway(id, props) {
        return new virtual_gateway_1.VirtualGateway(this, id, {
            ...props,
            mesh: this,
        });
    }
}
/**
 * (experimental) Define a new AppMesh mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html
 * @experimental
 */
class Mesh extends MeshBase {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.meshName || cdk.Lazy.stringValue({ produce: () => cdk.Names.uniqueId(this) }),
        });
        const mesh = new appmesh_generated_1.CfnMesh(this, 'Resource', {
            meshName: this.physicalName,
            spec: {
                egressFilter: props.egressFilter ? {
                    type: props.egressFilter,
                } : undefined,
            },
        });
        this.meshName = this.getResourceNameAttribute(mesh.attrMeshName);
        this.meshArn = this.getResourceArnAttribute(mesh.ref, {
            service: 'appmesh',
            resource: 'mesh',
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing mesh by arn.
     *
     * @experimental
     */
    static fromMeshArn(scope, id, meshArn) {
        const parts = cdk.Stack.of(scope).parseArn(meshArn);
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = parts.resourceName || '';
                this.meshArn = meshArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Import an existing mesh by name.
     *
     * @experimental
     */
    static fromMeshName(scope, id, meshName) {
        const arn = cdk.Stack.of(scope).formatArn({
            service: 'appmesh',
            resource: 'mesh',
            resourceName: meshName,
        });
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = meshName;
                this.meshArn = arn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Mesh = Mesh;
//# sourceMappingURL=data:application/json;base64,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