"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpApi = void 0;
const aws_cloudwatch_1 = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const route_1 = require("./route");
const stage_1 = require("./stage");
const vpc_link_1 = require("./vpc-link");
class HttpApiBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.vpcLinks = {};
    }
    /**
     * (experimental) Return the given named metric for this HTTP Api Gateway.
     *
     * @experimental
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ApiGateway',
            metricName,
            dimensions: { ApiId: this.httpApiId },
            ...props,
        }).attachTo(this);
    }
    /**
     * (experimental) Metric for the number of client-side errors captured in a given period.
     *
     * @experimental
     */
    metricClientError(props) {
        return this.metric('4XXError', { statistic: 'Sum', ...props });
    }
    /**
     * (experimental) Metric for the number of server-side errors captured in a given period.
     *
     * @experimental
     */
    metricServerError(props) {
        return this.metric('5XXError', { statistic: 'Sum', ...props });
    }
    /**
     * (experimental) Metric for the amount of data processed in bytes.
     *
     * @experimental
     */
    metricDataProcessed(props) {
        return this.metric('DataProcessed', { statistic: 'Sum', ...props });
    }
    /**
     * (experimental) Metric for the total number API requests in a given period.
     *
     * @experimental
     */
    metricCount(props) {
        return this.metric('Count', { statistic: 'SampleCount', ...props });
    }
    /**
     * (experimental) Metric for the time between when API Gateway relays a request to the backend and when it receives a response from the backend.
     *
     * @experimental
     */
    metricIntegrationLatency(props) {
        return this.metric('IntegrationLatency', props);
    }
    /**
     * (experimental) The time between when API Gateway receives a request from a client and when it returns a response to the client.
     *
     * The latency includes the integration latency and other API Gateway overhead.
     *
     * @experimental
     */
    metricLatency(props) {
        return this.metric('Latency', props);
    }
    /**
     * (experimental) Add a new VpcLink.
     *
     * @experimental
     */
    addVpcLink(options) {
        const { vpcId } = options.vpc;
        if (vpcId in this.vpcLinks) {
            return this.vpcLinks[vpcId];
        }
        const count = Object.keys(this.vpcLinks).length + 1;
        const vpcLink = new vpc_link_1.VpcLink(this, `VpcLink-${count}`, options);
        this.vpcLinks[vpcId] = vpcLink;
        return vpcLink;
    }
}
/**
 * (experimental) Create a new API Gateway HTTP API endpoint.
 *
 * @experimental
 * @resource AWS::ApiGatewayV2::Api
 */
class HttpApi extends HttpApiBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.httpApiName = (_a = props === null || props === void 0 ? void 0 : props.apiName) !== null && _a !== void 0 ? _a : id;
        let corsConfiguration;
        if (props === null || props === void 0 ? void 0 : props.corsPreflight) {
            const cors = props.corsPreflight;
            if (cors.allowOrigins && cors.allowOrigins.includes('*') && cors.allowCredentials) {
                throw new Error("CORS preflight - allowCredentials is not supported when allowOrigin is '*'");
            }
            const { allowCredentials, allowHeaders, allowMethods, allowOrigins, exposeHeaders, maxAge, } = props.corsPreflight;
            corsConfiguration = {
                allowCredentials,
                allowHeaders,
                allowMethods,
                allowOrigins,
                exposeHeaders,
                maxAge: maxAge === null || maxAge === void 0 ? void 0 : maxAge.toSeconds(),
            };
        }
        const apiProps = {
            name: this.httpApiName,
            protocolType: 'HTTP',
            corsConfiguration,
            description: props === null || props === void 0 ? void 0 : props.description,
        };
        const resource = new apigatewayv2_generated_1.CfnApi(this, 'Resource', apiProps);
        this.httpApiId = resource.ref;
        this.apiEndpoint = resource.attrApiEndpoint;
        if (props === null || props === void 0 ? void 0 : props.defaultIntegration) {
            new route_1.HttpRoute(this, 'DefaultRoute', {
                httpApi: this,
                routeKey: route_1.HttpRouteKey.DEFAULT,
                integration: props.defaultIntegration,
            });
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === undefined || props.createDefaultStage === true) {
            this.defaultStage = new stage_1.HttpStage(this, 'DefaultStage', {
                httpApi: this,
                autoDeploy: true,
                domainMapping: props === null || props === void 0 ? void 0 : props.defaultDomainMapping,
            });
            // to ensure the domain is ready before creating the default stage
            if (props === null || props === void 0 ? void 0 : props.defaultDomainMapping) {
                this.defaultStage.node.addDependency(props.defaultDomainMapping.domainName);
            }
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === false && props.defaultDomainMapping) {
            throw new Error('defaultDomainMapping not supported with createDefaultStage disabled');
        }
    }
    /**
     * (experimental) Import an existing HTTP API into this CDK app.
     *
     * @experimental
     */
    static fromApiId(scope, id, httpApiId) {
        class Import extends HttpApiBase {
            constructor() {
                super(...arguments);
                this.httpApiId = httpApiId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Get the URL to the default stage of this API.
     *
     * Returns `undefined` if `createDefaultStage` is unset.
     *
     * @experimental
     */
    get url() {
        return this.defaultStage ? this.defaultStage.url : undefined;
    }
    /**
     * (experimental) Add a new stage.
     *
     * @experimental
     */
    addStage(id, options) {
        const stage = new stage_1.HttpStage(this, id, {
            httpApi: this,
            ...options,
        });
        return stage;
    }
    /**
     * (experimental) Add multiple routes that uses the same configuration.
     *
     * The routes all go to the same path, but for different
     * methods.
     *
     * @experimental
     */
    addRoutes(options) {
        var _a;
        const methods = (_a = options.methods) !== null && _a !== void 0 ? _a : [route_1.HttpMethod.ANY];
        return methods.map((method) => new route_1.HttpRoute(this, `${method}${options.path}`, {
            httpApi: this,
            routeKey: route_1.HttpRouteKey.with(options.path, method),
            integration: options.integration,
        }));
    }
}
exports.HttpApi = HttpApi;
//# sourceMappingURL=data:application/json;base64,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