"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointType = exports.ApiKeySourceType = exports.RestApi = exports.SpecRestApi = exports.RestApiBase = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const api_key_1 = require("./api-key");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const gateway_response_1 = require("./gateway-response");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
const RESTAPI_SYMBOL = Symbol.for('@aws-cdk/aws-apigateway.RestApiBase');
/**
 * (experimental) Base implementation that are common to various implementations of IRestApi.
 *
 * @experimental
 */
class RestApiBase extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        this.restApiName = (_a = props.restApiName) !== null && _a !== void 0 ? _a : id;
        Object.defineProperty(this, RESTAPI_SYMBOL, { value: true });
    }
    /**
     * Checks if the given object is an instance of RestApiBase.
     * @internal
     */
    static _isRestApiBase(x) {
        return x !== null && typeof (x) === 'object' && RESTAPI_SYMBOL in x;
    }
    /**
     * (experimental) API Gateway deployment that represents the latest changes of the API.
     *
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     *
     * @experimental
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * (experimental) The first domain name mapped to this API, if defined through the `domainName` configuration prop, or added via `addDomainName`.
     *
     * @experimental
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * (experimental) Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     *
     * @experimental
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * (experimental) Defines an API Gateway domain name and maps it to this API.
     *
     * @param id The construct id.
     * @param options custom domain options.
     * @experimental
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this,
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * (experimental) Adds a usage plan.
     *
     * @experimental
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    /**
     * (experimental) Gets the "execute-api" ARN.
     *
     * @experimental
     */
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`,
        });
    }
    /**
     * (experimental) Adds a new gateway response.
     *
     * @experimental
     */
    addGatewayResponse(id, options) {
        return new gateway_response_1.GatewayResponse(this, id, {
            restApi: this,
            ...options,
        });
    }
    /**
     * (experimental) Returns the given named metric for this API.
     *
     * @experimental
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApiGateway',
            metricName,
            dimensions: { ApiName: this.restApiName },
            ...props,
        });
    }
    /**
     * (experimental) Metric for the number of client-side errors captured in a given period.
     *
     * @default - sum over 5 minutes
     * @experimental
     */
    metricClientError(props) {
        return this.metric('4XXError', { statistic: 'Sum', ...props });
    }
    /**
     * (experimental) Metric for the number of server-side errors captured in a given period.
     *
     * @default - sum over 5 minutes
     * @experimental
     */
    metricServerError(props) {
        return this.metric('5XXError', { statistic: 'Sum', ...props });
    }
    /**
     * (experimental) Metric for the number of requests served from the API cache in a given period.
     *
     * @default - sum over 5 minutes
     * @experimental
     */
    metricCacheHitCount(props) {
        return this.metric('CacheHitCount', { statistic: 'Sum', ...props });
    }
    /**
     * (experimental) Metric for the number of requests served from the backend in a given period, when API caching is enabled.
     *
     * @default - sum over 5 minutes
     * @experimental
     */
    metricCacheMissCount(props) {
        return this.metric('CacheMissCount', { statistic: 'Sum', ...props });
    }
    /**
     * (experimental) Metric for the total number API requests in a given period.
     *
     * @default - SampleCount over 5 minutes
     * @experimental
     */
    metricCount(props) {
        return this.metric('Count', { statistic: 'SampleCount', ...props });
    }
    /**
     * (experimental) Metric for the time between when API Gateway relays a request to the backend and when it receives a response from the backend.
     *
     * @default - no statistic
     * @experimental
     */
    metricIntegrationLatency(props) {
        return this.metric('IntegrationLatency', props);
    }
    /**
     * (experimental) The time between when API Gateway receives a request from a client and when it returns a response to the client.
     *
     * The latency includes the integration latency and other API Gateway overhead.
     *
     * @default - no statistic
     * @experimental
     */
    metricLatency(props) {
        return this.metric('Latency', props);
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        ignore(method);
    }
    /**
     * Associates a Deployment resource with this REST API.
     *
     * @internal
     */
    _attachDeployment(deployment) {
        ignore(deployment);
    }
    /**
     * @experimental
     */
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn,
        });
        resource.node.addDependency(apiResource);
    }
    /**
     * @experimental
     */
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments,
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions,
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error('Cannot set \'deployOptions\' if \'deploy\' is disabled');
            }
        }
    }
    /**
     * @internal
     */
    _configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId),
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
}
exports.RestApiBase = RestApiBase;
/**
 * (experimental) Represents a REST API in Amazon API Gateway, created with an OpenAPI specification.
 *
 * Some properties normally accessible on @see {@link RestApi} - such as the description -
 * must be declared in the specification. All Resources and Methods need to be defined as
 * part of the OpenAPI specification file, and cannot be added via the CDK.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @experimental
 * @resource AWS::ApiGateway::RestApi
 */
class SpecRestApi extends RestApiBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        const apiDefConfig = props.apiDefinition.bind(this);
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.restApiName,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            body: apiDefConfig.inlineDefinition ? apiDefConfig.inlineDefinition : undefined,
            bodyS3Location: apiDefConfig.inlineDefinition ? undefined : apiDefConfig.s3Location,
            endpointConfiguration: this._configureEndpoints(props),
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.restApiRootResourceId = resource.attrRootResourceId;
        this.root = new RootResource(this, {}, this.restApiRootResourceId);
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
    }
}
exports.SpecRestApi = SpecRestApi;
/**
 * (experimental) Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @experimental
 */
class RestApi extends RestApiBase {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        /**
         * (experimental) The list of methods bound to this RestApi.
         *
         * @experimental
         */
        this.methods = new Array();
        /**
         * This list of deployments bound to this RestApi
         */
        this.deployments = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.restApiName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this._configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
    }
    /**
     * (experimental) Import an existing RestApi.
     *
     * @experimental
     */
    static fromRestApiId(scope, id, restApiId) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
            get root() {
                throw new Error('root is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
            get restApiRootResourceId() {
                throw new Error('restApiRootResourceId is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Import an existing RestApi that can be configured with additional Methods and Resources.
     *
     * @experimental
     */
    static fromRestApiAttributes(scope, id, attrs) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = attrs.restApiId;
                this.restApiRootResourceId = attrs.rootResourceId;
                this.root = new RootResource(this, {}, this.restApiRootResourceId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) The deployed root URL of this REST API.
     *
     * @experimental
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * (experimental) Add an ApiKey.
     *
     * @experimental
     */
    addApiKey(id, options) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this],
            ...options,
        });
    }
    /**
     * (experimental) Adds a new model.
     *
     * @experimental
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * (experimental) Adds a new request validator.
     *
     * @experimental
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
        // add this method as a dependency to all deployments defined for this api
        // when additional deployments are added, _attachDeployment is called and
        // this method will be added there.
        for (const dep of this.deployments) {
            dep._addMethodDependency(method);
        }
    }
    /**
     * Attaches a deployment to this REST API.
     *
     * @internal
     */
    _attachDeployment(deployment) {
        this.deployments.push(deployment);
        // add all methods that were already defined as dependencies of this
        // deployment when additional methods are added, _attachMethod is called and
        // it will be added as a dependency to this deployment.
        for (const method of this.methods) {
            deployment._addMethodDependency(method);
        }
    }
    /**
     * (experimental) Performs validation of the REST API.
     *
     * @experimental
     */
    validate() {
        if (this.methods.length === 0) {
            return ["The REST API doesn't contain any methods"];
        }
        return [];
    }
}
exports.RestApi = RestApi;
/**
 * @experimental
 */
var ApiKeySourceType;
(function (ApiKeySourceType) {
    ApiKeySourceType["HEADER"] = "HEADER";
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
/**
 * @experimental
 */
var EndpointType;
(function (EndpointType) {
    EndpointType["EDGE"] = "EDGE";
    EndpointType["REGIONAL"] = "REGIONAL";
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.api = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (api instanceof RestApi) {
            this._restApi = api;
        }
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * Get the RestApi associated with this Resource.
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this._restApi) {
            throw new Error('RestApi is not available on Resource not connected to an instance of RestApi. Use `api` instead');
        }
        return this._restApi;
    }
}
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzdGFwaS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJlc3RhcGkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsbURBQW1ELENBQUMsMERBQTBEO0FBRTlHLHFDQUFxQyxDQUFDLG1EQUFtRDtBQUN6RixxQ0FBb0YsQ0FBQyxnREFBZ0Q7QUFHckksdUNBQTJEO0FBQzNELGlFQUFnRTtBQUVoRSw2Q0FBMEM7QUFDMUMsK0NBQThEO0FBQzlELHlEQUE2RTtBQUc3RSxtQ0FBOEM7QUFDOUMseURBQStFO0FBQy9FLHlDQUFzRTtBQUN0RSxtQ0FBOEM7QUFDOUMsNkNBQXlEO0FBQ3pELE1BQU0sY0FBYyxHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUMscUNBQXFDLENBQUMsQ0FBQzs7Ozs7O0FBOE16RSxNQUFzQixXQUFZLFNBQVEsZUFBUTs7OztJQW1EOUMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUEwQixFQUFFOztRQUNsRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLElBQUksQ0FBQyxXQUFXLFNBQUcsS0FBSyxDQUFDLFdBQVcsbUNBQUksRUFBRSxDQUFDO1FBQzNDLE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUF0REQ7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLGNBQWMsQ0FBQyxDQUFNO1FBQy9CLE9BQU8sQ0FBQyxLQUFLLElBQUksSUFBSSxPQUFPLENBQUMsQ0FBQyxDQUFDLEtBQUssUUFBUSxJQUFJLGNBQWMsSUFBSSxDQUFDLENBQUM7SUFDeEUsQ0FBQzs7Ozs7Ozs7O0lBTUQsSUFBVyxnQkFBZ0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsaUJBQWlCLENBQUM7SUFDbEMsQ0FBQzs7Ozs7O0lBS0QsSUFBVyxVQUFVO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLFdBQVcsQ0FBQztJQUM1QixDQUFDOzs7Ozs7OztJQXVDTSxVQUFVLENBQUMsT0FBZSxHQUFHO1FBQ2hDLElBQUksQ0FBQyxJQUFJLENBQUMsZUFBZSxFQUFFO1lBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsb0hBQW9ILENBQUMsQ0FBQztTQUN6STtRQUNELE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDakQsQ0FBQzs7Ozs7Ozs7SUFNTSxhQUFhLENBQUMsRUFBVSxFQUFFLE9BQTBCO1FBQ3ZELE1BQU0sVUFBVSxHQUFHLElBQUksd0JBQVUsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ3hDLEdBQUcsT0FBTztZQUNWLE9BQU8sRUFBRSxJQUFJO1NBQ2hCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ25CLElBQUksQ0FBQyxXQUFXLEdBQUcsVUFBVSxDQUFDO1NBQ2pDO1FBQ0QsT0FBTyxVQUFVLENBQUM7SUFDdEIsQ0FBQzs7Ozs7O0lBSU0sWUFBWSxDQUFDLEVBQVUsRUFBRSxRQUF3QixFQUFFO1FBQ3RELE9BQU8sSUFBSSxzQkFBUyxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDMUMsQ0FBQzs7Ozs7O0lBQ00sZ0JBQWdCLENBQUMsU0FBaUIsR0FBRyxFQUFFLE9BQWUsSUFBSSxFQUFFLFFBQWdCLEdBQUc7UUFDbEYsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQ0FBa0MsSUFBSSxHQUFHLENBQUMsQ0FBQztTQUM5RDtRQUNELElBQUksTUFBTSxDQUFDLFdBQVcsRUFBRSxLQUFLLEtBQUssRUFBRTtZQUNoQyxNQUFNLEdBQUcsR0FBRyxDQUFDO1NBQ2hCO1FBQ0QsT0FBTyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUM1QixPQUFPLEVBQUUsYUFBYTtZQUN0QixRQUFRLEVBQUUsSUFBSSxDQUFDLFNBQVM7WUFDeEIsR0FBRyxFQUFFLEdBQUc7WUFDUixZQUFZLEVBQUUsR0FBRyxLQUFLLElBQUksTUFBTSxHQUFHLElBQUksRUFBRTtTQUM1QyxDQUFDLENBQUM7SUFDUCxDQUFDOzs7Ozs7SUFJTSxrQkFBa0IsQ0FBQyxFQUFVLEVBQUUsT0FBK0I7UUFDakUsT0FBTyxJQUFJLGtDQUFlLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUNqQyxPQUFPLEVBQUUsSUFBSTtZQUNiLEdBQUcsT0FBTztTQUNiLENBQUMsQ0FBQztJQUNQLENBQUM7Ozs7OztJQUlNLE1BQU0sQ0FBQyxVQUFrQixFQUFFLEtBQWdDO1FBQzlELE9BQU8sSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDO1lBQ3pCLFNBQVMsRUFBRSxnQkFBZ0I7WUFDM0IsVUFBVTtZQUNWLFVBQVUsRUFBRSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ3pDLEdBQUcsS0FBSztTQUNYLENBQUMsQ0FBQztJQUNQLENBQUM7Ozs7Ozs7SUFNTSxpQkFBaUIsQ0FBQyxLQUFnQztRQUNyRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxHQUFHLEtBQUssRUFBRSxDQUFDLENBQUM7SUFDbkUsQ0FBQzs7Ozs7OztJQU1NLGlCQUFpQixDQUFDLEtBQWdDO1FBQ3JELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLEdBQUcsS0FBSyxFQUFFLENBQUMsQ0FBQztJQUNuRSxDQUFDOzs7Ozs7O0lBTU0sbUJBQW1CLENBQUMsS0FBZ0M7UUFDdkQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGVBQWUsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQ3hFLENBQUM7Ozs7Ozs7SUFPTSxvQkFBb0IsQ0FBQyxLQUFnQztRQUN4RCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLEVBQUUsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLEdBQUcsS0FBSyxFQUFFLENBQUMsQ0FBQztJQUN6RSxDQUFDOzs7Ozs7O0lBTU0sV0FBVyxDQUFDLEtBQWdDO1FBQy9DLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLEVBQUUsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLEdBQUcsS0FBSyxFQUFFLENBQUMsQ0FBQztJQUN4RSxDQUFDOzs7Ozs7O0lBT00sd0JBQXdCLENBQUMsS0FBZ0M7UUFDNUQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLG9CQUFvQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3BELENBQUM7Ozs7Ozs7OztJQVFNLGFBQWEsQ0FBQyxLQUFnQztRQUNqRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFDRDs7Ozs7T0FLRztJQUNJLGFBQWEsQ0FBQyxNQUFjO1FBQy9CLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUNuQixDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLGlCQUFpQixDQUFDLFVBQXNCO1FBQzNDLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUN2QixDQUFDOzs7O0lBQ1MsdUJBQXVCLENBQUMsV0FBdUI7UUFDckQsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtZQUM5QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsMEJBQTBCLENBQUM7WUFDL0QsZUFBZSxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyxtREFBbUQsQ0FBQyxDQUFDO1NBQ3JILENBQUMsQ0FBQztRQUNILE1BQU0sUUFBUSxHQUFHLElBQUksaUNBQVUsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQzdDLGlCQUFpQixFQUFFLElBQUksQ0FBQyxPQUFPO1NBQ2xDLENBQUMsQ0FBQztRQUNILFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzdDLENBQUM7Ozs7SUFDUyxtQkFBbUIsQ0FBQyxLQUFxQjtRQUMvQyxNQUFNLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDO1FBQ2hFLElBQUksTUFBTSxFQUFFO1lBQ1IsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksdUJBQVUsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO2dCQUN4RCxXQUFXLEVBQUUsZ0RBQWdEO2dCQUM3RCxHQUFHLEVBQUUsSUFBSTtnQkFDVCxpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCO2FBQzdDLENBQUMsQ0FBQztZQUNILDZHQUE2RztZQUM3RywyREFBMkQ7WUFDM0QsTUFBTSxTQUFTLEdBQUcsQ0FBQyxLQUFLLENBQUMsYUFBYSxJQUFJLEtBQUssQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLElBQUksTUFBTSxDQUFDO1lBQ25GLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxhQUFLLENBQUMsSUFBSSxFQUFFLG1CQUFtQixTQUFTLEVBQUUsRUFBRTtnQkFDbkUsVUFBVSxFQUFFLElBQUksQ0FBQyxpQkFBaUI7Z0JBQ2xDLEdBQUcsS0FBSyxDQUFDLGFBQWE7YUFDekIsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxnQkFBUyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUUsRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1NBQ3ZHO2FBQ0k7WUFDRCxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7Z0JBQ3JCLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELENBQUMsQ0FBQzthQUM3RTtTQUNKO0lBQ0wsQ0FBQztJQUNEOztPQUVHO0lBQ08sbUJBQW1CLENBQUMsS0FBbUI7O1FBQzdDLElBQUksS0FBSyxDQUFDLGFBQWEsSUFBSSxLQUFLLENBQUMscUJBQXFCLEVBQUU7WUFDcEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxtRkFBbUYsQ0FBQyxDQUFDO1NBQ3hHO1FBQ0QsSUFBSSxLQUFLLENBQUMscUJBQXFCLEVBQUU7WUFDN0IsT0FBTztnQkFDSCxLQUFLLEVBQUUsS0FBSyxDQUFDLHFCQUFxQixDQUFDLEtBQUs7Z0JBQ3hDLGNBQWMsY0FBRSxLQUFLLENBQUMscUJBQXFCLDBDQUFFLFlBQVksMENBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQzthQUMzRyxDQUFDO1NBQ0w7UUFDRCxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDckIsT0FBTyxFQUFFLEtBQUssRUFBRSxLQUFLLENBQUMsYUFBYSxFQUFFLENBQUM7U0FDekM7UUFDRCxPQUFPLFNBQVMsQ0FBQztJQUNyQixDQUFDO0NBQ0o7QUExUEQsa0NBMFBDOzs7Ozs7Ozs7Ozs7OztBQWVELE1BQWEsV0FBWSxTQUFRLFdBQVc7Ozs7SUFZeEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1QjtRQUM3RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN4QixNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNwRCxNQUFNLFFBQVEsR0FBRyxJQUFJLGlDQUFVLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUM5QyxJQUFJLEVBQUUsSUFBSSxDQUFDLFdBQVc7WUFDdEIsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNO1lBQ3BCLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztZQUNwQyxJQUFJLEVBQUUsWUFBWSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDL0UsY0FBYyxFQUFFLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsVUFBVTtZQUNuRixxQkFBcUIsRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDO1lBQ3RELFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtTQUMvQixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksR0FBRyxRQUFRLENBQUM7UUFDbEMsSUFBSSxDQUFDLFNBQVMsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO1FBQzlCLElBQUksQ0FBQyxxQkFBcUIsR0FBRyxRQUFRLENBQUMsa0JBQWtCLENBQUM7UUFDekQsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLFlBQVksQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ25FLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNoQyxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDbEIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxjQUFjLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQ3hEO1FBQ0QsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztRQUN4RixJQUFJLGNBQWMsRUFBRTtZQUNoQixJQUFJLENBQUMsdUJBQXVCLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDMUM7SUFDTCxDQUFDO0NBQ0o7QUFyQ0Qsa0NBcUNDOzs7Ozs7Ozs7OztBQXNCRCxNQUFhLE9BQVEsU0FBUSxXQUFXOzs7O0lBdUNwQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQXNCLEVBQUU7UUFDOUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztRQU5aLFlBQU8sR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO1FBQzlDOztXQUVHO1FBQ2MsZ0JBQVcsR0FBRyxJQUFJLEtBQUssRUFBYyxDQUFDO1FBR25ELE1BQU0sUUFBUSxHQUFHLElBQUksaUNBQVUsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzlDLElBQUksRUFBRSxJQUFJLENBQUMsV0FBVztZQUN0QixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNO1lBQ3BCLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztZQUNwQyxzQkFBc0IsRUFBRSxLQUFLLENBQUMsc0JBQXNCO1lBQ3BELGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7WUFDeEMscUJBQXFCLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQztZQUN0RCxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO1lBQ3hDLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNsRSxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7U0FDL0IsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEdBQUcsUUFBUSxDQUFDO1FBQ2xDLElBQUksQ0FBQyxTQUFTLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQztRQUM5QixNQUFNLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO1FBQ3hGLElBQUksY0FBYyxFQUFFO1lBQ2hCLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztTQUMxQztRQUNELElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNoQyxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDbEIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxjQUFjLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQ3hEO1FBQ0QsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLFlBQVksQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQ3ZFLElBQUksQ0FBQyxxQkFBcUIsR0FBRyxRQUFRLENBQUMsa0JBQWtCLENBQUM7SUFDN0QsQ0FBQzs7Ozs7O0lBN0RNLE1BQU0sQ0FBQyxhQUFhLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsU0FBaUI7UUFDdkUsTUFBTSxNQUFPLFNBQVEsV0FBVztZQUFoQzs7Z0JBQ29CLGNBQVMsR0FBRyxTQUFTLENBQUM7WUFPMUMsQ0FBQztZQU5HLElBQVcsSUFBSTtnQkFDWCxNQUFNLElBQUksS0FBSyxDQUFDLDBHQUEwRyxDQUFDLENBQUM7WUFDaEksQ0FBQztZQUNELElBQVcscUJBQXFCO2dCQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLDJIQUEySCxDQUFDLENBQUM7WUFDakosQ0FBQztTQUNKO1FBQ0QsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDakMsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLHFCQUFxQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXdCO1FBQ3RGLE1BQU0sTUFBTyxTQUFRLFdBQVc7WUFBaEM7O2dCQUNvQixjQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztnQkFDNUIsMEJBQXFCLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQztnQkFDN0MsU0FBSSxHQUFjLElBQUksWUFBWSxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUUsSUFBSSxDQUFDLHFCQUFxQixDQUFDLENBQUM7WUFDN0YsQ0FBQztTQUFBO1FBQ0QsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDakMsQ0FBQzs7Ozs7O0lBMENELElBQVcsR0FBRztRQUNWLE9BQU8sSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQzdCLENBQUM7Ozs7OztJQUlNLFNBQVMsQ0FBQyxFQUFVLEVBQUUsT0FBdUI7UUFDaEQsT0FBTyxJQUFJLGdCQUFNLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUN4QixTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUM7WUFDakIsR0FBRyxPQUFPO1NBQ2IsQ0FBQyxDQUFDO0lBQ1AsQ0FBQzs7Ozs7O0lBSU0sUUFBUSxDQUFDLEVBQVUsRUFBRSxLQUFtQjtRQUMzQyxPQUFPLElBQUksYUFBSyxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDdkIsR0FBRyxLQUFLO1lBQ1IsT0FBTyxFQUFFLElBQUk7U0FDaEIsQ0FBQyxDQUFDO0lBQ1AsQ0FBQzs7Ozs7O0lBSU0sbUJBQW1CLENBQUMsRUFBVSxFQUFFLEtBQThCO1FBQ2pFLE9BQU8sSUFBSSxtQ0FBZ0IsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ2xDLEdBQUcsS0FBSztZQUNSLE9BQU8sRUFBRSxJQUFJO1NBQ2hCLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDRDs7Ozs7T0FLRztJQUNJLGFBQWEsQ0FBQyxNQUFjO1FBQy9CLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzFCLDBFQUEwRTtRQUMxRSx5RUFBeUU7UUFDekUsbUNBQW1DO1FBQ25DLEtBQUssTUFBTSxHQUFHLElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRTtZQUNoQyxHQUFHLENBQUMsb0JBQW9CLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDcEM7SUFDTCxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLGlCQUFpQixDQUFDLFVBQXNCO1FBQzNDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2xDLG9FQUFvRTtRQUNwRSw0RUFBNEU7UUFDNUUsdURBQXVEO1FBQ3ZELEtBQUssTUFBTSxNQUFNLElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUMvQixVQUFVLENBQUMsb0JBQW9CLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDM0M7SUFDTCxDQUFDOzs7Ozs7SUFJUyxRQUFRO1FBQ2QsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDM0IsT0FBTyxDQUFDLDBDQUEwQyxDQUFDLENBQUM7U0FDdkQ7UUFDRCxPQUFPLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDSjtBQXpJRCwwQkF5SUM7Ozs7QUFvQkQsSUFBWSxnQkFTWDtBQVRELFdBQVksZ0JBQWdCO0lBSXhCLHFDQUFpQixDQUFBO0lBSWpCLDZDQUF5QixDQUFBO0FBQzdCLENBQUMsRUFUVyxnQkFBZ0IsR0FBaEIsd0JBQWdCLEtBQWhCLHdCQUFnQixRQVMzQjs7OztBQUNELElBQVksWUFhWDtBQWJELFdBQVksWUFBWTtJQUlwQiw2QkFBYSxDQUFBO0lBSWIscUNBQXFCLENBQUE7SUFJckIsbUNBQW1CLENBQUE7QUFDdkIsQ0FBQyxFQWJXLFlBQVksR0FBWixvQkFBWSxLQUFaLG9CQUFZLFFBYXZCO0FBQ0QsTUFBTSxZQUFhLFNBQVEsdUJBQVk7SUFTbkMsWUFBWSxHQUFnQixFQUFFLEtBQXNCLEVBQUUsVUFBa0I7UUFDcEUsS0FBSyxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUN0QixJQUFJLENBQUMsY0FBYyxHQUFHLFNBQVMsQ0FBQztRQUNoQyxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDO1FBQ25ELElBQUksQ0FBQyxvQkFBb0IsR0FBRyxLQUFLLENBQUMsb0JBQW9CLENBQUM7UUFDdkQsSUFBSSxDQUFDLDJCQUEyQixHQUFHLEtBQUssQ0FBQywyQkFBMkIsQ0FBQztRQUNyRSxJQUFJLENBQUMsR0FBRyxHQUFHLEdBQUcsQ0FBQztRQUNmLElBQUksQ0FBQyxVQUFVLEdBQUcsVUFBVSxDQUFDO1FBQzdCLElBQUksQ0FBQyxJQUFJLEdBQUcsR0FBRyxDQUFDO1FBQ2hCLElBQUksR0FBRyxZQUFZLE9BQU8sRUFBRTtZQUN4QixJQUFJLENBQUMsUUFBUSxHQUFHLEdBQUcsQ0FBQztTQUN2QjtRQUNELElBQUksSUFBSSxDQUFDLDJCQUEyQixFQUFFO1lBQ2xDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsMkJBQTJCLENBQUMsQ0FBQztTQUMzRDtJQUNMLENBQUM7SUFDRDs7O09BR0c7SUFDSCxJQUFXLE9BQU87UUFDZCxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUNoQixNQUFNLElBQUksS0FBSyxDQUFDLGlHQUFpRyxDQUFDLENBQUM7U0FDdEg7UUFDRCxPQUFPLElBQUksQ0FBQyxRQUFRLENBQUM7SUFDekIsQ0FBQztDQUNKO0FBQ0QsU0FBUyxNQUFNLENBQUMsRUFBTztJQUNuQixPQUFPO0FBQ1gsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNsb3Vkd2F0Y2ggZnJvbSBcIi4uLy4uL2F3cy1jbG91ZHdhdGNoXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWR3YXRjaCdcbmltcG9ydCB7IElWcGNFbmRwb2ludCB9IGZyb20gXCIuLi8uLi9hd3MtZWMyXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJ1xuaW1wb3J0ICogYXMgaWFtIGZyb20gXCIuLi8uLi9hd3MtaWFtXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtaWFtJ1xuaW1wb3J0IHsgQ2ZuT3V0cHV0LCBJUmVzb3VyY2UgYXMgSVJlc291cmNlQmFzZSwgUmVzb3VyY2UsIFN0YWNrIH0gZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IEFwaURlZmluaXRpb24gfSBmcm9tICcuL2FwaS1kZWZpbml0aW9uJztcbmltcG9ydCB7IEFwaUtleSwgQXBpS2V5T3B0aW9ucywgSUFwaUtleSB9IGZyb20gJy4vYXBpLWtleSc7XG5pbXBvcnQgeyBDZm5BY2NvdW50LCBDZm5SZXN0QXBpIH0gZnJvbSAnLi9hcGlnYXRld2F5LmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBDb3JzT3B0aW9ucyB9IGZyb20gJy4vY29ycyc7XG5pbXBvcnQgeyBEZXBsb3ltZW50IH0gZnJvbSAnLi9kZXBsb3ltZW50JztcbmltcG9ydCB7IERvbWFpbk5hbWUsIERvbWFpbk5hbWVPcHRpb25zIH0gZnJvbSAnLi9kb21haW4tbmFtZSc7XG5pbXBvcnQgeyBHYXRld2F5UmVzcG9uc2UsIEdhdGV3YXlSZXNwb25zZU9wdGlvbnMgfSBmcm9tICcuL2dhdGV3YXktcmVzcG9uc2UnO1xuaW1wb3J0IHsgSW50ZWdyYXRpb24gfSBmcm9tICcuL2ludGVncmF0aW9uJztcbmltcG9ydCB7IE1ldGhvZCwgTWV0aG9kT3B0aW9ucyB9IGZyb20gJy4vbWV0aG9kJztcbmltcG9ydCB7IE1vZGVsLCBNb2RlbE9wdGlvbnMgfSBmcm9tICcuL21vZGVsJztcbmltcG9ydCB7IFJlcXVlc3RWYWxpZGF0b3IsIFJlcXVlc3RWYWxpZGF0b3JPcHRpb25zIH0gZnJvbSAnLi9yZXF1ZXN0dmFsaWRhdG9yJztcbmltcG9ydCB7IElSZXNvdXJjZSwgUmVzb3VyY2VCYXNlLCBSZXNvdXJjZU9wdGlvbnMgfSBmcm9tICcuL3Jlc291cmNlJztcbmltcG9ydCB7IFN0YWdlLCBTdGFnZU9wdGlvbnMgfSBmcm9tICcuL3N0YWdlJztcbmltcG9ydCB7IFVzYWdlUGxhbiwgVXNhZ2VQbGFuUHJvcHMgfSBmcm9tICcuL3VzYWdlLXBsYW4nO1xuY29uc3QgUkVTVEFQSV9TWU1CT0wgPSBTeW1ib2wuZm9yKCdAYXdzLWNkay9hd3MtYXBpZ2F0ZXdheS5SZXN0QXBpQmFzZScpO1xuZXhwb3J0IGludGVyZmFjZSBJUmVzdEFwaSBleHRlbmRzIElSZXNvdXJjZUJhc2Uge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHJlc3RBcGlJZDogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcmVzdEFwaVJvb3RSZXNvdXJjZUlkOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgbGF0ZXN0RGVwbG95bWVudD86IERlcGxveW1lbnQ7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgZGVwbG95bWVudFN0YWdlOiBTdGFnZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcm9vdDogSVJlc291cmNlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBhcm5Gb3JFeGVjdXRlQXBpKG1ldGhvZD86IHN0cmluZywgcGF0aD86IHN0cmluZywgc3RhZ2U/OiBzdHJpbmcpOiBzdHJpbmc7XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBSZXN0QXBpQmFzZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZGVwbG95PzogYm9vbGVhbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZGVwbG95T3B0aW9ucz86IFN0YWdlT3B0aW9ucztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHJldGFpbkRlcGxveW1lbnRzPzogYm9vbGVhbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSByZXN0QXBpTmFtZT86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcGFyYW1ldGVycz86IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogc3RyaW5nO1xuICAgIH07XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBwb2xpY3k/OiBpYW0uUG9saWN5RG9jdW1lbnQ7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBmYWlsT25XYXJuaW5ncz86IGJvb2xlYW47XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZG9tYWluTmFtZT86IERvbWFpbk5hbWVPcHRpb25zO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgY2xvdWRXYXRjaFJvbGU/OiBib29sZWFuO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBlbmRwb2ludEV4cG9ydE5hbWU/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZW5kcG9pbnRUeXBlcz86IEVuZHBvaW50VHlwZVtdO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgUmVzdEFwaU9wdGlvbnMgZXh0ZW5kcyBSZXN0QXBpQmFzZVByb3BzLCBSZXNvdXJjZU9wdGlvbnMge1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBSZXN0QXBpUHJvcHMgZXh0ZW5kcyBSZXN0QXBpT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgYmluYXJ5TWVkaWFUeXBlcz86IHN0cmluZ1tdO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgbWluaW11bUNvbXByZXNzaW9uU2l6ZT86IG51bWJlcjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGNsb25lRnJvbT86IElSZXN0QXBpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGFwaUtleVNvdXJjZVR5cGU/OiBBcGlLZXlTb3VyY2VUeXBlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZW5kcG9pbnRDb25maWd1cmF0aW9uPzogRW5kcG9pbnRDb25maWd1cmF0aW9uO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFNwZWNSZXN0QXBpUHJvcHMgZXh0ZW5kcyBSZXN0QXBpQmFzZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGFwaURlZmluaXRpb246IEFwaURlZmluaXRpb247XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgUmVzdEFwaUJhc2UgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElSZXN0QXBpIHtcbiAgICAvKipcbiAgICAgKiBDaGVja3MgaWYgdGhlIGdpdmVuIG9iamVjdCBpcyBhbiBpbnN0YW5jZSBvZiBSZXN0QXBpQmFzZS5cbiAgICAgKiBAaW50ZXJuYWxcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIF9pc1Jlc3RBcGlCYXNlKHg6IGFueSk6IHggaXMgUmVzdEFwaUJhc2Uge1xuICAgICAgICByZXR1cm4geCAhPT0gbnVsbCAmJiB0eXBlb2YgKHgpID09PSAnb2JqZWN0JyAmJiBSRVNUQVBJX1NZTUJPTCBpbiB4O1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBnZXQgbGF0ZXN0RGVwbG95bWVudCgpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuX2xhdGVzdERlcGxveW1lbnQ7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGdldCBkb21haW5OYW1lKCkge1xuICAgICAgICByZXR1cm4gdGhpcy5fZG9tYWluTmFtZTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHJlc3RBcGlJZDogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSByZXN0QXBpUm9vdFJlc291cmNlSWQ6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgcm9vdDogSVJlc291cmNlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGRlcGxveW1lbnRTdGFnZSE6IFN0YWdlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyByZWFkb25seSByZXN0QXBpTmFtZTogc3RyaW5nO1xuICAgIHByaXZhdGUgX2xhdGVzdERlcGxveW1lbnQ/OiBEZXBsb3ltZW50O1xuICAgIHByaXZhdGUgX2RvbWFpbk5hbWU/OiBEb21haW5OYW1lO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBSZXN0QXBpQmFzZVByb3BzID0ge30pIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgdGhpcy5yZXN0QXBpTmFtZSA9IHByb3BzLnJlc3RBcGlOYW1lID8/IGlkO1xuICAgICAgICBPYmplY3QuZGVmaW5lUHJvcGVydHkodGhpcywgUkVTVEFQSV9TWU1CT0wsIHsgdmFsdWU6IHRydWUgfSk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyB1cmxGb3JQYXRoKHBhdGg6IHN0cmluZyA9ICcvJyk6IHN0cmluZyB7XG4gICAgICAgIGlmICghdGhpcy5kZXBsb3ltZW50U3RhZ2UpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGRldGVybWluZSBkZXBsb3ltZW50IHN0YWdlIGZvciBBUEkgZnJvbSBcImRlcGxveW1lbnRTdGFnZVwiLiBVc2UgXCJkZXBsb3lcIiBvciBleHBsaWNpdGx5IHNldCBcImRlcGxveW1lbnRTdGFnZVwiJyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMuZGVwbG95bWVudFN0YWdlLnVybEZvclBhdGgocGF0aCk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBhZGREb21haW5OYW1lKGlkOiBzdHJpbmcsIG9wdGlvbnM6IERvbWFpbk5hbWVPcHRpb25zKTogRG9tYWluTmFtZSB7XG4gICAgICAgIGNvbnN0IGRvbWFpbk5hbWUgPSBuZXcgRG9tYWluTmFtZSh0aGlzLCBpZCwge1xuICAgICAgICAgICAgLi4ub3B0aW9ucyxcbiAgICAgICAgICAgIG1hcHBpbmc6IHRoaXMsXG4gICAgICAgIH0pO1xuICAgICAgICBpZiAoIXRoaXMuX2RvbWFpbk5hbWUpIHtcbiAgICAgICAgICAgIHRoaXMuX2RvbWFpbk5hbWUgPSBkb21haW5OYW1lO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBkb21haW5OYW1lO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFkZFVzYWdlUGxhbihpZDogc3RyaW5nLCBwcm9wczogVXNhZ2VQbGFuUHJvcHMgPSB7fSk6IFVzYWdlUGxhbiB7XG4gICAgICAgIHJldHVybiBuZXcgVXNhZ2VQbGFuKHRoaXMsIGlkLCBwcm9wcyk7XG4gICAgfVxuICAgIHB1YmxpYyBhcm5Gb3JFeGVjdXRlQXBpKG1ldGhvZDogc3RyaW5nID0gJyonLCBwYXRoOiBzdHJpbmcgPSAnLyonLCBzdGFnZTogc3RyaW5nID0gJyonKSB7XG4gICAgICAgIGlmICghcGF0aC5zdGFydHNXaXRoKCcvJykpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgXCJwYXRoXCIgbXVzdCBiZWdpbiB3aXRoIGEgXCIvXCI6ICcke3BhdGh9J2ApO1xuICAgICAgICB9XG4gICAgICAgIGlmIChtZXRob2QudG9VcHBlckNhc2UoKSA9PT0gJ0FOWScpIHtcbiAgICAgICAgICAgIG1ldGhvZCA9ICcqJztcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHtcbiAgICAgICAgICAgIHNlcnZpY2U6ICdleGVjdXRlLWFwaScsXG4gICAgICAgICAgICByZXNvdXJjZTogdGhpcy5yZXN0QXBpSWQsXG4gICAgICAgICAgICBzZXA6ICcvJyxcbiAgICAgICAgICAgIHJlc291cmNlTmFtZTogYCR7c3RhZ2V9LyR7bWV0aG9kfSR7cGF0aH1gLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWRkR2F0ZXdheVJlc3BvbnNlKGlkOiBzdHJpbmcsIG9wdGlvbnM6IEdhdGV3YXlSZXNwb25zZU9wdGlvbnMpOiBHYXRld2F5UmVzcG9uc2Uge1xuICAgICAgICByZXR1cm4gbmV3IEdhdGV3YXlSZXNwb25zZSh0aGlzLCBpZCwge1xuICAgICAgICAgICAgcmVzdEFwaTogdGhpcyxcbiAgICAgICAgICAgIC4uLm9wdGlvbnMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBtZXRyaWMobWV0cmljTmFtZTogc3RyaW5nLCBwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgcmV0dXJuIG5ldyBjbG91ZHdhdGNoLk1ldHJpYyh7XG4gICAgICAgICAgICBuYW1lc3BhY2U6ICdBV1MvQXBpR2F0ZXdheScsXG4gICAgICAgICAgICBtZXRyaWNOYW1lLFxuICAgICAgICAgICAgZGltZW5zaW9uczogeyBBcGlOYW1lOiB0aGlzLnJlc3RBcGlOYW1lIH0sXG4gICAgICAgICAgICAuLi5wcm9wcyxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIG1ldHJpY0NsaWVudEVycm9yKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgICAgICByZXR1cm4gdGhpcy5tZXRyaWMoJzRYWEVycm9yJywgeyBzdGF0aXN0aWM6ICdTdW0nLCAuLi5wcm9wcyB9KTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgbWV0cmljU2VydmVyRXJyb3IocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgICAgIHJldHVybiB0aGlzLm1ldHJpYygnNVhYRXJyb3InLCB7IHN0YXRpc3RpYzogJ1N1bScsIC4uLnByb3BzIH0pO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgbWV0cmljQ2FjaGVIaXRDb3VudChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgcmV0dXJuIHRoaXMubWV0cmljKCdDYWNoZUhpdENvdW50JywgeyBzdGF0aXN0aWM6ICdTdW0nLCAuLi5wcm9wcyB9KTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBtZXRyaWNDYWNoZU1pc3NDb3VudChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgcmV0dXJuIHRoaXMubWV0cmljKCdDYWNoZU1pc3NDb3VudCcsIHsgc3RhdGlzdGljOiAnU3VtJywgLi4ucHJvcHMgfSk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgbWV0cmljQ291bnQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgICAgIHJldHVybiB0aGlzLm1ldHJpYygnQ291bnQnLCB7IHN0YXRpc3RpYzogJ1NhbXBsZUNvdW50JywgLi4ucHJvcHMgfSk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgbWV0cmljSW50ZWdyYXRpb25MYXRlbmN5KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgICAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ0ludGVncmF0aW9uTGF0ZW5jeScsIHByb3BzKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgbWV0cmljTGF0ZW5jeShwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgcmV0dXJuIHRoaXMubWV0cmljKCdMYXRlbmN5JywgcHJvcHMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBJbnRlcm5hbCBBUEkgdXNlZCBieSBgTWV0aG9kYCB0byBrZWVwIGFuIGludmVudG9yeSBvZiBtZXRob2RzIGF0IHRoZSBBUElcbiAgICAgKiBsZXZlbCBmb3IgdmFsaWRhdGlvbiBwdXJwb3Nlcy5cbiAgICAgKlxuICAgICAqIEBpbnRlcm5hbFxuICAgICAqL1xuICAgIHB1YmxpYyBfYXR0YWNoTWV0aG9kKG1ldGhvZDogTWV0aG9kKSB7XG4gICAgICAgIGlnbm9yZShtZXRob2QpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBc3NvY2lhdGVzIGEgRGVwbG95bWVudCByZXNvdXJjZSB3aXRoIHRoaXMgUkVTVCBBUEkuXG4gICAgICpcbiAgICAgKiBAaW50ZXJuYWxcbiAgICAgKi9cbiAgICBwdWJsaWMgX2F0dGFjaERlcGxveW1lbnQoZGVwbG95bWVudDogRGVwbG95bWVudCkge1xuICAgICAgICBpZ25vcmUoZGVwbG95bWVudCk7XG4gICAgfVxuICAgIHByb3RlY3RlZCBjb25maWd1cmVDbG91ZFdhdGNoUm9sZShhcGlSZXNvdXJjZTogQ2ZuUmVzdEFwaSkge1xuICAgICAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdDbG91ZFdhdGNoUm9sZScsIHtcbiAgICAgICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdhcGlnYXRld2F5LmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgICAgIG1hbmFnZWRQb2xpY2llczogW2lhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FtYXpvbkFQSUdhdGV3YXlQdXNoVG9DbG91ZFdhdGNoTG9ncycpXSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmbkFjY291bnQodGhpcywgJ0FjY291bnQnLCB7XG4gICAgICAgICAgICBjbG91ZFdhdGNoUm9sZUFybjogcm9sZS5yb2xlQXJuLFxuICAgICAgICB9KTtcbiAgICAgICAgcmVzb3VyY2Uubm9kZS5hZGREZXBlbmRlbmN5KGFwaVJlc291cmNlKTtcbiAgICB9XG4gICAgcHJvdGVjdGVkIGNvbmZpZ3VyZURlcGxveW1lbnQocHJvcHM6IFJlc3RBcGlPcHRpb25zKSB7XG4gICAgICAgIGNvbnN0IGRlcGxveSA9IHByb3BzLmRlcGxveSA9PT0gdW5kZWZpbmVkID8gdHJ1ZSA6IHByb3BzLmRlcGxveTtcbiAgICAgICAgaWYgKGRlcGxveSkge1xuICAgICAgICAgICAgdGhpcy5fbGF0ZXN0RGVwbG95bWVudCA9IG5ldyBEZXBsb3ltZW50KHRoaXMsICdEZXBsb3ltZW50Jywge1xuICAgICAgICAgICAgICAgIGRlc2NyaXB0aW9uOiAnQXV0b21hdGljYWxseSBjcmVhdGVkIGJ5IHRoZSBSZXN0QXBpIGNvbnN0cnVjdCcsXG4gICAgICAgICAgICAgICAgYXBpOiB0aGlzLFxuICAgICAgICAgICAgICAgIHJldGFpbkRlcGxveW1lbnRzOiBwcm9wcy5yZXRhaW5EZXBsb3ltZW50cyxcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgLy8gZW5jb2RlIHRoZSBzdGFnZSBuYW1lIGludG8gdGhlIGNvbnN0cnVjdCBpZCwgc28gaWYgd2UgY2hhbmdlIHRoZSBzdGFnZSBuYW1lLCBpdCB3aWxsIHJlY3JlYXRlIGEgbmV3IHN0YWdlLlxuICAgICAgICAgICAgLy8gc3RhZ2UgbmFtZSBpcyBwYXJ0IG9mIHRoZSBlbmRwb2ludCwgc28gdGhhdCBtYWtlcyBzZW5zZS5cbiAgICAgICAgICAgIGNvbnN0IHN0YWdlTmFtZSA9IChwcm9wcy5kZXBsb3lPcHRpb25zICYmIHByb3BzLmRlcGxveU9wdGlvbnMuc3RhZ2VOYW1lKSB8fCAncHJvZCc7XG4gICAgICAgICAgICB0aGlzLmRlcGxveW1lbnRTdGFnZSA9IG5ldyBTdGFnZSh0aGlzLCBgRGVwbG95bWVudFN0YWdlLiR7c3RhZ2VOYW1lfWAsIHtcbiAgICAgICAgICAgICAgICBkZXBsb3ltZW50OiB0aGlzLl9sYXRlc3REZXBsb3ltZW50LFxuICAgICAgICAgICAgICAgIC4uLnByb3BzLmRlcGxveU9wdGlvbnMsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ0VuZHBvaW50JywgeyBleHBvcnROYW1lOiBwcm9wcy5lbmRwb2ludEV4cG9ydE5hbWUsIHZhbHVlOiB0aGlzLnVybEZvclBhdGgoKSB9KTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIGlmIChwcm9wcy5kZXBsb3lPcHRpb25zKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3Qgc2V0IFxcJ2RlcGxveU9wdGlvbnNcXCcgaWYgXFwnZGVwbG95XFwnIGlzIGRpc2FibGVkJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICB9XG4gICAgLyoqXG4gICAgICogQGludGVybmFsXG4gICAgICovXG4gICAgcHJvdGVjdGVkIF9jb25maWd1cmVFbmRwb2ludHMocHJvcHM6IFJlc3RBcGlQcm9wcyk6IENmblJlc3RBcGkuRW5kcG9pbnRDb25maWd1cmF0aW9uUHJvcGVydHkgfCB1bmRlZmluZWQge1xuICAgICAgICBpZiAocHJvcHMuZW5kcG9pbnRUeXBlcyAmJiBwcm9wcy5lbmRwb2ludENvbmZpZ3VyYXRpb24pIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignT25seSBvbmUgb2YgdGhlIFJlc3RBcGkgcHJvcHMsIGVuZHBvaW50VHlwZXMgb3IgZW5kcG9pbnRDb25maWd1cmF0aW9uLCBpcyBhbGxvd2VkJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLmVuZHBvaW50Q29uZmlndXJhdGlvbikge1xuICAgICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgICAgICB0eXBlczogcHJvcHMuZW5kcG9pbnRDb25maWd1cmF0aW9uLnR5cGVzLFxuICAgICAgICAgICAgICAgIHZwY0VuZHBvaW50SWRzOiBwcm9wcy5lbmRwb2ludENvbmZpZ3VyYXRpb24/LnZwY0VuZHBvaW50cz8ubWFwKHZwY0VuZHBvaW50ID0+IHZwY0VuZHBvaW50LnZwY0VuZHBvaW50SWQpLFxuICAgICAgICAgICAgfTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocHJvcHMuZW5kcG9pbnRUeXBlcykge1xuICAgICAgICAgICAgcmV0dXJuIHsgdHlwZXM6IHByb3BzLmVuZHBvaW50VHlwZXMgfTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBTcGVjUmVzdEFwaSBleHRlbmRzIFJlc3RBcGlCYXNlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgcmVhZG9ubHkgcmVzdEFwaUlkOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHJlYWRvbmx5IHJlc3RBcGlSb290UmVzb3VyY2VJZDogc3RyaW5nO1xuICAgIHB1YmxpYyByZWFkb25seSByb290OiBJUmVzb3VyY2U7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFNwZWNSZXN0QXBpUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG4gICAgICAgIGNvbnN0IGFwaURlZkNvbmZpZyA9IHByb3BzLmFwaURlZmluaXRpb24uYmluZCh0aGlzKTtcbiAgICAgICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuUmVzdEFwaSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICAgICAgICBuYW1lOiB0aGlzLnJlc3RBcGlOYW1lLFxuICAgICAgICAgICAgcG9saWN5OiBwcm9wcy5wb2xpY3ksXG4gICAgICAgICAgICBmYWlsT25XYXJuaW5nczogcHJvcHMuZmFpbE9uV2FybmluZ3MsXG4gICAgICAgICAgICBib2R5OiBhcGlEZWZDb25maWcuaW5saW5lRGVmaW5pdGlvbiA/IGFwaURlZkNvbmZpZy5pbmxpbmVEZWZpbml0aW9uIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgYm9keVMzTG9jYXRpb246IGFwaURlZkNvbmZpZy5pbmxpbmVEZWZpbml0aW9uID8gdW5kZWZpbmVkIDogYXBpRGVmQ29uZmlnLnMzTG9jYXRpb24sXG4gICAgICAgICAgICBlbmRwb2ludENvbmZpZ3VyYXRpb246IHRoaXMuX2NvbmZpZ3VyZUVuZHBvaW50cyhwcm9wcyksXG4gICAgICAgICAgICBwYXJhbWV0ZXJzOiBwcm9wcy5wYXJhbWV0ZXJzLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5ub2RlLmRlZmF1bHRDaGlsZCA9IHJlc291cmNlO1xuICAgICAgICB0aGlzLnJlc3RBcGlJZCA9IHJlc291cmNlLnJlZjtcbiAgICAgICAgdGhpcy5yZXN0QXBpUm9vdFJlc291cmNlSWQgPSByZXNvdXJjZS5hdHRyUm9vdFJlc291cmNlSWQ7XG4gICAgICAgIHRoaXMucm9vdCA9IG5ldyBSb290UmVzb3VyY2UodGhpcywge30sIHRoaXMucmVzdEFwaVJvb3RSZXNvdXJjZUlkKTtcbiAgICAgICAgdGhpcy5jb25maWd1cmVEZXBsb3ltZW50KHByb3BzKTtcbiAgICAgICAgaWYgKHByb3BzLmRvbWFpbk5hbWUpIHtcbiAgICAgICAgICAgIHRoaXMuYWRkRG9tYWluTmFtZSgnQ3VzdG9tRG9tYWluJywgcHJvcHMuZG9tYWluTmFtZSk7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgY2xvdWRXYXRjaFJvbGUgPSBwcm9wcy5jbG91ZFdhdGNoUm9sZSAhPT0gdW5kZWZpbmVkID8gcHJvcHMuY2xvdWRXYXRjaFJvbGUgOiB0cnVlO1xuICAgICAgICBpZiAoY2xvdWRXYXRjaFJvbGUpIHtcbiAgICAgICAgICAgIHRoaXMuY29uZmlndXJlQ2xvdWRXYXRjaFJvbGUocmVzb3VyY2UpO1xuICAgICAgICB9XG4gICAgfVxufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgUmVzdEFwaUF0dHJpYnV0ZXMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcmVzdEFwaUlkOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSByb290UmVzb3VyY2VJZDogc3RyaW5nO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgUmVzdEFwaSBleHRlbmRzIFJlc3RBcGlCYXNlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBmcm9tUmVzdEFwaUlkKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHJlc3RBcGlJZDogc3RyaW5nKTogSVJlc3RBcGkge1xuICAgICAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBSZXN0QXBpQmFzZSB7XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgcmVzdEFwaUlkID0gcmVzdEFwaUlkO1xuICAgICAgICAgICAgcHVibGljIGdldCByb290KCk6IElSZXNvdXJjZSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdyb290IGlzIG5vdCBjb25maWd1cmVkIHdoZW4gaW1wb3J0ZWQgdXNpbmcgYGZyb21SZXN0QXBpSWQoKWAuIFVzZSBgZnJvbVJlc3RBcGlBdHRyaWJ1dGVzKClgIEFQSSBpbnN0ZWFkLicpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgcHVibGljIGdldCByZXN0QXBpUm9vdFJlc291cmNlSWQoKTogc3RyaW5nIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ3Jlc3RBcGlSb290UmVzb3VyY2VJZCBpcyBub3QgY29uZmlndXJlZCB3aGVuIGltcG9ydGVkIHVzaW5nIGBmcm9tUmVzdEFwaUlkKClgLiBVc2UgYGZyb21SZXN0QXBpQXR0cmlidXRlcygpYCBBUEkgaW5zdGVhZC4nKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgZnJvbVJlc3RBcGlBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBSZXN0QXBpQXR0cmlidXRlcyk6IElSZXN0QXBpIHtcbiAgICAgICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzdEFwaUJhc2Uge1xuICAgICAgICAgICAgcHVibGljIHJlYWRvbmx5IHJlc3RBcGlJZCA9IGF0dHJzLnJlc3RBcGlJZDtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSByZXN0QXBpUm9vdFJlc291cmNlSWQgPSBhdHRycy5yb290UmVzb3VyY2VJZDtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSByb290OiBJUmVzb3VyY2UgPSBuZXcgUm9vdFJlc291cmNlKHRoaXMsIHt9LCB0aGlzLnJlc3RBcGlSb290UmVzb3VyY2VJZCk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgICB9XG4gICAgcHVibGljIHJlYWRvbmx5IHJlc3RBcGlJZDogc3RyaW5nO1xuICAgIHB1YmxpYyByZWFkb25seSByb290OiBJUmVzb3VyY2U7XG4gICAgcHVibGljIHJlYWRvbmx5IHJlc3RBcGlSb290UmVzb3VyY2VJZDogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyByZWFkb25seSBtZXRob2RzID0gbmV3IEFycmF5PE1ldGhvZD4oKTtcbiAgICAvKipcbiAgICAgKiBUaGlzIGxpc3Qgb2YgZGVwbG95bWVudHMgYm91bmQgdG8gdGhpcyBSZXN0QXBpXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSBkZXBsb3ltZW50cyA9IG5ldyBBcnJheTxEZXBsb3ltZW50PigpO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBSZXN0QXBpUHJvcHMgPSB7fSkge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcbiAgICAgICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuUmVzdEFwaSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICAgICAgICBuYW1lOiB0aGlzLnJlc3RBcGlOYW1lLFxuICAgICAgICAgICAgZGVzY3JpcHRpb246IHByb3BzLmRlc2NyaXB0aW9uLFxuICAgICAgICAgICAgcG9saWN5OiBwcm9wcy5wb2xpY3ksXG4gICAgICAgICAgICBmYWlsT25XYXJuaW5nczogcHJvcHMuZmFpbE9uV2FybmluZ3MsXG4gICAgICAgICAgICBtaW5pbXVtQ29tcHJlc3Npb25TaXplOiBwcm9wcy5taW5pbXVtQ29tcHJlc3Npb25TaXplLFxuICAgICAgICAgICAgYmluYXJ5TWVkaWFUeXBlczogcHJvcHMuYmluYXJ5TWVkaWFUeXBlcyxcbiAgICAgICAgICAgIGVuZHBvaW50Q29uZmlndXJhdGlvbjogdGhpcy5fY29uZmlndXJlRW5kcG9pbnRzKHByb3BzKSxcbiAgICAgICAgICAgIGFwaUtleVNvdXJjZVR5cGU6IHByb3BzLmFwaUtleVNvdXJjZVR5cGUsXG4gICAgICAgICAgICBjbG9uZUZyb206IHByb3BzLmNsb25lRnJvbSA/IHByb3BzLmNsb25lRnJvbS5yZXN0QXBpSWQgOiB1bmRlZmluZWQsXG4gICAgICAgICAgICBwYXJhbWV0ZXJzOiBwcm9wcy5wYXJhbWV0ZXJzLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5ub2RlLmRlZmF1bHRDaGlsZCA9IHJlc291cmNlO1xuICAgICAgICB0aGlzLnJlc3RBcGlJZCA9IHJlc291cmNlLnJlZjtcbiAgICAgICAgY29uc3QgY2xvdWRXYXRjaFJvbGUgPSBwcm9wcy5jbG91ZFdhdGNoUm9sZSAhPT0gdW5kZWZpbmVkID8gcHJvcHMuY2xvdWRXYXRjaFJvbGUgOiB0cnVlO1xuICAgICAgICBpZiAoY2xvdWRXYXRjaFJvbGUpIHtcbiAgICAgICAgICAgIHRoaXMuY29uZmlndXJlQ2xvdWRXYXRjaFJvbGUocmVzb3VyY2UpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuY29uZmlndXJlRGVwbG95bWVudChwcm9wcyk7XG4gICAgICAgIGlmIChwcm9wcy5kb21haW5OYW1lKSB7XG4gICAgICAgICAgICB0aGlzLmFkZERvbWFpbk5hbWUoJ0N1c3RvbURvbWFpbicsIHByb3BzLmRvbWFpbk5hbWUpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMucm9vdCA9IG5ldyBSb290UmVzb3VyY2UodGhpcywgcHJvcHMsIHJlc291cmNlLmF0dHJSb290UmVzb3VyY2VJZCk7XG4gICAgICAgIHRoaXMucmVzdEFwaVJvb3RSZXNvdXJjZUlkID0gcmVzb3VyY2UuYXR0clJvb3RSZXNvdXJjZUlkO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGdldCB1cmwoKSB7XG4gICAgICAgIHJldHVybiB0aGlzLnVybEZvclBhdGgoKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWRkQXBpS2V5KGlkOiBzdHJpbmcsIG9wdGlvbnM/OiBBcGlLZXlPcHRpb25zKTogSUFwaUtleSB7XG4gICAgICAgIHJldHVybiBuZXcgQXBpS2V5KHRoaXMsIGlkLCB7XG4gICAgICAgICAgICByZXNvdXJjZXM6IFt0aGlzXSxcbiAgICAgICAgICAgIC4uLm9wdGlvbnMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWRkTW9kZWwoaWQ6IHN0cmluZywgcHJvcHM6IE1vZGVsT3B0aW9ucyk6IE1vZGVsIHtcbiAgICAgICAgcmV0dXJuIG5ldyBNb2RlbCh0aGlzLCBpZCwge1xuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgICAgICByZXN0QXBpOiB0aGlzLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFkZFJlcXVlc3RWYWxpZGF0b3IoaWQ6IHN0cmluZywgcHJvcHM6IFJlcXVlc3RWYWxpZGF0b3JPcHRpb25zKTogUmVxdWVzdFZhbGlkYXRvciB7XG4gICAgICAgIHJldHVybiBuZXcgUmVxdWVzdFZhbGlkYXRvcih0aGlzLCBpZCwge1xuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgICAgICByZXN0QXBpOiB0aGlzLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogSW50ZXJuYWwgQVBJIHVzZWQgYnkgYE1ldGhvZGAgdG8ga2VlcCBhbiBpbnZlbnRvcnkgb2YgbWV0aG9kcyBhdCB0aGUgQVBJXG4gICAgICogbGV2ZWwgZm9yIHZhbGlkYXRpb24gcHVycG9zZXMuXG4gICAgICpcbiAgICAgKiBAaW50ZXJuYWxcbiAgICAgKi9cbiAgICBwdWJsaWMgX2F0dGFjaE1ldGhvZChtZXRob2Q6IE1ldGhvZCkge1xuICAgICAgICB0aGlzLm1ldGhvZHMucHVzaChtZXRob2QpO1xuICAgICAgICAvLyBhZGQgdGhpcyBtZXRob2QgYXMgYSBkZXBlbmRlbmN5IHRvIGFsbCBkZXBsb3ltZW50cyBkZWZpbmVkIGZvciB0aGlzIGFwaVxuICAgICAgICAvLyB3aGVuIGFkZGl0aW9uYWwgZGVwbG95bWVudHMgYXJlIGFkZGVkLCBfYXR0YWNoRGVwbG95bWVudCBpcyBjYWxsZWQgYW5kXG4gICAgICAgIC8vIHRoaXMgbWV0aG9kIHdpbGwgYmUgYWRkZWQgdGhlcmUuXG4gICAgICAgIGZvciAoY29uc3QgZGVwIG9mIHRoaXMuZGVwbG95bWVudHMpIHtcbiAgICAgICAgICAgIGRlcC5fYWRkTWV0aG9kRGVwZW5kZW5jeShtZXRob2QpO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEF0dGFjaGVzIGEgZGVwbG95bWVudCB0byB0aGlzIFJFU1QgQVBJLlxuICAgICAqXG4gICAgICogQGludGVybmFsXG4gICAgICovXG4gICAgcHVibGljIF9hdHRhY2hEZXBsb3ltZW50KGRlcGxveW1lbnQ6IERlcGxveW1lbnQpIHtcbiAgICAgICAgdGhpcy5kZXBsb3ltZW50cy5wdXNoKGRlcGxveW1lbnQpO1xuICAgICAgICAvLyBhZGQgYWxsIG1ldGhvZHMgdGhhdCB3ZXJlIGFscmVhZHkgZGVmaW5lZCBhcyBkZXBlbmRlbmNpZXMgb2YgdGhpc1xuICAgICAgICAvLyBkZXBsb3ltZW50IHdoZW4gYWRkaXRpb25hbCBtZXRob2RzIGFyZSBhZGRlZCwgX2F0dGFjaE1ldGhvZCBpcyBjYWxsZWQgYW5kXG4gICAgICAgIC8vIGl0IHdpbGwgYmUgYWRkZWQgYXMgYSBkZXBlbmRlbmN5IHRvIHRoaXMgZGVwbG95bWVudC5cbiAgICAgICAgZm9yIChjb25zdCBtZXRob2Qgb2YgdGhpcy5tZXRob2RzKSB7XG4gICAgICAgICAgICBkZXBsb3ltZW50Ll9hZGRNZXRob2REZXBlbmRlbmN5KG1ldGhvZCk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHByb3RlY3RlZCB2YWxpZGF0ZSgpIHtcbiAgICAgICAgaWYgKHRoaXMubWV0aG9kcy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgIHJldHVybiBbXCJUaGUgUkVTVCBBUEkgZG9lc24ndCBjb250YWluIGFueSBtZXRob2RzXCJdO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBbXTtcbiAgICB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgRW5kcG9pbnRDb25maWd1cmF0aW9uIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgdHlwZXM6IEVuZHBvaW50VHlwZVtdO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSB2cGNFbmRwb2ludHM/OiBJVnBjRW5kcG9pbnRbXTtcbn1cbmV4cG9ydCBlbnVtIEFwaUtleVNvdXJjZVR5cGUge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgSEVBREVSID0gJ0hFQURFUicsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIEFVVEhPUklaRVIgPSAnQVVUSE9SSVpFUidcbn1cbmV4cG9ydCBlbnVtIEVuZHBvaW50VHlwZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgRURHRSA9ICdFREdFJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIFJFR0lPTkFMID0gJ1JFR0lPTkFMJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgUFJJVkFURSA9ICdQUklWQVRFJ1xufVxuY2xhc3MgUm9vdFJlc291cmNlIGV4dGVuZHMgUmVzb3VyY2VCYXNlIHtcbiAgICBwdWJsaWMgcmVhZG9ubHkgcGFyZW50UmVzb3VyY2U/OiBJUmVzb3VyY2U7XG4gICAgcHVibGljIHJlYWRvbmx5IGFwaTogUmVzdEFwaUJhc2U7XG4gICAgcHVibGljIHJlYWRvbmx5IHJlc291cmNlSWQ6IHN0cmluZztcbiAgICBwdWJsaWMgcmVhZG9ubHkgcGF0aDogc3RyaW5nO1xuICAgIHB1YmxpYyByZWFkb25seSBkZWZhdWx0SW50ZWdyYXRpb24/OiBJbnRlZ3JhdGlvbiB8IHVuZGVmaW5lZDtcbiAgICBwdWJsaWMgcmVhZG9ubHkgZGVmYXVsdE1ldGhvZE9wdGlvbnM/OiBNZXRob2RPcHRpb25zIHwgdW5kZWZpbmVkO1xuICAgIHB1YmxpYyByZWFkb25seSBkZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnM/OiBDb3JzT3B0aW9ucyB8IHVuZGVmaW5lZDtcbiAgICBwcml2YXRlIHJlYWRvbmx5IF9yZXN0QXBpPzogUmVzdEFwaTtcbiAgICBjb25zdHJ1Y3RvcihhcGk6IFJlc3RBcGlCYXNlLCBwcm9wczogUmVzb3VyY2VPcHRpb25zLCByZXNvdXJjZUlkOiBzdHJpbmcpIHtcbiAgICAgICAgc3VwZXIoYXBpLCAnRGVmYXVsdCcpO1xuICAgICAgICB0aGlzLnBhcmVudFJlc291cmNlID0gdW5kZWZpbmVkO1xuICAgICAgICB0aGlzLmRlZmF1bHRJbnRlZ3JhdGlvbiA9IHByb3BzLmRlZmF1bHRJbnRlZ3JhdGlvbjtcbiAgICAgICAgdGhpcy5kZWZhdWx0TWV0aG9kT3B0aW9ucyA9IHByb3BzLmRlZmF1bHRNZXRob2RPcHRpb25zO1xuICAgICAgICB0aGlzLmRlZmF1bHRDb3JzUHJlZmxpZ2h0T3B0aW9ucyA9IHByb3BzLmRlZmF1bHRDb3JzUHJlZmxpZ2h0T3B0aW9ucztcbiAgICAgICAgdGhpcy5hcGkgPSBhcGk7XG4gICAgICAgIHRoaXMucmVzb3VyY2VJZCA9IHJlc291cmNlSWQ7XG4gICAgICAgIHRoaXMucGF0aCA9ICcvJztcbiAgICAgICAgaWYgKGFwaSBpbnN0YW5jZW9mIFJlc3RBcGkpIHtcbiAgICAgICAgICAgIHRoaXMuX3Jlc3RBcGkgPSBhcGk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHRoaXMuZGVmYXVsdENvcnNQcmVmbGlnaHRPcHRpb25zKSB7XG4gICAgICAgICAgICB0aGlzLmFkZENvcnNQcmVmbGlnaHQodGhpcy5kZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnMpO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEdldCB0aGUgUmVzdEFwaSBhc3NvY2lhdGVkIHdpdGggdGhpcyBSZXNvdXJjZS5cbiAgICAgKiBAZGVwcmVjYXRlZCAtIFRocm93cyBhbiBlcnJvciBpZiB0aGlzIFJlc291cmNlIGlzIG5vdCBhc3NvY2lhdGVkIHdpdGggYW4gaW5zdGFuY2Ugb2YgYFJlc3RBcGlgLiBVc2UgYGFwaWAgaW5zdGVhZC5cbiAgICAgKi9cbiAgICBwdWJsaWMgZ2V0IHJlc3RBcGkoKTogUmVzdEFwaSB7XG4gICAgICAgIGlmICghdGhpcy5fcmVzdEFwaSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdSZXN0QXBpIGlzIG5vdCBhdmFpbGFibGUgb24gUmVzb3VyY2Ugbm90IGNvbm5lY3RlZCB0byBhbiBpbnN0YW5jZSBvZiBSZXN0QXBpLiBVc2UgYGFwaWAgaW5zdGVhZCcpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB0aGlzLl9yZXN0QXBpO1xuICAgIH1cbn1cbmZ1bmN0aW9uIGlnbm9yZShfeDogYW55KSB7XG4gICAgcmV0dXJuO1xufVxuIl19