"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaIntegration = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const aws_1 = require("./aws");
/**
 * (experimental) Integrates an AWS Lambda function to an API Gateway method.
 *
 * @experimental
 * @example
 *
 *    const handler = new lambda.Function(this, 'MyFunction', ...);
 *    api.addMethod('GET', new LambdaIntegration(handler));
 */
class LambdaIntegration extends aws_1.AwsIntegration {
    /**
     * @experimental
     */
    constructor(handler, options = {}) {
        const proxy = options.proxy === undefined ? true : options.proxy;
        super({
            proxy,
            service: 'lambda',
            path: `2015-03-31/functions/${handler.functionArn}/invocations`,
            options,
        });
        this.handler = handler;
        this.enableTest = options.allowTestInvoke === undefined ? true : options.allowTestInvoke;
    }
    /**
     * (experimental) Can be overridden by subclasses to allow the integration to interact with the method being integrated, access the REST API object, method ARNs, etc.
     *
     * @experimental
     */
    bind(method) {
        const bindResult = super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        const desc = `${core_1.Names.nodeUniqueId(method.api.node)}.${method.httpMethod}.${method.resource.path.replace(/\//g, '.')}`;
        this.handler.addPermission(`ApiPermission.${desc}`, {
            principal,
            scope: method,
            sourceArn: core_1.Lazy.stringValue({ produce: () => method.methodArn }),
        });
        // add permission to invoke from the console
        if (this.enableTest) {
            this.handler.addPermission(`ApiPermission.Test.${desc}`, {
                principal,
                scope: method,
                sourceArn: method.testMethodArn,
            });
        }
        let functionName;
        if (this.handler instanceof lambda.Function) {
            // if not imported, extract the name from the CFN layer to reach
            // the literal value if it is given (rather than a token)
            functionName = this.handler.node.defaultChild.functionName;
        }
        else {
            // imported, just take the function name.
            functionName = this.handler.functionName;
        }
        let deploymentToken;
        if (!core_1.Token.isUnresolved(functionName)) {
            deploymentToken = JSON.stringify({ functionName });
        }
        return {
            ...bindResult,
            deploymentToken,
        };
    }
}
exports.LambdaIntegration = LambdaIntegration;
//# sourceMappingURL=data:application/json;base64,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