"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectionType = exports.PassthroughBehavior = exports.IntegrationType = exports.ContentHandling = exports.Integration = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const vpc_link_1 = require("./vpc-link");
/**
 * (experimental) Base class for backend integrations for an API Gateway method.
 *
 * Use one of the concrete classes such as `MockIntegration`, `AwsIntegration`, `LambdaIntegration`
 * or implement on your own by specifying the set of props.
 *
 * @experimental
 */
class Integration {
    /**
     * @experimental
     */
    constructor(props) {
        this.props = props;
        const options = this.props.options || {};
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
    }
    /**
     * (experimental) Can be overridden by subclasses to allow the integration to interact with the method being integrated, access the REST API object, method ARNs, etc.
     *
     * @experimental
     */
    bind(_method) {
        let uri = this.props.uri;
        const options = this.props.options;
        if ((options === null || options === void 0 ? void 0 : options.connectionType) === ConnectionType.VPC_LINK && uri === undefined) {
            uri = core_1.Lazy.stringValue({
                // needs to be a lazy since the targets can be added to the VpcLink construct after initialization.
                produce: () => {
                    const vpcLink = options.vpcLink;
                    if (vpcLink instanceof vpc_link_1.VpcLink) {
                        const targets = vpcLink._targetDnsNames;
                        if (targets.length > 1) {
                            throw new Error("'uri' is required when there are more than one NLBs in the VPC Link");
                        }
                        else {
                            return `http://${targets[0]}`;
                        }
                    }
                    else {
                        throw new Error("'uri' is required when the 'connectionType' is VPC_LINK");
                    }
                },
            });
        }
        return {
            options: {
                ...options,
                connectionType: (options === null || options === void 0 ? void 0 : options.vpcLink) ? ConnectionType.VPC_LINK : options === null || options === void 0 ? void 0 : options.connectionType,
            },
            type: this.props.type,
            uri,
            integrationHttpMethod: this.props.integrationHttpMethod,
        };
    }
}
exports.Integration = Integration;
/**
 * @experimental
 */
var ContentHandling;
(function (ContentHandling) {
    ContentHandling["CONVERT_TO_BINARY"] = "CONVERT_TO_BINARY";
    ContentHandling["CONVERT_TO_TEXT"] = "CONVERT_TO_TEXT";
})(ContentHandling = exports.ContentHandling || (exports.ContentHandling = {}));
/**
 * @experimental
 */
var IntegrationType;
(function (IntegrationType) {
    IntegrationType["AWS"] = "AWS";
    IntegrationType["AWS_PROXY"] = "AWS_PROXY";
    IntegrationType["HTTP"] = "HTTP";
    IntegrationType["HTTP_PROXY"] = "HTTP_PROXY";
    IntegrationType["MOCK"] = "MOCK";
})(IntegrationType = exports.IntegrationType || (exports.IntegrationType = {}));
/**
 * @experimental
 */
var PassthroughBehavior;
(function (PassthroughBehavior) {
    PassthroughBehavior["WHEN_NO_MATCH"] = "WHEN_NO_MATCH";
    PassthroughBehavior["NEVER"] = "NEVER";
    PassthroughBehavior["WHEN_NO_TEMPLATES"] = "WHEN_NO_TEMPLATES";
})(PassthroughBehavior = exports.PassthroughBehavior || (exports.PassthroughBehavior = {}));
/**
 * @experimental
 */
var ConnectionType;
(function (ConnectionType) {
    ConnectionType["INTERNET"] = "INTERNET";
    ConnectionType["VPC_LINK"] = "VPC_LINK";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
//# sourceMappingURL=data:application/json;base64,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